/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.reflection.utils;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;


/**
 *
 *
 * @author Christian Bick
 * @since 02.12.2010
 */
public class PropertyUtils {
	
	/**
	 * Sets the value of an object's property.<br/>
     * <br/>
     * If the field is public, the new value will be assigned to field directly.<br/>
     * Otherwise, a setter method will be looked up and invoked with the value to set.
	 * 
	 * @param owner The object that owns the property
	 * @param field The property field
	 * @param value The value to set
	 * @return The modified object
     * @throws IllegalAccessException Failed to access field or setter method
     * @throws NoSuchMethodException Setter method not found
     * @throws InvocationTargetException Failed to invoke setter method
	 */
	public static Object setProperty(Object owner, Field field, Object value) throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {
        if (Modifier.isPublic(field.getModifiers())) {
            field.set(owner, value);
        } else {
            try {
                // For none-public field try to access property via setter
                Method setter = owner.getClass().getMethod(ReflUtils.setterName(field.getName()), field.getType());
                setter.invoke(owner, value);
            } catch (NoSuchMethodException e) {
                // If there is no setter then make the field temporary accessible and set the field directly
                field.setAccessible(true);
                field.set(owner, value);
                field.setAccessible(false);
            }
        }
        return owner;
	}
	
	/**
	 * Gets the value of an object's property.
	 * 
	 * @param owner The object
	 * @param field The property field
	 * @return The value of the property
     * @throws IllegalAccessException Failed to access field or getter method
     * @throws NoSuchMethodException Getter method not found
     * @throws InvocationTargetException Failed to invoke getter method
	 */
	public static Object getProperty(Object owner, Field field) throws IllegalAccessException, NoSuchMethodException, InvocationTargetException {
        if (Modifier.isPublic(field.getModifiers())) {
            return field.get(owner);
        } else {
            Method getter = owner.getClass().getMethod(ReflUtils.getterName(field.getName()));
            return getter.invoke(owner);
        }
	}
}
