/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.reflection.utils;

import java.io.File;
import java.io.IOException;
import java.net.JarURLConnection;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * Provides utilities for package actions.
 *
 * @author Christian Bick
 * @since 14.03.2011
 */
public class PackageUtils {

    /**
     * <p>Finds all classes accessible from the class loader and belonging to the given package and its sub-packages.
     * Uses classes from the file system if the given package can be obtained as a system resource by the class
     * loader.</p>
     * <p>Only if no such resource is available, the package will be loaded from a jar file resource if available.</p>
     * <p>Note: This behaviour is a consequence of the class loader behaviour and may differ in Java versions other
     * then 1.6.</p>
     *
     * @param packageName The package
     * @return The classes
     */
        public static List<Class<?>> findClasses(String packageName) {
        ArrayList<Class<?>> classes = new ArrayList<Class<?>>();
        try{
            ClassLoader classLoader = PackageUtils.class.getClassLoader();
            assert classLoader != null;
            String packagePath = nameToPath(packageName);
            Enumeration<URL> resources = classLoader.getResources(packagePath);

            while (resources.hasMoreElements()) {
                URL resourceUrl = resources.nextElement();
                if (resourceUrl.getProtocol().equals("jar")) {
                    JarURLConnection con = (JarURLConnection) resourceUrl.openConnection();
                    JarFile jarFile = con.getJarFile();
                    classes.addAll(findClassesInJar(jarFile, packagePath));
                } else {
                    File directory = new File(resourceUrl.toURI());
                    classes.addAll(findClassesInDir(directory, packageName));
                }
            }
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("Failed to initialize a class during package scan", e);
        } catch (IOException e) {
            throw new RuntimeException("Failed to read a class file during package scan", e);
        } catch (URISyntaxException e) {
            throw new RuntimeException("Failed to convert resource URL to file path", e);
        }
            return classes;
    }

    /**
     * Recursive method used to find all classes in a given directory and sub directories.
     *
     * @param directory   The base directory
     * @param packageName The package name for classes found inside the base directory
     * @return The classes found
     * @throws ClassNotFoundException if class loading fails
     */
    public static List<Class<?>> findClassesInDir(File directory, String packageName) throws ClassNotFoundException {
        List<Class<?>> classes = new ArrayList<Class<?>>();
        if (!directory.exists()) {
            return classes;
        }
        File[] files = directory.listFiles();
        for (File file : files) {
            if (file.isDirectory()) {
                assert !file.getName().contains(".");
                classes.addAll(findClassesInDir(file, packageName + "." + file.getName()));
            } else if (file.getName().endsWith(".class")) {
                classes.add(Class.forName(packageName + '.' + file.getName().substring(0, file.getName().length() - 6)));
            }
        }
        return classes;
    }

    /**
     * Finds all classes in a the package path of the given jar file.
     *
     * @param jarFile The jar file.
     * @param packagePath The package path
     * @return The classes found
     * @throws ClassNotFoundException if class loading fails
     * @throws IOException if the jar file cannot be read
     */
    public static List<Class<?>> findClassesInJar(JarFile jarFile, String packagePath) throws ClassNotFoundException, IOException {
        List<Class<?>> classes = new ArrayList<Class<?>>();
        Enumeration<JarEntry> jarEntries = jarFile.entries();
        while (jarEntries.hasMoreElements()) {
            JarEntry jarEntry = jarEntries.nextElement();
            if (jarEntry.getName().startsWith(packagePath)
                    && jarEntry.getName().endsWith(".class")) {
                String className = pathToName(jarEntry.getName().substring(0, jarEntry.getName().length() - 6));
                classes.add(Class.forName(className));
            }
        }
        return classes;
    }

    /**
     * <p>Converts a package name to a package path.</p>
     * <p><b>example:</b> net.craftforge.essential -> net/craftforge/example</p>
     *
     * @param packageName The package name
     * @return The package path
     */
    public static String nameToPath(String packageName) {
        return packageName.replace('.', '/');
    }

    /**
     * <p>Converts a package path to a package name.</p>
     * <p><b>example:</b> net/craftforge/essential -> net.craftforge.example</p>
     *
     * @param packagePath The package path
     * @return The package name
     */
    public static String pathToName(String packagePath) {
        return packagePath.replace('/', '.');
    }
}
