/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.reflection.utils;

import sun.reflect.generics.reflectiveObjects.TypeVariableImpl;

import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;

/**
 * Provides utilities for class-related reflection operations.
 *
 * @author Christian Bick
 * @since 02.12.2010
 */
public class ClassUtils {

    /**
     * Gets a class hierarchy of a class as a list containing the class itself and all
     * its super classes except of java.lang.Object. (bottom up)
     *
     * @param c The class
     * @return List containing the class and all its super classes except of java.lang.Object
     */
    public static List<Class<?>> getClassHierarchy(Class<?> c) {
        List<Class<?>> classHierarchy = new ArrayList<Class<?>>();
        Class<?> next = c;
        while (next != null && ! next.equals(Object.class)) {
            classHierarchy.add(next);
            next = next.getSuperclass();
        }
		return classHierarchy;
    }

    /**
     * Gets a interface hierarchy of a class as a list containing the class's
     * implemented interfaces and all their super interfaces. (bottom up)
     *
     * @param c The class
     * @return List of classes that represent the interfaces
     */
    public static List<Class<?>> getInterfaceHierarchy(Class<?> c) {
        List<Class<?>> interfaceHierarchy = new ArrayList<Class<?>>();
        for (Class<?> directInterface : c.getInterfaces()) {
            interfaceHierarchy.add(directInterface);
            if (directInterface.getInterfaces().length > 0) {
                interfaceHierarchy.addAll(getInterfaceHierarchy(directInterface));
            }
        }
		return interfaceHierarchy;
    }

    /**
     * Looks for the first implementation or overriding of a method within
     * the class hierarchy of the given class.
     *
     * @param c The class to look within the class hierarchy
     * @param method The implemented or overridden method
     * @return The first found implementing or overriding method
     */
    public static Method getFirstImplementation(Class<?> c, Method method) {
        for (Class<?> candidateClass : getClassHierarchy(c)) {
            for (Method candidateMethod : candidateClass.getDeclaredMethods()) {
                if (isMethodExchangeableBy(method, candidateMethod)) {
                    return candidateMethod;
                }
            }
        }
        return null;
    }

    /**
     * Compares the name and parameters of both methods with each other and looks for the method
     * parameters to be compatible with the candidate method parameters. Treats generic types
     * as wild cards for any kind of type.
     *
     * @param method The method
     * @param candidateMethod The candidate method
     * @return true, if the method parameters are compatible, false otherwise
     */
    public static boolean isMethodExchangeableBy(Method method, Method candidateMethod) {
        if (! method.getName().equals(candidateMethod.getName())) {
            return false;
        }

        Class<?>[] params = method.getParameterTypes();
        Class<?>[] candidateParams = candidateMethod.getParameterTypes();
        Type[] genericParams = method.getGenericParameterTypes();
        Type[] genericCandidateParams = candidateMethod.getGenericParameterTypes();

        if (params.length != candidateParams.length) {
            return false;
        }

        for (int i = 0; i < params.length; i++) {
            Class<?> param = params[i];
            Class<?> candidateParam = candidateParams[i];
            Type genericParam = genericParams[i];
            Type genericCandidateParam = genericCandidateParams[i];

            if (genericCandidateParam instanceof TypeVariableImpl && param.equals(Object.class)) {
                return false; // @TODO: Improve criteria, not perfect because it does not honor the case for the actual generic being indeed of class Object
            }

            if (genericParam instanceof TypeVariableImpl && candidateParam.equals(Object.class)) {
                return false; // @TODO: Improve criteria, not perfect because it does not honor the case for the actual generic being indeed of class Object
            }

            if (! candidateParam.equals(param)
                    && ! (genericCandidateParam instanceof TypeVariableImpl)
                    && ! (genericParam instanceof TypeVariableImpl)) {
                return false;
            }
        }
        return true;
    }
}
