/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.reflection.managers;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * A reference manager takes care an object's reference structure.
 *
 * @author Christian Bick
 * @since 13.08.2011
 */
public class ReferenceManager<T> {

    private static final Logger LOGGER = LoggerFactory.getLogger(ReferenceManager.class);

    private static ConcurrentMap<String, ReferenceManager> instances = new ConcurrentHashMap<String, ReferenceManager>();

    /**
     * Gets an instance of a reference manager for a class of objects
     * being managed by the manager.
     *
     * @param clazz The class
     * @param <T> The class type
     * @return The reference manager instance
     */
    @SuppressWarnings("unchecked")
    public static <T> ReferenceManager<T> getInstance(Class<T> clazz) {
        if (! instances.containsKey(clazz.getName())) {
            LOGGER.debug("[Reference manager initialization] {}" + clazz.getName());
            instances.putIfAbsent(clazz.getName(), new ReferenceManager<T>(clazz));
        }
        return instances.get(clazz.getName());
    }

    private ClassManager classManager;

    /**
     * Initializes a reference manager for a class.
     *
     * @param clazz The class
     */
    private ReferenceManager(Class<T> clazz) {
        classManager = ClassManager.getInstance(clazz);
    }

    /**
     * Nullify all references of the given object ignoring references
     * to virtual-primitives as strings and dates.
     *
     * @param object The object
     * @return The object without any references
     */
    public T nullifyActualReferences(T object) {
        for (Field field : classManager.getActualReferences()) {
            field.setAccessible(true);
            try {
                field.set(object, null);
            } catch (IllegalAccessException e) {
                throw new RuntimeException(e);
            }
        }
        return object;
    }

}
