/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.reflection.managers;

import net.craftforge.reflection.utils.ClassUtils;
import net.craftforge.reflection.utils.ReflUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * Manages a class by caching all reflection actions provided by the manager.
 *
 * @author Christian Bick
 * @since 14.03.2011
 */
public class ClassManager {

    private static final Logger LOGGER = LoggerFactory.getLogger(ClassManager.class);

    /**
     * The class manager instances
     */
    private static ConcurrentMap<String, ClassManager> instances = new ConcurrentHashMap<String, ClassManager>();

    /**
     * Gets the class manager  corresponding to the field's declaring class
     *
     * @param field The field
     * @return The corresponding ClassManager
     */
    public static ClassManager getInstance(Field field) {
        return getInstance(field.getDeclaringClass());
    }

    /**
     * Gets the class manager  corresponding to the method's declaring class
     *
     * @param method The method
     * @return The corresponding class manager
     */
    public static ClassManager getInstance(Method method) {
        return getInstance(method.getDeclaringClass());
    }

    /**
     * Gets the class manager corresponding to the class.
     *
     * @param clazz The class
     * @return The corresponding class manager
     */
    public static ClassManager getInstance(Class<?> clazz) {
        if (! instances.containsKey(clazz.getName())) {
            LOGGER.debug("[Class manager initialization] {}", clazz.getName());
            instances.putIfAbsent(clazz.getName(), new ClassManager(clazz));
        }
        return instances.get(clazz.getName());
    }

    /**
     * Synchronization monitor to avoid concurrent modification
     * of properties
     */
    private final Object monitor = new Object();

    /**
     * The managed class
     */
    private Class<?> clazz;

    /**
     * The cached class hierarchy
     */
    private List<Class<?>> classHierarchy;

    /**
     * The cached interface hierarchy
     */
    private List<Class<?>> interfaceHierarchy;

    /**
     * The cached actual reference fields
     */
    private List<Field> actualReferences;

    /**
     * The cached actual primitive fields
     */
    private List<Field> virtualPrimitives;

    /**
     * The cached class hierarchy combined with the cached interface hierarchy of the class and all its super classes
     */
    private List<Class<?>> completeClassHierarchy;

    /**
     * The cached public methods of the complete class hierarchy
     */
    private List<Method> allMethods;

    /**
     * The cached property annotations found in the complete class hierarchy
     */
    private ConcurrentMap<Field, ConcurrentMap<String, Annotation>> propertyAnnotations = new ConcurrentHashMap<Field, ConcurrentMap<String, Annotation>>();

    /**
     * The cached method annotations found in the complete class hierarchy
     */
    private ConcurrentMap<Method, ConcurrentMap<String, Annotation>> methodAnnotations = new ConcurrentHashMap<Method, ConcurrentMap<String, Annotation>>();

    /**
     * The cached parameter annotations found in the complete class hierarchy
     */
    private ConcurrentMap<Method, ConcurrentMap<String, Annotation[][]>> methodParameterAnnotations = new ConcurrentHashMap<Method, ConcurrentMap<String, Annotation[][]>>();

    /**
     * The cached type annotations of the complete class hierarchy
     */
    private ConcurrentMap<String, Annotation> allTypeAnnotations = new ConcurrentHashMap<String, Annotation>();

    /**
     * The cached annotated methods of the complete class hierarchy grouped by annotation class path
     */
    private ConcurrentMap<String, List<Method>> allAnnotatedMethods = new ConcurrentHashMap<String, List<Method>>();

    /**
     * The cached annotated fields of the complete class hierarchy grouped by annotation class path
     */
    private ConcurrentMap<String, List<Field>> allAnnotatedProperties = new ConcurrentHashMap<String, List<Field>>();

    /**
     * The cached getter methods of the complete class hierarchy
     */
    private ConcurrentMap<String, List<Method>> allGetterMethods;

    /**
     * The cached setter methods of the complete class hierarchy
     */
    private ConcurrentMap<String, List<Method>> allSetterMethods;

    /**
     * Constructs a class manager managing a class
     *
     * @param clazz The managed class
     */
    private ClassManager(Class<?> clazz) {
        this.clazz = clazz;
    }

    /**
     * Gets the first occurrence of an annotation of the given annotation class at type level.
     * Searches within the complete class and interface hierarchy.
     *
     * @param annotationClass The annotation class
     * @return First occurrence of the searched annotation, null if not found
     */
    public Annotation getTypeLevelAnnotation(Class<? extends Annotation> annotationClass) {
        if (! allTypeAnnotations.containsKey(annotationClass.getName())) {
            synchronized (monitor) {
                for (Class<?> classInHierarchy : getCompleteClassHierarchy()) {
                    if (classInHierarchy.isAnnotationPresent(annotationClass)) {
                        allTypeAnnotations.putIfAbsent(annotationClass.getName(), classInHierarchy.getAnnotation(annotationClass));
                        break;
                    }
                }
            }
        }
        return allTypeAnnotations.get(annotationClass.getName());
    }

    /**
     * Gets the first occurrence of an annotation of the given annotation class at field level or
     * at method level of the corresponding setter and getter methods.
     * Searches within the complete class and interface hierarchy of the field's declaring class.
     *
     * @param field The field
     * @param annotationClass The annotation class
     * @return First occurrence of the searched annotation, null if not found
     */
    public Annotation getPropertyLevelAnnotation(Field field, Class<? extends Annotation> annotationClass) {
        ConcurrentMap<String, Annotation> annotationMap = propertyAnnotations.putIfAbsent(field, new ConcurrentHashMap<String, Annotation>());
        annotationMap = annotationMap != null ? annotationMap : propertyAnnotations.get(field);

        if (! annotationMap.containsKey(annotationClass.getName())) {
            synchronized (monitor) {
                for (Field candidateField : getAllPropertiesAnnotatedWith(annotationClass)) {
                    if (field.getName().equals(candidateField.getName())
                            && candidateField.isAnnotationPresent(annotationClass)) {
                        annotationMap.put(annotationClass.getName(), candidateField.getAnnotation(annotationClass));
                        break;
                    }
                }
                if (! annotationMap.containsKey(annotationClass.getName())) {
                    List<Method> getters = getAllGetterMethods().get(field.getName());
                    if (getters != null) {
                        for (Method candidateMethod : getters) {
                            if (candidateMethod.isAnnotationPresent(annotationClass)) {
                                annotationMap.put(annotationClass.getName(), candidateMethod.getAnnotation(annotationClass));
                                break;
                            }
                        }
                    }
                    if (! annotationMap.containsKey(annotationClass.getName())) {
                        List<Method> setters = getAllSetterMethods().get(field.getName());
                        if (setters != null) {
                            for (Method candidateMethod : setters) {
                                if (candidateMethod.isAnnotationPresent(annotationClass)) {
                                    annotationMap.put(annotationClass.getName(), candidateMethod.getAnnotation(annotationClass));
                                    break;
                                }
                            }
                        }
                    }
                }
            }
        }
        return propertyAnnotations.get(field).get(annotationClass.getName());
    }

    /**
     * Gets the first occurrence of an annotation of the given annotation class at method level.
     * Searches within the complete class and interface hierarchy of the method's declaring class.
     *
     * @param method The method
     * @param annotationClass The annotation class
     * @return First occurrence of the searched annotation, null if not found
     */
    public Annotation getMethodLevelAnnotation(Method method, Class<? extends Annotation> annotationClass) {
        ConcurrentMap<String, Annotation> annotationMap = methodAnnotations.putIfAbsent(method, new ConcurrentHashMap<String, Annotation>());
        annotationMap = annotationMap != null ? annotationMap : methodAnnotations.get(method);

        if (! annotationMap.containsKey(annotationClass.getName())) {
            synchronized (monitor) {
                for (Method candidateMethod : getAllMethodsAnnotatedWith(annotationClass)) {
                    if (ClassUtils.isMethodExchangeableBy(method, candidateMethod)) {
                        annotationMap.put(annotationClass.getName(), candidateMethod.getAnnotation(annotationClass));
                        break;
                    }
                }
            }
        }
        return methodAnnotations.get(method).get(annotationClass.getName());
    }

    /**
     * Gets all parameter annotations of a method. Searches for a signature
     * using the annotation of type annotation class within the complete class and
     * interface hierarchy of the method's declaring class.
     *
     * @param method The method
     * @param annotationClass The annotation class
     * @return Two dimensional array - first dimension for the parameters, the second one for each parameter's annotation.
     */
    public Annotation[][] getMethodParameterAnnotations(Method method, Class<? extends Annotation> annotationClass) {
        ConcurrentMap<String, Annotation[][]> annotationMap = methodParameterAnnotations.putIfAbsent(method, new ConcurrentHashMap<String, Annotation[][]>());
        annotationMap = annotationMap != null ? annotationMap : methodParameterAnnotations.get(method);

        if (! annotationMap.containsKey(annotationClass.getName())) {
            synchronized (monitor) {
                annotationMap.put(annotationClass.getName(), new Annotation[][] {});
                for (Class<?> clazz : getCompleteClassHierarchy()) {
                    for (Method candidateMethod : clazz.getDeclaredMethods()) {
                        if (! ClassUtils.isMethodExchangeableBy(method, candidateMethod)) {
                            continue;
                        }
                        Annotation[][] annotations = candidateMethod.getParameterAnnotations();
                        outer : for (Annotation[] paramAnnotations : annotations) {
                            for (Annotation paramAnnotation : paramAnnotations) {
                                if (annotationClass.equals(paramAnnotation.annotationType())) {
                                    annotationMap.put(annotationClass.getName(), annotations);
                                    break outer;
                                }
                            }
                        }
                    }
                }
            }
        }
        return methodParameterAnnotations.get(method).get(annotationClass.getName());
    }

    /**
     * Gets the first annotation of an annotation class on the given method's or type level.
     * Makes use of the complete class hierarchy. Prefers method annotations about type
     * annotations.
     *
     * @param method The method
     * @param annotationClass The annotation class
     * @return The first found annotation
     */
    public Annotation getMethodOrTypeLevelAnnotation(Method method, Class<? extends Annotation> annotationClass) {
        Annotation methodLevelAnnotation = getMethodLevelAnnotation(method, annotationClass);
        return methodLevelAnnotation != null ? methodLevelAnnotation : getTypeLevelAnnotation(annotationClass);
    }

    /**
     * Gets all methods being annotated with an annotation of the given annotation class.
     * Makes use of the complete class hierarchy.
     *
     * @param annotationClass The annotation class
     * @return List of all annotated methods
     */
    public List<Method> getAllMethodsAnnotatedWith(Class<? extends Annotation> annotationClass) {
        if (! allAnnotatedMethods.containsKey(annotationClass.getName())) {
            synchronized (monitor) {
                List<Method> methods = new ArrayList<Method>();
                for (Class<?> classInHierarchy : getCompleteClassHierarchy()) {
                    for (Method methodInHierarchy : classInHierarchy.getDeclaredMethods()) {
                        if (Modifier.isPublic(methodInHierarchy.getModifiers())
                                && methodInHierarchy.isAnnotationPresent(annotationClass)) {
                            methods.add(methodInHierarchy);
                        }
                    }
                }
                allAnnotatedMethods.put(annotationClass.getName(), methods);
            }
        }
        return Collections.unmodifiableList(allAnnotatedMethods.get(annotationClass.getName()));
    }

    /**
     * Gets all fields which are annotated themselves or whose corresponding getters or setters are annotated with
     * an annotation of the given annotation class. Makes use of the complete class hierarchy.
     *
     * @param annotationClass The annotation class
     * @return List of all annotated properties
     */
    public List<Field> getAllPropertiesAnnotatedWith(Class<? extends Annotation> annotationClass) {
        if (! allAnnotatedProperties.containsKey(annotationClass.getName())) {
            synchronized (monitor) {
                List<Field> annotatedFields = new ArrayList<Field>();
                for (Class<?> classInHierarchy : getCompleteClassHierarchy()) {
                    for (Field fieldInHierarchy : classInHierarchy.getDeclaredFields()) {
                        if (fieldInHierarchy.isAnnotationPresent(annotationClass)) {
                            annotatedFields.add(fieldInHierarchy);
                            continue;
                        }
                        List<Method> getters = getAllGetterMethods().get(fieldInHierarchy.getName());
                        if (getters != null) {
                            for (Method getter : getters) {
                                if (getter.isAnnotationPresent(annotationClass)) {
                                    annotatedFields.add(fieldInHierarchy);
                                }
                            }
                        }
                        List<Method> setters = getAllSetterMethods().get(fieldInHierarchy.getName());
                        if (setters != null) {
                            for (Method setter : setters) {
                                if (setter.isAnnotationPresent(annotationClass)) {
                                    annotatedFields.add(fieldInHierarchy);
                                }
                            }
                        }
                    }
                }
                allAnnotatedProperties.put(annotationClass.getName(), annotatedFields);
            }
        }
        return Collections.unmodifiableList(allAnnotatedProperties.get(annotationClass.getName()));
    }

    /**
     * Gets a class hierarchy of the class as a list containing the class itself and all
     * its super classes except of java.lang.Object. (bottom up)
     *
     * @return List containing the class and all its super classes except of java.lang.Object
     */
    public List<Class<?>> getClassHierarchy() {
        if (classHierarchy == null) {
            classHierarchy = Collections.unmodifiableList(ClassUtils.getClassHierarchy(clazz));
        }
        return classHierarchy;
    }

    /**
     * Gets a interface hierarchy of the class as a list containing the class's
     * implemented interfaces and all their super interfaces. (bottom up)
     *
     * @return List of classes that represent the interfaces
     */
    public List<Class<?>> getInterfaceHierarchy() {
        if (interfaceHierarchy == null) {
            interfaceHierarchy = Collections.unmodifiableList(ClassUtils.getInterfaceHierarchy(clazz));
        }
        return interfaceHierarchy;
    }

    /**
     * Gets a class hierarchy of the class as a list containing the class itself and
     * all its super classes as well as all implemented interfaces within the class hierarchy
     * and their super interfaces. (bottom up)
     *
     * @return List of classes - first all classes, second all interfaces
     */
    public List<Class<?>> getCompleteClassHierarchy() {
        if (completeClassHierarchy == null) {
            synchronized (monitor) {
                List<Class<?>> completeClassHierarchyAssemble = new ArrayList<Class<?>>();
                List<Class<?>> cHierarchy = getClassHierarchy();
                List<Class<?>> iHierarchy = new ArrayList<Class<?>>();
                for (Class<?> classInHierarchy : cHierarchy) {
                    for (Class<?> i : ClassManager.getInstance(classInHierarchy).getInterfaceHierarchy()) {
                        iHierarchy.add(i);
                    }
                }
                completeClassHierarchyAssemble.addAll(cHierarchy);
                completeClassHierarchyAssemble.addAll(iHierarchy);
                completeClassHierarchy = Collections.unmodifiableList(completeClassHierarchyAssemble);
            }
        }
        return completeClassHierarchy;
    }

    /**
     * Gets all declared methods. Searches within the complete class and interface hierarchy.
     *
     * @return List of all methods declared in the class hierarchy - first declared by classes, second
     * declared by interfaces
     */
    public List<Method> getCompleteClassHierarchyMethods() {
        if (allMethods == null) {
            synchronized (monitor) {
                List<Method> allMethodsAssemble = new ArrayList<Method>();
                for (Class<?> classInHierarchy : getCompleteClassHierarchy()) {
                    allMethodsAssemble.addAll(Arrays.asList(classInHierarchy.getDeclaredMethods()));
                }
                allMethods = Collections.unmodifiableList(allMethodsAssemble);
            }
        }
        return allMethods;
    }

    /**
     * Gets all getter methods of a class and groups them by their corresponding field names.
     *
     * @return Map with keys representing the field names and values representing the corresponding getter methods
     */
    public ConcurrentMap<String, List<Method>> getAllGetterMethods() {
        if (allGetterMethods == null)  {
            synchronized (monitor) {
                allGetterMethods = new ConcurrentHashMap<String, List<Method>>();
                for (Class<?> classInHierarchy : getCompleteClassHierarchy()) {
                    for (Method methodInHierarchy : classInHierarchy.getDeclaredMethods()) {
                        if (! Modifier.isPublic(methodInHierarchy.getModifiers())) {
                            continue;
                        }
                        String methodName = methodInHierarchy.getName();
                        String fieldName = null;
                        if (methodName.startsWith("get") && methodName.length() > 3) {
                            fieldName = methodName.replace("get", "");
                        } else if (methodName.startsWith("is") && methodName.length() > 2) {
                            fieldName = methodName.replace("is", "").toLowerCase();
                        }
                        if (fieldName != null) {
                            fieldName = fieldName.replace(fieldName.substring(0, 1), fieldName.substring(0, 1).toLowerCase());
                            List<Method> fieldGetters = allGetterMethods.get(fieldName);
                            if (fieldGetters == null) {
                                fieldGetters = new ArrayList<Method>();
                                allGetterMethods.put(fieldName, fieldGetters);
                            }
                            allGetterMethods.get(fieldName).add(methodInHierarchy);
                        }
                    }
                }
            }
        }
        return allGetterMethods;
    }

    /**
     * Gets all setter methods of a class and groups them by their corresponding field names.
     *
     * @return Map with keys representing the field names and values representing the corresponding setter methods
     */
    public ConcurrentMap<String, List<Method>> getAllSetterMethods() {
        if (allSetterMethods == null) {
            synchronized (monitor) {
                allSetterMethods = new ConcurrentHashMap<String, List<Method>>();
                for (Class<?> classInHierarchy : getCompleteClassHierarchy()) {
                    for (Method methodInHierarchy : classInHierarchy.getDeclaredMethods()) {
                        if (! Modifier.isPublic(methodInHierarchy.getModifiers())) {
                            continue;
                        }
                        String methodName = methodInHierarchy.getName();
                        if (methodName.startsWith("set") && methodName.length() > 3) {
                            String fieldName = methodName.replace("set", "");
                            fieldName = fieldName.replace(fieldName.substring(0, 1), fieldName.substring(0, 1).toLowerCase());
                            List<Method> fieldSetters = allSetterMethods.get(fieldName);
                            if (fieldSetters == null) {
                                fieldSetters = new ArrayList<Method>();
                                allSetterMethods.put(fieldName, fieldSetters);
                            }
                            allSetterMethods.get(fieldName).add(methodInHierarchy);
                        }
                    }
                }
            }
        }
        return allSetterMethods;
    }

    /**
     * Gets all actual reference fields of a class. Searches within the complete
     * class hierarchy. Actual references are  represented by all classes that
     * are not virtual primitives. So this method ignores references to primitive
     * objects (Integer, Long, etc.) as well as to String and Date.
     *
     * @return List containing all actual references in the class hierarchy
     */
    public List<Field> getActualReferences() {
        if (actualReferences == null) {
            synchronized (monitor) {
                List<Field> actualReferencesAssemble = new LinkedList<Field>();
                for (Class<?> clazz : getClassHierarchy()) {
                    for (Field field : clazz.getDeclaredFields()) {
                        if (ReflUtils.isActualReference(field.getType())) {
                            actualReferencesAssemble.add(field);
                        }
                    }
                }
                actualReferences = Collections.unmodifiableList(actualReferencesAssemble);
            }
        }
        return actualReferences;
    }

    /**
     * Gets all virtual primitives of a class. Searches within the complete
     * class hierarchy. Virtual primitives are all real primitives, their
     * object representation (Integer, Long etc.) as well as String and
     * Date.
     *
     * @return List containing all virtual primitives in the class hierarchy
     */
    public List<Field> getVirtualPrimitives() {
        if (virtualPrimitives == null) {
            synchronized (monitor) {
                List<Field> virtualPrimitivesAssemble = new LinkedList<Field>();
                for (Class<?> clazz : getClassHierarchy()) {
                    for (Field field : clazz.getDeclaredFields()) {
                        if (ReflUtils.isVirtualPrimitive(field.getType())) {
                            virtualPrimitivesAssemble.add(field);
                        }
                    }
                }
                virtualPrimitives = Collections.unmodifiableList(virtualPrimitivesAssemble);
            }
        }
        return virtualPrimitives;
    }


}
