/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.supply.jaxb;

import net.craftforge.essential.controller.Configuration;
import net.craftforge.essential.controller.ControllerException;
import net.craftforge.essential.controller.annotations.Produces;
import net.craftforge.essential.supply.Producer;
import org.codehaus.jackson.JsonGenerator;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.map.SerializationConfig;
import org.codehaus.jackson.xc.JaxbAnnotationIntrospector;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.util.Map;

/**
 * Serializes objects using JAXB annotations.
 * 
 * @author Christian Bick
 * @since 02.12.2010
 */
@SuppressWarnings("unused")
public class JaxbProducer extends Producer {

    /**
     * Converts a JAXB Marshaller property value into the desired boolean for
     * boolean Marshaller properties.
     *
     * @param propertyName The property name
     * @param propertyValue The property value
     * @return The converted property value
     */
    private static Object convertPropertyValue(String propertyName, String propertyValue) {
        if (propertyName.equals(Marshaller.JAXB_FORMATTED_OUTPUT) || propertyName.equals(Marshaller.JAXB_FRAGMENT)) {
            return Boolean.valueOf(propertyValue);
        }
        return propertyValue;
    }

    /**
     * Constructs the producer by supplying the controller configuration.
     *
     * @param config The configuration
     */
    public JaxbProducer(Configuration config) {
        super(config);
    }

    /**
     * Serializes the result object to XML, using the specified character encoding.
     * Uses the output stream for streaming.
     *
     * @param result The result
     * @param out The output stream
     * @param charset The character encoding
     * @throws JAXBException Failed to serialize object to XML
     * @throws UnsupportedEncodingException Unsupported charset encoding
     */
    @Produces({"text/xml", "application/xml"})
    public void xml(Object result, OutputStream out, String charset) throws JAXBException, IOException {
        JAXBContext context = JAXBContext.newInstance(result.getClass());
        Marshaller marshaller = context.createMarshaller();
        for (Map.Entry<String, String> entry : config.getXmlSerializationProperties().entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            marshaller.setProperty(key, convertPropertyValue(key, value));
        }
        OutputStreamWriter writer = new OutputStreamWriter(out, charset);
        marshaller.marshal(result, writer);
        writer.close();
    }

    /**
     * Serializes the result object to JSON, using the specified character encoding.
     * Uses the output stream for streaming.
     *
     * @param result The result
     * @param out The output stream
     * @param charset The character encoding
     * @throws IOException Failed to serialize object to JSON
     * @throws ControllerException Invalid JSON property
     */
    @Produces({"text/json", "application/json"})
    public void json(Object result, OutputStream out, String charset) throws ControllerException, IOException {
        ObjectMapper mapper = new ObjectMapper();
        mapper.getDeserializationConfig().withAnnotationIntrospector(new JaxbAnnotationIntrospector());
        for (Map.Entry<String, String> entry : config.getXmlSerializationProperties().entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            mapper.getSerializationConfig().set(SerializationConfig.Feature.valueOf(key), Boolean.valueOf(value));
        }
        OutputStreamWriter writer = new OutputStreamWriter(out, charset);
        JsonGenerator generator = mapper.getJsonFactory().createJsonGenerator(writer);
        generator.writeObject(result);
        generator.close();
    }
}
