/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.supply.jaxb;

import net.craftforge.essential.controller.Configuration;
import net.craftforge.essential.controller.annotations.Consumes;
import net.craftforge.essential.supply.Consumer;
import org.codehaus.jackson.map.DeserializationConfig;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.xc.JaxbAnnotationIntrospector;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;

/**
 * Serializes objects.
 *
 * @author Christian Bick
 * @since 26.02.2011
 */
@SuppressWarnings("unused")
public class JaxbConsumer extends Consumer {

    public JaxbConsumer(Configuration config) {
        super(config);
    }

    /**
     * Deserializes the input stream in XML format to an object of the specified type.
     *
     * @param type The type
     * @param input The input stream
     * @param charset The charset encoding
     * @return The object
     * @throws JAXBException Failed to convert the input to an object
     * @throws UnsupportedEncodingException Failed to process the charset
     */
    @Consumes({"text/xml", "application/xml"})
    public Object xml(Class<?> type, InputStream input, String charset) throws JAXBException, IOException {
        JAXBContext jc = JAXBContext.newInstance(type);
        Unmarshaller u = jc.createUnmarshaller();
        for (String key : config.getXmlDeserializationProperties().keySet()) {
            String value = config.getXmlDeserializationProperties().get(key);
            u.setProperty(key, value);
        }
        InputStreamReader reader = new InputStreamReader(input, charset);
        Object value = u.unmarshal(reader);
        reader.close();
        return value;
    }

    /**
     * Deserializes the input stream in JSON format to an object of the specified type.
     *
     * @param type The type
     * @param input The input stream
     * @param charset The charset encoding
     * @return The object
     * @throws IOException Failed to convert the input to an object
     */
    @Consumes({"text/json", "application/json"})
    public Object json(Class<?> type, InputStream input, String charset) throws IOException {
        ObjectMapper mapper = new ObjectMapper();
        mapper.getDeserializationConfig().withAnnotationIntrospector(new JaxbAnnotationIntrospector());
        for (String key : config.getJsonDeserializationProperties().keySet()) {
            String value = config.getJsonDeserializationProperties().get(key);
            mapper.getDeserializationConfig().set(DeserializationConfig.Feature.valueOf(key), Boolean.valueOf(value));
        }
        InputStreamReader reader = new InputStreamReader(input, charset);
        Object value = mapper.readValue(reader, type);
        reader.close();
        return value;
    }

}
