/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.supply;

import net.craftforge.essential.controller.Configuration;
import net.craftforge.essential.controller.ControllerException;
import net.craftforge.essential.controller.annotations.Produces;
import net.craftforge.essential.controller.utils.AnnotationUtils;

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * Provides an implementation of the Controllable interface in
 * a view setup.
 *
 * @author Christian Bick
 * @since 02.12.2010
 */
@SuppressWarnings("unused")
public class Producer {

    /**
     * The configuration
     */
    protected Configuration config;

    /**
     * Constructs the producer by supplying the controller configuration.
     *
     * @param config The configuration
     */
    public Producer(Configuration config) {
        this.config = config;    
    }

    /**
     * Produces the given result object, serializing to the given media type
     * and streaming it using the specified output stream.
     * The given charset is used to encode the stream.
     * The producer method is chosen accordingly to the given
     * media type. The responsible producer method must contain this media type
     * in its @Produces annotation.
     *
     * @param accepted The comma separated enumeration of accepted media types
     * @param result The result object to serialize
     * @param outputStream The output stream
     * @param charset The charset encoding
     * @throws ControllerException if the production process fails
     */
    public void produce(String accepted, Object result, OutputStream outputStream, String charset) throws ControllerException {
        Method producerMethod = AnnotationUtils.getProducerMethod(this, accepted);
        try {
            producerMethod.invoke(this, result, outputStream, charset);
        } catch (InvocationTargetException e) {
            Throwable cause = e.getCause();
            if (cause instanceof ControllerException) {
                throw (ControllerException)cause;
            }
            throw new ControllerException("The producer method failed during execution: " + producerMethod, cause);
        } catch (IllegalAccessException e) {
            throw new ControllerException("The producer method could not be invoked: " + producerMethod, e);
        }
    }

    public String getBestMatchingMediaType(String accepted) throws ControllerException {
        return AnnotationUtils.getBestMatchingMediaType(this, Produces.class, accepted);
    }

    /**
     * Serializes the result object to plain text, using the specified character encoding.
     * Uses the output stream for streaming.
     *
     * @param result The result
     * @param out The output stream
     * @param charset The character encoding
     * @throws IOException Failed to initialize the writer for the output stream
     */
    @Produces({"text/plain"})
    public void plain(Object result, OutputStream out, String charset) throws IOException {
        String text = result.toString();
        OutputStreamWriter writer = new OutputStreamWriter(out, charset);
        writer.write(text);
        writer.close();
    }

}
