/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.supply;

import net.craftforge.commons.io.StreamUtils;
import net.craftforge.essential.controller.Configuration;
import net.craftforge.essential.controller.ControllerException;
import net.craftforge.essential.controller.annotations.Consumes;
import net.craftforge.essential.controller.constants.Charset;
import net.craftforge.essential.controller.constants.HttpStatusCode;
import net.craftforge.essential.controller.utils.AnnotationUtils;
import net.craftforge.reflection.utils.ReflUtils;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * De-serializes strings to objects.
 *
 * @author Christian Bick
 * @since 02.12.2010
 */
public class Consumer {

    protected Configuration config;

    public Consumer(Configuration config) {
        this.config = config;    
    }

    /**
     * Consumes the given input string, deserializing it to an object of the
     * specified class. The consumer method is chosen accordingly to the given
     * media type. The responsible consumer method must contain this media type
     * in its @Consumes annotation.
     *
     * @param clazz The class of the desired result object
     * @param mediaType The media type (e.g. text/xml)
     * @param input The input string
     * @return The result object
     * @throws ControllerException if the consumption process fails
     */
    public Object consume(Class<?> clazz, String mediaType, String input) throws ControllerException {
        if (AnnotationUtils.isStreamingSupported(this, mediaType)) {
            try {
                InputStream inputStream = new ByteArrayInputStream(input.getBytes(Charset.UTF8));
                return consume(clazz, mediaType, inputStream, Charset.UTF8);
            } catch (UnsupportedEncodingException e) {
                throw new ControllerException("Failed to read input string", e);
            }
        }

        Method method = AnnotationUtils.getConsumerMethod(this, mediaType);
        try {
            return method.invoke(this, clazz, input);
        } catch (InvocationTargetException e) {
            Throwable cause = e.getCause();
            if (cause instanceof ControllerException) {
                throw (ControllerException)cause;
            }
            throw new ControllerException("The consumer method failed during execution: " + method, cause);
        } catch (IllegalAccessException e) {
            throw new ControllerException("The consumer method could not be invoked: " +
                    "" + method, e);
        }
    }

    /**
     * Consumes the given input stream, deserializing it to an object of the
     * specified class. The given charset is used to decode the stream.
     * The consumer method is chosen accordingly to the given
     * media type. The responsible consumer method must contain this media type
     * in its @Consumes annotation.
     *
     * @param clazz The class of the desired result object
     * @param mediaType The media type (e.g. text/xml)
     * @param inputStream The input stream
     * @param charset The charset
     * @return The result object
     * @throws ControllerException if the consumption process fails
     */
    public Object consume(Class<?> clazz, String mediaType, InputStream inputStream, String charset) throws ControllerException {
        try {
            if (inputStream.available() == 0) {
                return null;
            } else if (! AnnotationUtils.isStreamingSupported(this, mediaType)) {
                return consume(clazz, mediaType, StreamUtils.readStringFromInputStream(inputStream, charset));

            }
        } catch (IOException e) {
            throw new ControllerException("Failed to read input stream", e);
        }

        Method method = AnnotationUtils.getConsumerMethod(this, mediaType);
        try {
            return method.invoke(this, clazz, inputStream, charset);
        } catch (IllegalAccessException e) {
            Throwable cause = e.getCause();
            if (cause instanceof ControllerException) {
                throw (ControllerException)cause;
            }
            throw new ControllerException("The consumer method failed during execution: " + method, cause);
        } catch (InvocationTargetException e) {
            throw new ControllerException("The consumer method could not be invoked: " + method, e);
        }
    }

    /**
     * Consumes the given input string, deserializing it to an object of the
     * specified class.
     *
     * @param clazz The class of the desired result object
     * @param input The input stream
     * @return The result object
     * @throws ControllerException Failed to find a convenient method to initialize object from input
     * @throws InstantiationException Failed to instantiate object from constructor
     * @throws InvocationTargetException Failed to invoke 'valueOf' method
     * @throws IllegalAccessException Failed to invoke constructor or 'valueOf' method
     */
    @Consumes({"text/plain"})
    public Object plain(Class<?> clazz, String input) throws ControllerException, InstantiationException, InvocationTargetException, IllegalAccessException {
        if (clazz.isPrimitive()) {
            clazz = ReflUtils.primitiveToObject(clazz);
        }
        try {
            Constructor constructor = clazz.getConstructor(new Class[] {String.class});
            return constructor.newInstance(input);
        } catch (NoSuchMethodException e) {
            try {
                Method method = clazz.getMethod("valueOf", new Class[] {String.class});
                return method.invoke(null, input);
            } catch (NoSuchMethodException e1) {
                throw new ControllerException("Unable to find a convenient method to initialize object from input");
            }
        } catch (InvocationTargetException e) {
            Throwable cause = e.getCause();
            if (cause instanceof NumberFormatException) {
                throw new ControllerException("Unable to interpret the value " + input +  " as " + clazz.getSimpleName(),
                        cause, HttpStatusCode.BadRequest);
            }
            throw e;
        }
    }
}
