/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller.utils;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Utilities for controller actions.
 *
 * @author Christian Bick
 * @since 02.12.2010
 */
public class UriUtils {

    /**
     * Replaces all duplicate slashes of an URI and removes
     * leading as well as following slashes.
     *
     * @param uri The URI
     * @return The slimmed URI
     */
    public static String slimUri(String uri) {
        uri = uri.replaceAll("([^://])/+", "$1/");
        if (uri.endsWith("/")) {
            uri = uri.substring(0, uri.length()-1);
        }
        if (uri.startsWith("/")) {
            uri = uri.substring(1, uri.length());
        }
        return uri;
    }

    /**
     * Replaces all duplicate slashes of an URI,
     * removes all following slashes and ensures
     * a leading slash.
     *
     * @param uri The URI
     * @return The standardized URI
     */
    public static String standardUri(String uri) {
        if (uri.endsWith("/")) {
            uri = uri.substring(0, uri.length()-1);
        }
        if (! uri.startsWith("/")) {
            uri = "/" + uri;
        }
        return uri;
    }

    /**
     * Gets the parameters from an uri query part as a map-list combination.
     * Accepts params with same name and multiple values ("param=5&param=6") as well as array like
     * params ("param[0]=7&param[1]=8").
     *
     * @param encoding The URL encoding
     * @param uriQueryPart The query part of the uri without leading question mark
     * @return The parameters as a map-list combination
     * @throws UnsupportedEncodingException if encoding is not supported by URLDecoder
     */
    public static Map<String, String[]> getUrlParameters(String encoding, String uriQueryPart)
        throws UnsupportedEncodingException {

        uriQueryPart = uriQueryPart.replaceAll("\\[[0-9]+\\]", "");

        Map<String, List<String>> paramWorkingMap = new HashMap<String, List<String>>();
        for (String param : uriQueryPart.split("&")) {
            String pair[] = param.split("=");
            String paramName = URLDecoder.decode(pair[0], encoding);
            String paramValue = "";
            if (pair.length == 2) {
                 URLDecoder.decode(pair[1], encoding);
            }
            List<String> paramValues = paramWorkingMap.get(paramName);
            if (paramValues == null) {
                paramValues = new ArrayList<String>();
                paramWorkingMap.put(paramName, paramValues);
            }
            paramValues.add(paramValue);
        }

        Map<String, String[]> paramReturnMap = new HashMap<String, String[]>();
        for (String paramName : paramWorkingMap.keySet()) {
            List<String> paramValues = paramWorkingMap.get(paramName);
            String[] paramValueArray = new String[paramValues.size()];
            paramReturnMap.put(paramName, paramValues.toArray(paramValueArray));
        }
        return paramReturnMap;
    }

    /**
     * Splits a resource path into path parts.
     *
     * @param path The resource path
     * @return The path parts
     */
    public static String[] splitPath(String path) {
        String[] pathParts = RegExUtils.FIND_SLASH.split(path);
        // Splitting "/" will return an empty array, so this case has to handled manually
        if (pathParts.length == 0) {
            pathParts = new String[] { "" };
        }
        return pathParts;
    }
}
