/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller.utils;

import net.craftforge.commons.collections.CaseInsensitiveMap;
import org.apache.commons.codec.binary.Base64;

import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

/**
 * @author Christian Bick
 * @since 29.08.11
 */
public class HeaderUtils {

    /**
     * Gets an HTTP basic authorization header value from a username and password.
     *
     * @param username The username
     * @param password The password
     * @return The HTTP basic authorization header
     */
    public static String getBasicAuthorizationHeader(String username, String password) {
        String authPair = username + ":" + password;
        return "Basic " + new String(Base64.encodeBase64(authPair.getBytes()));
    }

    /**
     * Gets a username password pair from the HTTP authorization header values.
     *
     * @param authorizationHeaders HTTP authorization header values
     * @return The username password pair as Array of length 2
     */
    public static String[] getAuthorizationPair(String[] authorizationHeaders) {
        if (authorizationHeaders == null || authorizationHeaders.length < 1 || authorizationHeaders[0] == null) {
            return new String[] { null, null };
        }
        String authorizationHeader = authorizationHeaders[0];

        StringTokenizer st = new StringTokenizer(authorizationHeader);
        String basic = st.nextToken();

        if (basic == null || ! basic.equalsIgnoreCase("Basic")) {
            return new String[] { null, null };
        }

        String credentials = st.nextToken();
        String userPass = new String(Base64.decodeBase64(credentials));
        int pivot = userPass.indexOf(":");
        if (pivot == -1 ) {
            return new String[] { null, null };
        }

        String user = userPass.substring(0, pivot);
        String pass = userPass.substring(pivot+1);

        return new String[] { user, pass };
    }

    /**
     * Gets a content type pair, extracting the content media type
     * and content charset from a Content-Type header.
     *
     * @param contentTypeHeader The Content-Type header
     * @return The content type pair
     */
    public static String[] getContentTypePair(String contentTypeHeader) {
        if (contentTypeHeader == null || contentTypeHeader.isEmpty()) {
            return new String[] { null, null };
        }
        if (! contentTypeHeader.contains(";")) {
            return new String[] { contentTypeHeader.trim() , null };
        }
        int pivot = contentTypeHeader.indexOf(";");
        String mediaType = contentTypeHeader.substring(0, pivot).trim();
        String charsetCandidate = contentTypeHeader.substring(pivot+1).trim();

        String charset = null;

        if (RegExUtils.FIND_CHARSET.matcher(charsetCandidate).matches()) {
            charset = RegExUtils.FIND_CHARSET.matcher(charsetCandidate).replaceAll("$1");
        } else if (RegExUtils.FIND_NO_EQUAL_CONTAINED.matcher(charsetCandidate).matches()) {
            charset = charsetCandidate;
        }
        return new String[] { mediaType, charset };
    }

    /**
     * Converts an input map holding its values as a list into an output map holding
     * its values as an array.
     *
     * @param inputMap The input map
     * @return The output map
     */
    public static Map<String, String[]> normalizeHeaderMap(Map<String, List<String>> inputMap) {
        Map<String, String[]> normalizedMap = new CaseInsensitiveMap<String[]>();
        for (String headerName : inputMap.keySet()) {
            List<String> headerValueList = inputMap.get(headerName);
            String[] headerValues = headerValueList.toArray(new String[headerValueList.size()]);
            normalizedMap.put(headerName, headerValues);
        }
        return normalizedMap;
    }
}
