/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller.utils;

import net.craftforge.essential.controller.ControllerException;
import net.craftforge.essential.controller.annotations.*;
import net.craftforge.essential.controller.constants.HttpMethod;
import net.craftforge.essential.controller.constants.HttpStatusCode;
import net.craftforge.reflection.managers.ClassManager;

import java.io.InputStream;
import java.io.OutputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.*;

/**
 * Utilities for annotation actions.
 *
 * @author Christian Bick
 * @since 17.02.2011
 */
public class AnnotationUtils {

    /**
     * Gets the @Path annotation value of a class, looking it up on type level. Uses the complete class
     * hierarchy, including interfaces.
     *
     * @param clazz The class
     * @return The @Path annotation value, null if not found
     */
    public static String getPathFromClass(Class<?> clazz) {
        Path annotation = (Path) ClassManager.getInstance(clazz).getTypeLevelAnnotation(Path.class);
        return annotation == null ? null : UriUtils.standardUri(annotation.value());
    }

    /**
     * Gets the @Path annotation value of a method, looking it up on method level. Uses the complete class
     * hierarchy, including interfaces.
     *
     * @param method The method
     * @return The @Path annotation value, null if not found
     */
    public static String getPathFromMethod(Method method) {
        Path annotation = (Path) ClassManager.getInstance(method).getMethodLevelAnnotation(method, Path.class);
        return annotation == null ? null : UriUtils.standardUri(annotation.value());
    }

    /**
     * Gets the @Param annotations value of a property, looking it up on field level
     * and on method level of getters and setters. Uses the complete class
     * hierarchy, including interfaces.
     *
     * @param field The field
     * @return The @Param annotation value, null if not found
     */
    public static String getParamFromProperty(Field field) {
        Param annotation = (Param) ClassManager.getInstance(field).getPropertyLevelAnnotation(field, Param.class);
        return annotation == null ? null : annotation.value();
    }

    /**
     * Gets the @Header annotations value of a property, looking it up on field level
     * and on method level of getters and setters. Uses the complete class
     * hierarchy, including interfaces.
     *
     * @param field The field
     * @return The @Header annotation value, null if not found
     */
    public static String getHeaderFromProperty(Field field) {
        Header annotation = (Header) ClassManager.getInstance(field).getPropertyLevelAnnotation(field, Header.class);
        return annotation == null ? null : annotation.value();
    }

    /**
     * Gets the @Property annotation value of a property, looking it up on field level
     * and on method level of getters and setters. Uses the complete class
     * hierarchy, including interfaces.
     *
     * @param field The field
     * @return The @Property annotation value, null if not found
     */
    public static String getPropertyFromProperty(Field field) {
        Property annotation = (Property) ClassManager.getInstance(field).getPropertyLevelAnnotation(field, Property.class);
        return annotation == null ? null : annotation.value();
    }

    /**
     * Determines whether the @Body annotation is present on a property, looking it up on field level
     * and on method level of getters and setters. Uses the complete class
     * hierarchy, including interfaces.
     *
     * @param field The field
     * @return The @Header annotation value, null if not found
     */
    public static boolean isBodyOnProperty(Field field) {
        Body annotation = (Body) ClassManager.getInstance(field).getPropertyLevelAnnotation(field, Body.class);
        return annotation != null;
    }

    /**
     * Gets the @DefaultValue annotation value(s) of a field, looking it up on field level
     * and on method level of getters and setters. Uses the complete class
     * hierarchy, including interfaces.
     *
     * @param field The field
     * @return The @DefaultValue annotation value, null if not found
     */
    public static String[] getDefaultValuesFromProperty(Field field) {
        DefaultValue annotation =(DefaultValue) ClassManager.getInstance(field).getPropertyLevelAnnotation(field, DefaultValue.class);
        return annotation == null ? null : annotation.value();
    }

    /**
     * Gets the method of a producer being annotated with the @Produces annotation
     * which value matches best the specified media types.
     *
     * @param producer The producer method
     * @param mediaTypes The media types
     * @return The method matching best the annotation requirements
     * @throws ControllerException Failed to find a producer method supplying one of the accepted media types
     */
    public static Method getProducerMethod(Object producer, String mediaTypes) throws ControllerException {
        Method method = getBestMatchingMethod(producer, Produces.class, mediaTypes);
        if (! isValidProducerMethod(method)) {
            throw new ControllerException("Producer method " + method.getName() + " has an invalid signature");
        }
        return method;
    }

    /**
     * Gets the method of a consumer being annotated with the @Consumes annotation
     * which value matches best the media types.
     *
     * @param consumer The consumer
     * @param mediaTypes The media types
     * @return The method matching best the annotation requirements
     * @throws ControllerException Failed to find a consumer method supplying one of the accepted media types
     */
    public static Method getConsumerMethod(Object consumer, String mediaTypes) throws ControllerException {
        Method method = getBestMatchingMethod(consumer, Consumes.class, mediaTypes);
        if (! isValidConsumerMethod(method)) {
            throw new ControllerException("Consumer method " + method.getName() + " has an invalid signature");
        }
        return method;
    }

    /**
     * Gets the method of a consumer or producer being annotated with the @Produces or @Consumes annotation
     * which value matches best the specified media types. The annotation type is used to distinct
     * between a consumer and a producer.
     *
     * @param supplier The supplier
     * @param annotationType The annotation type.
     * @param mediaTypes The media types
     * @return The method matching best the annotation requirements
     * @throws ControllerException Failed to find a producer method supplying one of the accepted media types
     */
    public static Method getBestMatchingMethod(Object supplier, Class<? extends Annotation> annotationType, String mediaTypes) throws ControllerException {
        int maxQuality = 0;
        Method bestMethod = null;
        for (Method method : supplier.getClass().getMethods()) {
            if (! method.isAnnotationPresent(annotationType)) {
                continue;
            }
            Annotation annotation = method.getAnnotation(annotationType);
            String[] value;
            if (annotationType.equals(Produces.class)) {
                Produces produces = (Produces)annotation;
                value = produces.value();
            } else if (annotationType.equals(Consumes.class)) {
                Consumes consumes = (Consumes)annotation;
                value = consumes.value();
            } else {
                throw new ControllerException("Unsupported annotation type: " + annotationType);
            }
            for (String produces : value) {
                int matchingQuality = getMatchingQuality(produces, mediaTypes);
                if (matchingQuality > maxQuality) {
                    bestMethod = method;
                    maxQuality = matchingQuality;
                }
            }
        }
        if (bestMethod == null) {
            throw new ControllerException("Supplier class " + supplier.getClass() + " does not supply any of the " +
                    "media types: " + mediaTypes, HttpStatusCode.UnsupportedMediaType);
        }
        return bestMethod;
    }

     /**
     * Gets the media type of a consumer or producer being annotated with the @Produces or @Consumes annotation
     * which value matches best the specified media types. The annotation type is used to distinct
     * between a consumer and a producer.
     *
     * @param supplier The supplier
     * @param annotationType The annotation type.
     * @param mediaTypes The media types
     * @return The media type matching best the annotation requirements
     * @throws ControllerException Failed to find a producer method supplying one of the accepted media types
     */
    public static String getBestMatchingMediaType(Object supplier, Class<? extends Annotation> annotationType, String mediaTypes) throws ControllerException {
        int maxQuality = 0;
        String bestMediaType = null;
        for (Method method : supplier.getClass().getMethods()) {
            if (! method.isAnnotationPresent(annotationType)) {
                continue;
            }
            Annotation annotation = method.getAnnotation(annotationType);
            String[] value;
            if (annotationType.equals(Produces.class)) {
                Produces produces = (Produces)annotation;
                value = produces.value();
            } else if (annotationType.equals(Consumes.class)) {
                Consumes consumes = (Consumes)annotation;
                value = consumes.value();
            } else {
                throw new ControllerException("Unsupported annotation type: " + annotationType);
            }
            for (String produces : value) {
                int matchingQuality = getMatchingQuality(produces, mediaTypes);
                if (matchingQuality > maxQuality) {
                    bestMediaType = produces;
                    maxQuality = matchingQuality;
                }
            }
        }
        if (bestMediaType == null) {
            throw new ControllerException("Supplier class " + supplier.getClass() + " does not supply any of the " +
                    "media types: " + mediaTypes, HttpStatusCode.UnsupportedMediaType);
        }
        return bestMediaType;
    }

    /**
     * Gets the matching quality of an accepted and an available media type enumeration.
     * The quality depends on how good the best of the available media types matches against
     * the accepted media type. The higher the number, the better is the quality.<br>
     * <br>
     * 0: None of the available media types matches.<br>
     * 1: The best match was found at &#42;/&#42; vs. static/static<br>
     * 2: The best match was found at &#42;/&#42; vs. static/&#42;<br>
     * 3: The best match was found at &#42;/&#42; vs. &#42;/&#42;<br>
     * 4: The best match was fount at static/&#42; vs static/static<br>
     * 5: The best match was found at static/&#42; vs. static/&#42;<br>
     * 6: The best match was found at static/static vs. static/static (exact match)<br>
     *
     * @param availableEnum The comma separated enumeration of available media types
     * @param acceptedEnum The comma separated enumeration of accepted media types
     * @return The quality as a figure between 0 (lowest) and 6 (highest)
     * @throws ControllerException Invalid media type declaration
     */
    private static int getMatchingQuality(String availableEnum, String acceptedEnum) throws ControllerException {
        int maxQuality = 0;
        for (String acceptedType : acceptedEnum.split(",")) {
            acceptedType = acceptedType.split(";")[0];
            acceptedType = acceptedType.trim();
            String[] acceptedParts = acceptedType.split("/");
            if (acceptedParts.length == 1) {
                acceptedParts = new String[] { acceptedParts[0], "*" };   
            }
            for (String availableType : availableEnum.split(",")) {
                availableType = availableType.trim();
                String[] availableParts = availableType.split("/");
                if (availableParts.length == 1) {
                    availableParts = new String[] { availableParts[0], "*" };   
                }
                int currentQuality = 0;
                if (availableParts[0].equals("*") && availableParts[1].equals("*")) {
                    if (acceptedParts[0].equals("*") && acceptedParts[1].equals("*")) {
                        currentQuality = 3;
                    } else if (acceptedParts[1].equals("*")) {
                        currentQuality = 2;
                    } else {
                        currentQuality = 1;
                    }
                } else if (availableParts[0].equals(acceptedParts[0]) && availableParts[1].equals("*")) {
                    if (acceptedParts[1].equals("*")) {
                        currentQuality = 5;
                    } else {
                        currentQuality = 4;
                    }
                } else if (availableParts[0].equals(acceptedParts[0]) && availableParts[1].equals(acceptedParts[1])) {
                    currentQuality = 6;
                }
                maxQuality = Math.max(currentQuality, maxQuality);
            }
        }
        return maxQuality;
    }
    /**
     * Gets the @Producer annotation value of a method, looking it up on method level first, then class level.
     * Uses the complete class hierarchy, including interfaces.
     *
     * @param method The method
     * @return The @Producer annotation value or null if not found
     */
    public static Class<?> getProducerFromMethodOrClass(Method method) {
        Producer annotation = (Producer) ClassManager.getInstance(method).getMethodOrTypeLevelAnnotation(method, Producer.class);
        return annotation == null ? null : annotation.value();
    }

    /**
     * Gets the @Consumer annotation value of a method, looking it up on method level first, then class level.
     * Uses the complete class hierarchy, including interfaces.
     *
     * @param method The method
     * @return The @Consumer annotation value or null if not found
     */
    public static Class<?> getConsumerFromMethodOrClass(Method method) {
        Consumer annotation = (Consumer) ClassManager.getInstance(method).getMethodOrTypeLevelAnnotation(method, Consumer.class);
        return annotation == null ? null : annotation.value();
    }

    /**
     * Checks if the @Public annotation is present, looking it up on method level first, then class level.
     * Uses the complete class hierarchy, including interfaces.
     *
     * @param method The method
     * @return Whether the @Public annotation is present or not
     */
    public static boolean isPublic(Method method) {
        Public annotation = (Public) ClassManager.getInstance(method).getMethodOrTypeLevelAnnotation(method, Public.class);
        return annotation != null;
    }

    /**
     * Gets a list of all fields with an @Param annotation, looking them up on field level
     * and on method level of getters and setters. Uses the complete class
     * hierarchy, including interfaces.
     *
     * @param clazz The class
     * @return The list of fields
     */
    public static List<Field> getParamFieldsFromClass(Class<?> clazz) {
        return ClassManager.getInstance(clazz).getAllPropertiesAnnotatedWith(Param.class);
    }

    /**
     * Gets a list of all fields with an @Header annotation, looking them up on field level
     * and on method level of getters and setters. Uses the complete class
     * hierarchy, including interfaces.
     *
     * @param clazz The class
     * @return The list of fields
     */
    public static List<Field> getHeaderFieldsFromClass(Class<?> clazz) {
        return ClassManager.getInstance(clazz).getAllPropertiesAnnotatedWith(Header.class);
    }

    /**
     * Gets a list of all fields with an @Body annotation, looking them up on field level
     * and on method level of getters and setters. Uses the complete class
     * hierarchy, including interfaces.
     *
     * @param clazz The class
     * @return The list of fields
     */
    public static List<Field> getBodyFieldsFromClass(Class<?> clazz) {
        return ClassManager.getInstance(clazz).getAllPropertiesAnnotatedWith(Body.class);
    }

    /**
     * Gets a list of all fields with an @Property annotation, looking them up on field level
     * and on method level of getters and setters. Uses the complete class
     * hierarchy, including interfaces.
     *
     * @param clazz The class
     * @return The list of fields
     */
    public static List<Field> getPropertyFieldsFromClass(Class<?> clazz) {
        return ClassManager.getInstance(clazz).getAllPropertiesAnnotatedWith(Property.class);
    }

    /**
     * Gets a list of all fields with an @Param, @Body, @Header or @Property annotation, looking them
     * up on field level and on method level of getters and setters. Uses the complete class
     * hierarchy, including interfaces.
     *
     * @param clazz The class
     * @return The list of fields
     */
    public static List<Field> getAnnotatedFieldsFromClass(Class<?> clazz) {
        List<Field> annotatedFields = new LinkedList<Field>();
        annotatedFields.addAll(getParamFieldsFromClass(clazz));
        annotatedFields.addAll(getBodyFieldsFromClass(clazz));
        annotatedFields.addAll(getHeaderFieldsFromClass(clazz));
        annotatedFields.addAll(getPropertyFieldsFromClass(clazz));
        return annotatedFields;
    }

    /**
     * Gets a list of all methods with a REST annotation (@Get, @Post, etc.)
     * in the class, looking it up on method level. Uses the complete class
     * hierarchy, including interfaces.
     *
     * @param clazz The class
     * @return The list of resource methods
     */
    public static List<Method> getResourceMethodsFromClass(Class<?> clazz) {
        List<Method> methods = new ArrayList<Method>();
        methods.addAll(ClassManager.getInstance(clazz).getAllMethodsAnnotatedWith(Get.class));
        methods.addAll(ClassManager.getInstance(clazz).getAllMethodsAnnotatedWith(Post.class));
        methods.addAll(ClassManager.getInstance(clazz).getAllMethodsAnnotatedWith(Put.class));
        methods.addAll(ClassManager.getInstance(clazz).getAllMethodsAnnotatedWith(Delete.class));
        methods.addAll(ClassManager.getInstance(clazz).getAllMethodsAnnotatedWith(Head.class));
        methods.addAll(ClassManager.getInstance(clazz).getAllMethodsAnnotatedWith(Trace.class));
        return methods;
    }

    /**
     * Gets the supported producer media types from a producer class.
     *
     * @param clazz The producer class
     * @return The supported media types
     */
    public static List<String> getSupportedProducerMediaTypes(Class<?> clazz) {
        Set<String> mediaTypes = new HashSet<String>();
        List<Method> methods = ClassManager.getInstance(clazz).getAllMethodsAnnotatedWith(Produces.class);
        for (Method method : methods) {
            Produces ann = (Produces)ClassManager.getInstance(method).getMethodLevelAnnotation(method, Produces.class);
            mediaTypes.addAll(Arrays.asList(ann.value()));
        }
        return new ArrayList<String>(mediaTypes);

    }

    /**
     * Gets the supported consumer media types from a consumer class.
     *
     * @param clazz The consumer class
     * @return The supported media types
     */
    public static List<String> getSupportedConsumerMediaTypes(Class<?> clazz) {
        Set<String> mediaTypes = new HashSet<String>();
        List<Method> methods = ClassManager.getInstance(clazz).getAllMethodsAnnotatedWith(Consumes.class);
        for (Method method : methods) {
            Consumes ann = (Consumes)ClassManager.getInstance(method).getMethodLevelAnnotation(method, Consumes.class);
            mediaTypes.addAll(Arrays.asList(ann.value()));
        }
        return new ArrayList<String>(mediaTypes);
    }

    /**
     * Gets all valid consumer parameter type combinations.
     *
     * @return The Array of valid consumer parameter type combinations
     */
    public static Class[][] getValidConsumerParameterTypes() {
        return new Class[][] {
            new Class[] { Class.class, String.class },
            new Class[] { Class.class, InputStream.class, String.class }
        };
    }

    /**
     * Gets all valid producer parameter type combinations.
     *
     * @return The Array of valid producer parameter type combinations
     */
    public static Class[][] getValidProducerParameterTypes() {
        return new Class[][] {
            new Class[] { Object.class, OutputStream.class, String.class }
        };
    }

    /**
     * Checks if a producer method has a valid signature.
     *
     * @param method The producer method
     * @return Whether the producer method is valid or nor
     */
    public static boolean isValidProducerMethod(Method method) {
        return isValidMethod(method, getValidProducerParameterTypes());
    }

    /**
     * Checks if a consumer method has a valid signature.
     *
     * @param method The consumer method
     * @return Whether the consumer method is valid or not
     */
    public static boolean isValidConsumerMethod(Method method) {
        return isValidMethod(method, getValidConsumerParameterTypes());
    }

    /**
     * Checks if a method has a method signature by comparing its
     * parameter types against a set of valid parameter type combinations.
     *
     * @param method The method
     * @param validParameterTypes The valid parameter type combinations
     * @return Whether the method has a valid signature or not
     */
    public static boolean isValidMethod(Method method, Class[][] validParameterTypes) {
        Class<?>[] actualParameterTypes = method.getParameterTypes();
        for (Class[] possibleParameterTypes : validParameterTypes) {
            if (Arrays.equals(actualParameterTypes, possibleParameterTypes)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Checks if for the given media type streaming is supported by the specified consumer.
     *
     * @param consumer The consumer
     * @param mediaType The media type
     * @return Whether streaming is supported or not
     * @throws ControllerException if the specified consumer does not supply the given media type
     */
    public static boolean isStreamingSupported(net.craftforge.essential.supply.Consumer consumer, String mediaType) throws ControllerException {
        Method method = getConsumerMethod(consumer, mediaType);
        return method.getParameterTypes()[1].equals(InputStream.class);
    }

    /**
     * Checks if for the given media type streaming is supported by the specified producer.
     *
     * @param producer The producer
     * @param mediaType The media type
     * @return Whether streaming is supported or not
     * @throws ControllerException if the specified producer does not supply the given media type
     */
    public static boolean isStreamingSupported(net.craftforge.essential.supply.Producer producer, String mediaType) throws ControllerException {
        Method method = getProducerMethod(producer, mediaType);
        return method.getParameterTypes()[1].equals(OutputStream.class);
    }

    /**
     * Gets the REST annotation (@Get, @Post, etc.) of a method, looking it up on method
     * level. Uses the complete class hierarchy, including interfaces.
     *
     * @param method The method
     * @return The REST annotation or null if none found
     */
    public static String getHttpMethod(Method method) {
        Annotation annotation = ClassManager.getInstance(method).getMethodLevelAnnotation(method, Get.class);
        if (annotation != null) {
            return HttpMethod.GET;
        }
        annotation = ClassManager.getInstance(method).getMethodLevelAnnotation(method, Post.class);
        if (annotation != null) {
            return HttpMethod.POST;
        }
        annotation = ClassManager.getInstance(method).getMethodLevelAnnotation(method, Put.class);
        if (annotation != null) {
            return HttpMethod.PUT;
        }
        annotation = ClassManager.getInstance(method).getMethodLevelAnnotation(method, Delete.class);
        if (annotation != null) {
            return HttpMethod.DELETE;
        }
        annotation = ClassManager.getInstance(method).getMethodLevelAnnotation(method, Head.class);
        if (annotation != null) {
            return HttpMethod.HEAD;
        }
        annotation = ClassManager.getInstance(method).getMethodLevelAnnotation(method, Trace.class);
        if (annotation != null) {
            return HttpMethod.TRACE;
        }
        return null;
    }
}
