/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller.phases.flow;

import net.craftforge.essential.controller.ControllerException;
import net.craftforge.essential.controller.Phase;
import net.craftforge.essential.controller.Setup;
import net.craftforge.essential.controller.State;
import net.craftforge.essential.controller.constants.HttpMethod;
import net.craftforge.essential.controller.phases.*;
import net.craftforge.essential.controller.utils.AnnotationUtils;

/**
 * The standard phase flow defines a default flow for phases.
 *
 * @author Christian Bick
 * @since 30.05.2011
 */
public class StandardPhaseFlow implements PhaseFlow {

    private State state;
    private Setup setup;

    /**
     * {@inheritDoc}
     */
    public void setState(State state) {
        this.state = state;
    }

    /**
     * {@inheritDoc}
     */
    public void setSetup(Setup setup) {
        this.setup = setup;
    }

    /**
     * {@inheritDoc}
     */
    public Phase getFirstPhase() {
        if (setup.getHttpMethod().equalsIgnoreCase(HttpMethod.OPTIONS)) {
            return new DocumentationPhase(state, setup);
        } else {
            return new AllocationPhase(state, setup);
        }
    }

    /**
     * {@inheritDoc}
     */
    public Phase getPhaseAfter(Phase phase) throws ControllerException {
        if (phase instanceof AllocationPhase) {
            return getPhaseAfterAllocation();
        } else if (phase instanceof AuthenticationPhase) {
            return getPhaseAfterAuthentication();
        } else if (phase instanceof InvocationPhase) {
            return getPhaseAfterInvocation();
        } else if (phase instanceof DocumentationPhase) {
            return getPhaseAfterDocumentation();
        } else if (phase instanceof ProductionPhase) {
            return getPhaseAfterProduction();
        } else {
            throw new ControllerException("Undefined flow for phase: " + phase.getClass());
        }
    }

    /**
     * Gets the phase following an allocation phase.
     *
     * @return The phase following an allocation phase
     */
    protected Phase getPhaseAfterAllocation() {
        // The authentication phase is skipped if no authentication handler is given or the resource method is
        // annotated as public
        if (setup.getConfiguration().getAuthenticationHandler() == null
                || AnnotationUtils.isPublic(state.getResourceMethod())) {
            return getPhaseAfterAuthentication();
        }
        return new AuthenticationPhase(state, setup);
    }

    /**
     * Gets the phase following an authentication phase.
     *
     * @return The phase following an authentication phase
     */
    protected Phase getPhaseAfterAuthentication() {
        return new InvocationPhase(state, setup);
    }

    /**
     * Gets the phase following an invocation phase.
     *
     * @return The phase following an invocation phase
     */
    protected Phase getPhaseAfterInvocation() {
        return new ProductionPhase(state, setup);
    }

    /**
     * Gets the phase following a documentation phase.
     *
     * @return The phase following a documentation phase
     */
    protected Phase getPhaseAfterDocumentation() {
        return new ProductionPhase(state, setup);
    }

    /**
     * Gets the phase following a production phase.
     *
     * @return The phase following a production phase
     */
    protected Phase getPhaseAfterProduction() {
        // The production phase is the last phase of the flow
        return null;
    }
}