/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller.phases;

import net.craftforge.essential.controller.*;
import net.craftforge.essential.controller.constants.HttpHeader;
import net.craftforge.essential.controller.constants.HttpMethod;
import net.craftforge.essential.controller.utils.AnnotationUtils;
import net.craftforge.essential.supply.Producer;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;

/**
 * Produces a serialized response from the result.
 *
 * @author Christian Bick
 * @since 17.01.2011
 */
public class ProductionPhase implements Phase {

    private State state;
    private Setup setup;

    /**
     * Creates a production phase from a controller state and setup.
     *
     * @param state The controller state
     * @param setup The controller setup
     */
    public ProductionPhase(State state, Setup setup) {
        this.state = state;
        this.setup = setup;
    }

    /**
     * {@inheritDoc}
     */
    public void run() throws ControllerException {
        try {
            if (state.getResult() == null) {
                // If the result is null, only send headers
                setup.getResponse().sendHeaders(state.getStatus(), getHeadersWithoutContent());
            } else {
                // Otherwise obtain the producer before sending the headers
                Producer producer = getProducer();
                // Send the headers
                setup.getResponse().sendHeaders(state.getStatus(), getHeadersWithContent(producer));
                // Produce the response body
                producer.produce(
                    setup.getAcceptedMediaType(),
                    state.getResult(),
                    setup.getResponseBodyOutputStream(),
                    setup.getAcceptedCharset()
                );
            }

        } catch (RuntimeException e) {
            throw new ControllerException("An unexpected exception occurred during production", e);
        }
    }

    protected Map<String, String[]> getHeadersWithoutContent() throws ControllerException {
        return new HashMap<String, String[]>();
    }

    protected Map<String, String[]> getHeadersWithContent(Producer producer) throws ControllerException {
        Map<String, String[]> headers = getHeadersWithoutContent();
        String mediaType = producer.getBestMatchingMediaType(setup.getAcceptedMediaType());
        String charset = setup.getAcceptedCharset();
        headers.put(HttpHeader.CONTENT_TYPE, new String[] { mediaType + "; charset=" + charset } );
        return headers;
    }

    protected Producer getProducer() throws ControllerException {
        Class<?> producerClass = null;
        // Try to get the producer class from an annotation, except in case of OPTIONS
        if (! setup.getHttpMethod().equals(HttpMethod.OPTIONS)) {
            producerClass = AnnotationUtils.getProducerFromMethodOrClass(state.getResourceMethod());
        }
        // If producer class was not set via annotation, then take the default one
        if (producerClass == null) {
            producerClass = setup.getConfiguration().getDefaultProducer();
        }

        // Instantiate the producer
        Producer producer;
        try {
            try {
                // Try to call constructor with configuration
                producer = (Producer)producerClass.getConstructor(Configuration.class).newInstance(setup.getConfiguration());
            } catch (NoSuchMethodException e) {
                // Call default constructor of constructor with configuration does not exist
                producer = (Producer)producerClass.newInstance();
            }
        } catch (InstantiationException e) {
            throw new ControllerException("The producer class specified in the controller" +
                    " configuration or annotated at the resource method could not be" +
                    " instantiated: " + producerClass.getName(), e);
        } catch (IllegalAccessException e) {
            throw new ControllerException("The producer class specified in the controller" +
                    " configuration or annotated at the resource method could not be" +
                    " instantiated: " + producerClass.getName(), e);
        } catch (InvocationTargetException e) {
            throw new ControllerException("The producer class specified in the controller" +
                    " configuration or annotated at the resource method could not be" +
                    " instantiated: " + producerClass.getName(), e);
        }
        return producer;
    }
}
