/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller.phases;

import net.craftforge.essential.controller.*;
import net.craftforge.essential.controller.constants.HttpStatusCode;
import net.craftforge.essential.controller.managers.InjectionManager;
import net.craftforge.essential.controller.utils.AnnotationUtils;
import net.craftforge.essential.supply.Consumer;
import net.craftforge.reflection.utils.PropertyUtils;
import net.craftforge.reflection.utils.ReflUtils;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.util.Date;
import java.util.List;

/**
 * Creates an instance of the resource class, initializes annotated class properties
 * and invokes the resource method. Sets the result of the resource method.
 *
 * @author Christian Bick
 * @since 03.02.2011
 */
public class InvocationPhase implements Phase {

    private State state;
    private Setup setup;

    /**
     * Creates an invocation phase from a controller state and setup.
     *
     * @param state The controller state
     * @param setup The controller setup
     */
    public InvocationPhase(State state, Setup setup) {
        this.state = state;
        this.setup = setup;
    }

    /**
     * {@inheritDoc}
     */
    public void run() throws ControllerException {
        try {
            // Initialize consumer
            Class<?> consumerClass = AnnotationUtils.getConsumerFromMethodOrClass(state.getResourceMethod());
            if (consumerClass == null) {
                consumerClass = setup.getConfiguration().getDefaultConsumer();
            }

            Consumer consumer;
            try {
                try {
                    consumer = (Consumer)consumerClass.getConstructor(Configuration.class).newInstance(setup.getConfiguration());
                } catch (NoSuchMethodException e) {
                    consumer = (Consumer)consumerClass.newInstance();
                }
            } catch (InstantiationException e) {
                throw new ControllerException("The consumer class specified in the controller" +
                        " configuration or annotated at the resource method could not be" +
                        " instantiated: " + consumerClass.getName(), e);
            } catch (IllegalAccessException e) {
                throw new ControllerException("The consumer class specified in the controller" +
                        " configuration or annotated at the resource method could not be" +
                        " instantiated: " + consumerClass.getName(), e);
            } catch (InvocationTargetException e) {
                throw new ControllerException("The consumer class specified in the controller" +
                        " configuration or annotated at the resource method could not be" +
                        " instantiated: " + consumerClass.getName(), e);
            }

            // Initialize injection manager
            InjectionManager injectionManager = new InjectionManager(
                    consumer,
                    setup.getBodyHandler(),
                    setup.getHeaderHandler(),
                    setup.getParameterHandler(),
                    setup.getPropertyHandler()
            );

            // Initialize resource object
            Object resource;
            try {
                resource = state.getResourceClass().newInstance();
            } catch (InstantiationException e) {
                Throwable cause = e.getCause();
                if (cause instanceof ControllerException) {
                    throw (ControllerException)cause;
                }
                throw new ControllerException("The allocated resource class could not be" +
                        " instantiated: " + state.getResourceClass(), cause);
            } catch (IllegalAccessException e) {
                throw new ControllerException("The allocated resource class could not be" +
                        " instantiated: " + state.getResourceClass(), e);
            }

            for (Field field : AnnotationUtils.getAnnotatedFieldsFromClass(resource.getClass())) {
                try {
                    PropertyUtils.setProperty(resource, field, injectionManager.getInputForField(field));
                } catch (NoSuchMethodException e) {
                    throw new ControllerException("A property of the resource class could not be" +
                            " set via the setter method: " + ReflUtils.setterName(field.getName()) + "()" +
                            " in resource class " + state.getResourceClass().getName(), e);
                } catch (InvocationTargetException e) {
                    Throwable cause = e.getCause();
                    if (cause instanceof ControllerException) {
                        throw (ControllerException)cause;
                    }
                    throw new ControllerException("A property of the resource class could not be" +
                            " set via the setter method: " + ReflUtils.setterName(field.getName()) + "()" +
                            " in resource class " + state.getResourceClass(), cause);
                } catch (IllegalAccessException e) {
                    throw new ControllerException("A property of the resource class could not be" +
                            " set: " + field.getName() +
                            " in resource class " + state.getResourceClass(), e);
                }
            }

            // Invoke resource method
            List<Object> parameterValues = injectionManager.getInputForMethodParameters(state.getResourceMethod());
            Object result = null;
            state.setInvocationStart(new Date());
            try {
                if (! state.getResourceMethod().getReturnType().equals(void.class)) {
                    if (parameterValues.size() == 0) {
                        result = state.getResourceMethod().invoke(resource);
                    } else {
                        result = state.getResourceMethod().invoke(resource, parameterValues.toArray());
                    }
                    state.setStatus(HttpStatusCode.OK);
                } else {
                    if (parameterValues.size() == 0) {
                        state.getResourceMethod().invoke(resource);
                    } else {
                        state.getResourceMethod().invoke(resource, parameterValues.toArray());
                    }
                    state.setStatus(HttpStatusCode.NoContent);
                }
            } catch (InvocationTargetException e) {
                Throwable cause = e.getCause();
                if (cause instanceof ControllerException) {
                    throw (ControllerException)cause;
                }
                throw new ControllerException("The resource method failed during execution: " +
                        "" + state.getResourceMethod(), cause);
            } catch (IllegalAccessException e) {
                throw new ControllerException("The resource method could not be invoked: " +
                        "" + state.getResourceMethod(), e);
            }
            state.setInvocationEnd(new Date());
            state.setResult(result);
        } catch (RuntimeException e) {
            throw new ControllerException("An unexpected exception occurred during resource method invocation", e);
        }
    }
}
