/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller.phases;

import net.craftforge.essential.controller.ControllerException;
import net.craftforge.essential.controller.Phase;
import net.craftforge.essential.controller.Setup;
import net.craftforge.essential.controller.State;
import net.craftforge.essential.controller.constants.HttpStatusCode;
import net.craftforge.essential.controller.handlers.AuthenticationHandler;

/**
 * Authenticates the requester via HTTP basic authorization.
 *
 * @author Christian Bick
 * @since 22.05.2011
 */
public class AuthenticationPhase implements Phase {

    private State state;
    private Setup setup;

    /**
     * Creates an authentication phase from a controller state and setup.
     *
     * @param state The controller state
     * @param setup The controller setup
     */
    public AuthenticationPhase(State state, Setup setup) {
        this.state = state;
        this.setup = setup;
    }

    /**
     * {@inheritDoc}
     */
    public void run() throws ControllerException {
        try {
            Class<?> handlerClass = setup.getConfiguration().getAuthenticationHandler();
            if (handlerClass == null ) {
                return;
            }

            AuthenticationHandler handler;
            try {
                handler = (AuthenticationHandler)handlerClass.newInstance();
            } catch (InstantiationException e) {
                throw new ControllerException("The authentication handler class specified in the controller" +
                        " configuration could not be instantiated: " + handlerClass.getName(), e);
            } catch (IllegalAccessException e) {
                throw new ControllerException("The authentication handler class specified in the controller" +
                        " configuration could not be instantiated: " + handlerClass.getName(), e);
            }
            String user = setup.getHeaderHandler().getAuthorizationUser();
            String pass = setup.getHeaderHandler().getAuthorizationPassword();

            if (user == null || pass == null) {
                throw new ControllerException("Authentication failed because no user and/or password " +
                        " was transmitted in the authorization header", HttpStatusCode.Unauthorized);
            }
            if (! handler.authenticate(setup)) {
                throw new ControllerException("Authentication failed because the user does not exist or" +
                        " user and password do not match", HttpStatusCode.Unauthorized);
            }

        } catch (RuntimeException e) {
            throw new ControllerException("An unexpected exception occurred during authentication", e);
        }
    }
}
