/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller.phases;

import net.craftforge.essential.controller.ControllerException;
import net.craftforge.essential.controller.Phase;
import net.craftforge.essential.controller.Setup;
import net.craftforge.essential.controller.State;
import net.craftforge.essential.controller.constants.HttpStatusCode;
import net.craftforge.essential.controller.managers.ResourceManager;

import java.lang.reflect.Method;

/**
 * Allocates and sets the resource class and resource method.
 *
 * @author Christian Bick
 * @since 17.01.2011
 */
public class AllocationPhase implements Phase {

    private State state;
    private Setup setup;

    /**
     * Creates an allocation phase from a controller state and setup.
     *
     * @param state The controller state
     * @param setup The controller setup
     */
    public AllocationPhase(State state, Setup setup) {
        this.state = state;
        this.setup = setup;
    }

    /**
     * {@inheritDoc}
     */
    public void run() throws ControllerException {
        try {
            if (state.getResourceMethod() == null) {
                // Obtain HTTP method, resource URL and resource manager from setup
                String httpMethod = setup.getHttpMethod();
                String resourcePath = setup.getRequestUrlResourcePart();
                ResourceManager resourceManager = setup.getResourceManager();

                // Allocate resource class
                Class<?> resourceClass = resourceManager.getResourceClass(resourcePath);

                // Resource class must be allocated
                if (resourceClass == null) {
                    throw new ControllerException("'" + httpMethod + " " + resourcePath + "' has no responsible resource class", HttpStatusCode.NotFound);
                }

                // Allocate resource method
                Method resourceMethod = resourceManager.getResourceMethod(httpMethod, resourcePath, resourceClass);

                // Resource method must be allocated
                if (resourceMethod == null) {
                    throw new ControllerException("'" + httpMethod + " " + resourcePath + "' has no responsible resource method", HttpStatusCode.NotFound);
                }
                state.setResourceClass(resourceClass);
                state.setResourceMethod(resourceMethod);
            }
        } catch (RuntimeException e) {
            throw new ControllerException("An unexpected exception occurred during resource allocation", e);
        }
    }
}
