/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */


package net.craftforge.essential.controller.managers;

import net.craftforge.essential.controller.allocation.ResourceNode;
import net.craftforge.essential.controller.allocation.ResourceTree;
import net.craftforge.essential.controller.utils.AnnotationUtils;
import net.craftforge.essential.controller.utils.RegExUtils;
import net.craftforge.essential.controller.utils.UriUtils;
import net.craftforge.reflection.utils.ClassUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * Manages the matching of resource URLs against resource classes and methods.
 *
 * @author Christian Bick
 * @since 02.02.2011
 */
public class ResourceManager {

    private static final Logger LOGGER = LoggerFactory.getLogger(ResourceManager.class);

    /**
     * Map of resource manager instances
     */
    private static ConcurrentMap<String, ResourceManager> instances = new ConcurrentHashMap<String, ResourceManager>();

    /**
     * Controls instantiation process to ensure that only one instance per package path is used.
     *
     * @param packagePath The package path
     * @return The resource manager responsible for the given package path
     */
    public static ResourceManager getInstance(String packagePath) {
        if (! instances.containsKey(packagePath)) {
            LOGGER.info("[Resource manager initialization] {}", packagePath);
            instances.putIfAbsent(packagePath, new ResourceManager(packagePath));
        }
        return instances.get(packagePath);
    }

    private ResourceTree resourceTree;

    /**
     * Initializes a resource manager from a package path.
     *
     * @param packagePath The package path
     */
    private ResourceManager(String packagePath) {
        this.resourceTree = ResourceTree.getInstance(packagePath);
    }

    /**
     * Gets the request path parameter map corresponding to the request path.
     *
     * @param requestPath The request requestPath
     * @return The request requestPath parameter map
     */
    public Map<String, String[]> getPathParameters(String requestPath) {
        String[] requestPathParts = UriUtils.splitPath(requestPath);
        Map<String, String[]> pathParameterMap = new HashMap<String, String[]>(requestPathParts.length);
        List<ResourceNode> resourceNodes = resourceTree.findResourceNodesAlongPath(requestPath);
        for (ResourceNode resourceNode : resourceNodes) {
            String pathPart = resourceNode.getPathPart();
            if (RegExUtils.FIND_ENCLOSED_BY_CURLY_BRACKETS.matcher(pathPart).matches()) {
                String requestPathPart = requestPathParts[resourceNode.getLevel()];
                String key = RegExUtils.getParamNameFromPathPart(resourceNode.getPathPart());
                String[] values = RegExUtils.getValuesFromPathPart(requestPathPart);
                pathParameterMap.put(key, values );
            }
        }
        return pathParameterMap;
    }

    /**
     * Gets the resource class corresponding to the request path.
     *
     * @param path The request path
     * @return The resource class
     */
    public Class<?> getResourceClass(String path) {
        ResourceNode resourceNode = resourceTree.findResourceNode(path);
        return resourceNode == null ? null : resourceNode.getResourceClass();
    }

    /**
     * Gets the resource method corresponding to the HTTP method
     * and request path.
     *
     * @param httpMethod The HTTP method
     * @param path The request path
     * @param resourceClass The resource class for this method
     * @return The resource method
     */
    public Method getResourceMethod(String httpMethod, String path, Class<?> resourceClass) {
        ResourceNode resourceNode = resourceTree.findResourceNode(path);
        if (resourceNode == null) {
            return null;
        }
        for (Method resourceMethod : resourceNode.getResourceMethods()) {
            if (httpMethod.equalsIgnoreCase(AnnotationUtils.getHttpMethod(resourceMethod))) {
                return ClassUtils.getFirstImplementation(resourceClass, resourceMethod);
            }
        }
        return null;
    }
    
}
