/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller.managers;

import net.craftforge.essential.controller.ControllerException;
import net.craftforge.essential.controller.constants.HttpStatusCode;
import net.craftforge.essential.controller.constants.MediaType;
import net.craftforge.essential.supply.Consumer;
import net.craftforge.reflection.utils.ReflUtils;

import java.util.Collection;
import java.util.Map;

/**
 * @author Christian Bick
 * @since 21.08.11
 */
public class ParameterHandler {

    private Map<String, String[]> parameters;

    public ParameterHandler (Map<String, String[]> parameters) {
        this.parameters = parameters;
    }

    public String[] getParameterValues(String parameterName, String[] defaultValues) throws ControllerException {
        String[] parameterValues = parameters.get(parameterName);
        if (parameterValues != null) {
            return parameterValues;
        } else if (defaultValues != null) {
            return defaultValues;
        } else {
            throw getMissingParameterException(parameterName);
        }
    }

    public String getParameterValue(String parameterName, String[] defaultValues) throws ControllerException {
        String[] parameterValues = parameters.get(parameterName);
        if (parameterValues != null && parameterValues.length > 0) {
            return parameterValues[0];
        } else if (defaultValues != null && defaultValues.length > 0) {
            return defaultValues[0];
        } else {
            throw getMissingParameterException(parameterName);
        }
    }

    public Object getParameterValueAsObject(Consumer consumer, Class<?> type, String parameterName, String[] defaultValues) throws ControllerException {
        String parameterValue = getParameterValue(parameterName, defaultValues);
        return convertToObject(consumer, type, parameterValue);
    }

    /**
     * Gets the input for a collection type using the generic type as deserialization hint
     * for the parameter values associated to the given parameter name.
     *
     * @param consumer The consumer
     * @param collectionType The collection type
     * @param typeArgument The collection's generic type argument
     * @param parameterName The parameter name
     * @param defaultValues The default values to use if no parameter values are given
     * @param <G> The generic type parameter
     * @return The collection initialized from the parameter values
     * @throws net.craftforge.essential.controller.ControllerException Failed to initialize the collection
     */
    @SuppressWarnings("unchecked")
    public <G> Collection<G> getParameterValuesAsObject(Consumer consumer, Class<?> collectionType, Class<G> typeArgument, String parameterName, String[] defaultValues) throws ControllerException {
        Collection<G> collection = ReflUtils.createCollection(collectionType);
        for (String paramValue : getParameterValues(parameterName, defaultValues)) {
            collection.add((G) convertToObject(consumer, typeArgument, paramValue));
        }
        return collection;
    }

    protected Object convertToObject(Consumer consumer, Class<?> type, String parameterValue) throws ControllerException {
        if (! ReflUtils.isVirtualPrimitive(type)) {
            throw new ControllerException("Failed to consume a parameter value because the " +
                    "target type is not a virtual primitive like int, Long or String", HttpStatusCode.BadRequest);
        }
        return consumer.consume(type, MediaType.TEXT_PLAIN, parameterValue);
    }

    protected ControllerException getMissingParameterException(String paramName) {
        return new ControllerException("The parameter " + paramName + " is " +
                    "expected to be passed in the request but is missing", HttpStatusCode.BadRequest);
    }

}
