/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller.managers;

import net.craftforge.essential.controller.ControllerException;
import net.craftforge.essential.controller.annotations.*;
import net.craftforge.essential.controller.constants.Charset;
import net.craftforge.essential.controller.constants.MediaType;
import net.craftforge.essential.controller.utils.AnnotationUtils;
import net.craftforge.essential.supply.Consumer;
import net.craftforge.reflection.managers.ClassManager;

import java.io.InputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.*;

/**
 * @author Christian Bick
 * @since 21.08.11
 */
public class InjectionManager {

    private Consumer consumer;
    private ParameterHandler parameterHandler;
    private HeaderHandler headerHandler;
    private BodyHandler bodyHandler;
    private PropertyHandler propertyHandler;

    public InjectionManager(Consumer consumer,BodyHandler bodyHandler, HeaderHandler headerHandler,
                            ParameterHandler parameterHandler, PropertyHandler propertyHandler) {
        this.consumer = consumer;
        this.parameterHandler = parameterHandler;
        this.headerHandler = headerHandler;
        this.bodyHandler = bodyHandler;
        this.propertyHandler = propertyHandler;
    }

    /**
     * Gets the input for a field annotated with @Param from the corresponding parameter.
     *
     * @param field The field to get the input for
     * @return The input for the field
     * @throws net.craftforge.essential.controller.ControllerException Failed to estimate a field's input
     */
    public Object getInputForField(Field field) throws ControllerException {
        Class<?> type = field.getType();
        // Try to obtain a body annotation
        if (AnnotationUtils.isBodyOnProperty(field)) {
            return getBodyInput(consumer, type);
        }
        String[] defaultValues = AnnotationUtils.getDefaultValuesFromProperty(field);
        // Try to obtain a header annotation
        String headerName = AnnotationUtils.getHeaderFromProperty(field);
        if (headerName != null) {
            return getHeaderInput(type, headerName, defaultValues);
        }
        // Try to obtain a parameter annotation
        String paramName = AnnotationUtils.getParamFromProperty(field);
        if (paramName != null) {
            paramName = paramName.isEmpty() ? field.getName() : paramName;
            Type genericType = field.getGenericType();
            return getParameterInput(consumer, type, genericType, paramName, defaultValues);
        }
        // Try to obtain a property annotation
        String propertyName = AnnotationUtils.getPropertyFromProperty(field);
        if (propertyName != null) {
            return getPropertyInput(type, propertyName, defaultValues);
        }
        return null;
    }

    /**
     * Gets the input for a method's parameters annotated with @Param from the corresponding parameter.
     *
     * @param method The method to get the inputs for its parameters
     * @return The list of inputs (ordered by parameter order)
     * @throws ControllerException Failed to estimate a parameter's input
     */
    public List<Object> getInputForMethodParameters(Method method) throws ControllerException {
        int paramIterator = 0;
        Class<?>[] paramTypes = method.getParameterTypes();
        Type[] paramGenericTypes = method.getGenericParameterTypes();

        List<Object> parameterInputs = new ArrayList<Object>(method.getParameterTypes().length);

        Annotation[][] usedAnnotations = ClassManager.getInstance(method).getMethodParameterAnnotations(method, Body.class);
        if (usedAnnotations == null || usedAnnotations.length < 1) {
            usedAnnotations = ClassManager.getInstance(method).getMethodParameterAnnotations(method, Header.class);
        }
        if (usedAnnotations == null || usedAnnotations.length < 1) {
            usedAnnotations = ClassManager.getInstance(method).getMethodParameterAnnotations(method, Param.class);
        }
        if (usedAnnotations == null || usedAnnotations.length < 1) {
            usedAnnotations = ClassManager.getInstance(method).getMethodParameterAnnotations(method, Property.class);
        }
        if (usedAnnotations == null || usedAnnotations.length < 1) {
            return Collections.emptyList();
        }

        for (Annotation[] annotations : usedAnnotations) {
            Class<?> type = paramTypes[paramIterator];
            Type genericType = paramGenericTypes[paramIterator];

            String[] defaultValues = null;
            for (Annotation ann : annotations) {
                if (ann.annotationType().equals(DefaultValue.class)) {
                    defaultValues = ((DefaultValue)ann).value();
                    break;
                }
            }

            for (Annotation ann : annotations) {
                if (ann.annotationType().equals(Body.class)) {
                    Object input = getBodyInput(consumer, type);
                    parameterInputs.add(input);
                    break;
                } else if (ann.annotationType().equals(Header.class)) {
                    String headerName = ((Header)ann).value();
                    Object input = getHeaderInput(type, headerName, defaultValues);
                    parameterInputs.add(input);
                    break;
                } else if (ann.annotationType().equals(Param.class)) {
                    String paramName = ((Param)ann).value();
                    Object input = getParameterInput(consumer, type, genericType, paramName, defaultValues);
                    parameterInputs.add(input);
                    break;
                } else if (ann.annotationType().equals(Property.class)) {
                    String propertyName = ((Property)ann).value();
                    Object input = getPropertyInput(type, propertyName, defaultValues);
                    parameterInputs.add(input);
                    break;
                }
            }
            paramIterator++;
        }
        return parameterInputs;
    }

    public Object getBodyInput(Consumer consumer, Class<?> type) throws ControllerException {
        String mediaType = headerHandler.getContentMediaType(MediaType.TEXT_XML);
        String charset = headerHandler.getContentCharset(Charset.UTF8);
        if (type.isAssignableFrom(InputStream.class)) {
            return bodyHandler.getBodyInputStream();
        } else if (type.isAssignableFrom(String.class)) {
            return bodyHandler.getBodyContent(charset);
        } else {
            return bodyHandler.getBodyObject(consumer, type, mediaType, charset);
        }
    }

    public Object getHeaderInput(Class<?> type, String headerName, String[] defaultValues) throws ControllerException {
        if (type.isAssignableFrom(String[].class)) {
            return headerHandler.getHeaderValues(headerName, defaultValues);
        }
        return headerHandler.getHeaderValue(headerName, defaultValues);
    }

    public Object getParameterInput(Consumer consumer, Class<?> type, Type genericType, String parameterName, String[] defaultValues) throws  ControllerException {
        if (type.isAssignableFrom(String[].class)) {
            return parameterHandler.getParameterValues(parameterName, defaultValues);
        } else if (type.isAssignableFrom(String.class)) {
            return parameterHandler.getParameterValue(parameterName, defaultValues);
        } else if (Collection.class.isAssignableFrom(type) && genericType instanceof ParameterizedType) {
            Class<?> typeArgument = (Class<?>)((ParameterizedType)genericType).getActualTypeArguments()[0];
            return parameterHandler.getParameterValuesAsObject(consumer, type, typeArgument, parameterName, defaultValues);
        } else {
            return parameterHandler.getParameterValueAsObject(consumer, type, parameterName, defaultValues);
        }
    }

    public Object getPropertyInput(Class<?> type, String propertyName, String[] defaultValues) throws ControllerException {
        if (Map.class.isAssignableFrom(type)) {
            return propertyHandler.getProperties(propertyName, defaultValues);
        } else if (type.isAssignableFrom(String.class)) {
            return propertyHandler.getProperty(propertyName, defaultValues);
        }
        throw new ControllerException("The configuration property " + propertyName + " cannot " +
                "be initialized because its type is neither applicable to String nor to Map");
    }

}
