/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller.managers;

import net.craftforge.essential.controller.ControllerException;
import net.craftforge.essential.controller.constants.HttpHeader;
import net.craftforge.essential.controller.constants.HttpStatusCode;
import net.craftforge.essential.controller.utils.HeaderUtils;

import java.util.Map;

/**
 * @author Christian Bick
 * @since 21.08.11
 */
public class HeaderHandler {

    private Map<String, String[]> headers;

    private String authorizationUser;
    private String authorizationPassword;

    private String contentMediaType;
    private String contentCharset;

    public HeaderHandler (Map<String, String[]> headers) {
        this.headers = headers;
        initAuthorization();
        initContentType();
    }

    protected void initAuthorization() {
        String[] authorizationHeaders = headers.get(HttpHeader.AUTHORIZATION);
        if (authorizationHeaders == null || authorizationHeaders.length < 1) {
            return;
        }
        String[] authPair = HeaderUtils.getAuthorizationPair(authorizationHeaders);
        authorizationUser = authPair[0];
        authorizationPassword = authPair[1];
    }

    protected void initContentType() {
        String[] contentTypeHeaders = headers.get(HttpHeader.CONTENT_TYPE);
        if (contentTypeHeaders == null || contentTypeHeaders.length < 1) {
            return;
        }
        String[] contentTypePair = HeaderUtils.getContentTypePair(contentTypeHeaders[0]);
        contentMediaType = contentTypePair[0];
        contentCharset = contentTypePair[1];
    }

    public String[] getHeaderValues(String headerName, String[] defaultValues) throws ControllerException {
        String[] headerValues = headers.get(headerName);
        if (headerValues != null) {
            return headerValues;
        } else if (defaultValues != null) {
            return defaultValues;
        } else {
            throw getMissingHeaderException(headerName);
        }
    }

    public String getHeaderValue(String headerName, String defaultValue) throws ControllerException {
        if (headerName.equals(HttpHeader.AUTHORIZATION + "#user")) {
            return getAuthorizationUser();
        } else if (headerName.equalsIgnoreCase(HttpHeader.AUTHORIZATION + "#pass")) {
            return getAuthorizationPassword();
        } else if (headerName.equalsIgnoreCase(HttpHeader.CONTENT_TYPE + "#media")) {
            return getContentMediaType(defaultValue);
        } else if (headerName.equalsIgnoreCase(HttpHeader.CONTENT_TYPE + "#charset")) {
            return getContentCharset(defaultValue);
        }
        String headerValues[] = headers.get(headerName);
        if (headerValues != null && headerValues.length > 0) {
            return headerValues[0];
        } else if (defaultValue != null) {
            return defaultValue;
        } else {
            throw getMissingHeaderException(headerName);
        }
    }

    public String getHeaderValue(String headerName, String[] defaultValues) throws ControllerException {
        String headerValues[] = headers.get(headerName);
        if (headerValues != null && headerValues.length > 0) {
            return headerValues[0];
        } else if (defaultValues != null && defaultValues.length > 0) {
            return defaultValues[0];
        } else {
            throw getMissingHeaderException(headerName);
        }
    }

    public String getAuthorizationUser() {
        return authorizationUser;
    }

    public String getAuthorizationPassword() {
        return authorizationPassword;
    }

    public String getContentMediaType(String defaultValue) {
        return contentMediaType == null ? defaultValue : contentMediaType;
    }

    public String getContentCharset(String defaultValue) {
        return contentCharset == null ? defaultValue : contentCharset;
    }

    protected ControllerException getMissingHeaderException(String headerName) {
        return new ControllerException("The header " + headerName + " is " +
                    "expected to be part of the request but is missing", HttpStatusCode.BadRequest);
    }

}
