/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller.managers;

import net.craftforge.essential.controller.Configuration;
import net.craftforge.essential.controller.allocation.ResourceNode;
import net.craftforge.essential.controller.allocation.ResourceTree;
import net.craftforge.essential.controller.annotations.DefaultValue;
import net.craftforge.essential.controller.annotations.Param;
import net.craftforge.essential.controller.constants.HttpMethod;
import net.craftforge.essential.controller.constants.HttpStatusCode;
import net.craftforge.essential.controller.documentation.jaxb.*;
import net.craftforge.essential.controller.utils.AnnotationUtils;
import net.craftforge.essential.controller.utils.RegExUtils;
import net.craftforge.reflection.managers.ClassManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * Manages documentation of resources.
 *
 * @author Christian Bick
 * @since 25.07.11
 */
public class DocumentationManager {

    private static final Logger LOGGER = LoggerFactory.getLogger(DocumentationManager.class);

    /**
     * Map of documentation manager instances
     */
    private static ConcurrentMap<String, DocumentationManager> instances = new ConcurrentHashMap<String, DocumentationManager>();

    /**
     * Controls instantiation process to ensure that only one instance per package path is used.
     *
     * @param packageName The package name
     * @return The documentation manager responsible for the given package path
     */
    public static DocumentationManager getInstance(String packageName) {
        if (! instances.containsKey(packageName)) {
            LOGGER.info("[Documentation manager initialization] {}", packageName);
            instances.putIfAbsent(packageName, new DocumentationManager(packageName));
        }
        return instances.get(packageName);
    }

    private ResourceTree resourceTree;

    /**
     * Initializes a documentation manager from a package path.
     *
     * @param packageName The package name
     */
    private DocumentationManager(String packageName) {
        this.resourceTree = ResourceTree.getInstance(packageName);
    }

    /**
     * Gets the documentation in WADL for a resource identified via the given path in
     * context of the specified configuration.
     *
     * @param path The path
     * @param config The configuration
     * @return The documentation in WADL
     */
    public WadlApplication getDocumentation(String path, Configuration config) {
        WadlApplication application = new WadlApplication();
        List<WadlResource> resources = new ArrayList<WadlResource>(1);
        resources.add(getResource(path, config));
        application.setResources(resources);
        return application;
    }

    /**
     * <p>Gets a resource from a resource path and
     * a controller configuration.</p>
     * <p>Includes:</p>
     * <ul>
     *     <li>resource parameters</li>
     *     <li>resource methods</li>
     *     <li>sub resources</li>
     * </ul>
     *
     * @param path The resource path
     * @param configuration The controller configuration
     * @return The resource
     */
    protected WadlResource getResource(String path, Configuration configuration) {
        ResourceNode resourceNode = resourceTree.findResourceNode(path);
        if (resourceNode == null) {
            return null;
        }

        Class<?> resourceClass = resourceNode.getResourceClass();
        List<Method> resourceMethods = resourceNode.getResourceMethods();

        WadlResource resource = new WadlResource();
        resource.setPath(resourceNode.getPath());
        if (resourceClass != null && resourceMethods != null && resourceMethods.size() > 0) {
            resource.setParams(getResourceParams(resourceClass, resourceMethods.get(0)));
            resource.setMethods(getMethods(resourceMethods, configuration));
            resource.setResources(getSubResources(path));
        }
        return resource;
    }

    /**
     * <p>Gets the list of all sub-resources of a resource matching a resource path.</p>
     *
     * @param path The resource path
     * @return The list of all sub-resources
     */
    protected List<WadlResource> getSubResources(String path) {
        List<WadlResource> resources = new LinkedList<WadlResource>();
        for (ResourceNode resourceNode : resourceTree.findResourceNodesInPathSkippingRoot(path)) {
            WadlResource resource = new WadlResource();
            resource.setPath(resourceNode.getPath());
            resources.add(resource);
        }
        return resources;
    }

    /**
     * <p>Gets the list of all resource HTTP methods from a list of resource Java methods
     * and a controller configuration.</p>
     *
     * @param resourceMethods The resource Java methods
     * @param configuration The controller configuration
     * @return The list of all resource HTTP methods
     */
    protected List<WadlMethod> getMethods(List<Method> resourceMethods, Configuration configuration) {
        List<WadlMethod> methods = new LinkedList<WadlMethod>();
        for (Method resourceMethod : resourceMethods) {
            WadlMethod method = new WadlMethod();
            method.setName(AnnotationUtils.getHttpMethod(resourceMethod));
            method.setRequest(getRequest(resourceMethod, configuration));
            method.setResponses(getResponses(resourceMethod, configuration));
            methods.add(method);
        }
        return methods;
    }

    /**
     * Gets a list of possible responses of a resource Java method and a controller configuration.
     *
     * @param resourceMethod The resource Java method
     * @param configuration The controller configuration
     * @return The list of possible responses
     */
    protected List<WadlResponse> getResponses(Method resourceMethod, Configuration configuration) {
        List<WadlResponse> responses = new ArrayList<WadlResponse>(1);
        WadlResponse response = new WadlResponse();
        if (! resourceMethod.getReturnType().equals(void.class)) {
            response.setStatus(HttpStatusCode.OK.getCode());
            response.setRepresentations(getResponseRepresentations(resourceMethod, configuration));
        } else {
            response.setStatus(HttpStatusCode.NoContent.getCode());
            response.setRepresentations(Collections.<WadlRepresentation>emptyList());
        }
        responses.add(response);
        return responses;
    }

    /**
     * Gets the request for a resource Java method and a controller configuration.
     *
     * @param resourceMethod The resource Java method
     * @param configuration The controller configuration
     * @return The request
     */
    protected WadlRequest getRequest(Method resourceMethod, Configuration configuration) {
        WadlRequest request = new WadlRequest();
        request.setParams(getRequestParams(resourceMethod));
        request.setRepresentations(getRequestRepresentations(resourceMethod, configuration));
        return request;
    }

    protected List<WadlParam> filterNoneTemplateParams(List<WadlParam> params) {
        List<WadlParam> noneTemplateParams = new LinkedList<WadlParam>();
        for (WadlParam param : params) {
            if (! param.getStyle().equals(WadlParamStyle.TEMPLATE)) {
                noneTemplateParams.add(param);
            }
        }
        return noneTemplateParams;
    }

    /**
     * Filters all template parameters from a list of parameters.
     *
     * @param params The list of parameters
     * @return The list of template parameters
     */
    protected List<WadlParam> filterTemplateParams(List<WadlParam> params) {
        List<WadlParam> templateParams = new LinkedList<WadlParam>();
        for (WadlParam param : params) {
            if (param.getStyle().equals(WadlParamStyle.TEMPLATE)) {
                templateParams.add(param);
            }
        }
        return templateParams;
    }

    /**
     * Gets the list of all resource parameters for a resource Java class and method.
     *
     * @param resourceClass The resource class The resource Java class
     * @param resourceMethod The resource method The resource Java method
     * @return The list of all resource parameters
     */
    protected List<WadlParam> getResourceParams(Class<?> resourceClass, Method resourceMethod) {
        List<WadlParam> classParams = getClassParams(resourceClass);
        List<WadlParam> methodParams = getMethodParams(resourceMethod);

        ArrayList<WadlParam> allParams = new ArrayList<WadlParam>(classParams.size() + methodParams.size());
        allParams.addAll(classParams);
        allParams.addAll(methodParams);

        // Resource params are those annotated at class level and all template params
        // A set is used to to remove all duplicates
        Set<WadlParam> resourceParams = new HashSet<WadlParam>();
        resourceParams.addAll(filterTemplateParams(allParams));
        resourceParams.addAll(filterNoneTemplateParams(classParams));
        // As a list is needed, the set is converted into a list before returned
        return new ArrayList<WadlParam>(resourceParams);
    }

    /**
     * Gets the list of request parameters for a resource Java method.
     *
     * @param resourceMethod resource Java method
     * @return The list of all request parameters
     */
    protected List<WadlParam> getRequestParams(Method resourceMethod) {
        // Request params are those annotated at method level which are not template params
        List<WadlParam> methodParams = getMethodParams(resourceMethod);
        return filterNoneTemplateParams(methodParams);
    }

    /**
     * Gets the list of resource and request parameters from a resource Java method.
     *
     * @param resourceMethod The resource Java method
     * @return The list of request parameters
     */
    protected List<WadlParam> getMethodParams(Method resourceMethod) {
        List<WadlParam> params = new LinkedList<WadlParam>();
        // int paramIterator = 0;
        // Class<?>[] paramTypes = resourceMethod.getParameterTypes();
        for (Annotation[] annotations : ClassManager.getInstance(resourceMethod).getMethodParameterAnnotations(resourceMethod, Param.class)) {
            String defaultValue = null;
            for (Annotation ann : annotations) {
                if (ann.annotationType().equals(DefaultValue.class)) {
                    defaultValue = defaultValuesAsString(((DefaultValue) ann).value());
                    break;
                }
            }
            for (Annotation ann : annotations) {
                if (ann.annotationType().equals(Param.class)) {
                    String paramName = ((Param)ann).value();
                    WadlParam param = getParam(paramName, defaultValue);
                    params.add(param);
                }
            }
            // paramIterator++;
        }
        return params;
    }

    /**
     * Gets the list of resource parameters from a resource Java class
     *
     * @param resourceClass The resource Java class
     * @return The list of request parameters
     */
    protected List<WadlParam> getClassParams(Class<?> resourceClass) {
        List<WadlParam> params = new LinkedList<WadlParam>();
        for (Field field : AnnotationUtils.getParamFieldsFromClass(resourceClass)) {
            String paramName = AnnotationUtils.getParamFromProperty(field);
            String defaultValue = defaultValuesAsString(AnnotationUtils.getDefaultValuesFromProperty(field));
            if (paramName.isEmpty()) {
                paramName = field.getName();
            }
            WadlParam param = getParam(paramName, defaultValue);
            params.add(param);
        }
        return params;
    }

    /**
     * Gets a request parameter from a parameter name and a parameter default value
     *
     * @param paramName The parameter name
     * @param defaultValue The parameter default value
     * @return The request parameter
     */
    protected WadlParam getParam(String paramName, String defaultValue) {
        WadlParam param = new WadlParam();
        if (RegExUtils.FIND_ENCLOSED_BY_CURLY_BRACKETS.matcher(paramName).matches()) {
            param.setName(RegExUtils.getParamNameFromPathPart(paramName, false));
            param.setStyle(WadlParamStyle.TEMPLATE);
        } else {
            param.setName(paramName);
            param.setStyle(WadlParamStyle.QUERY);
            if (defaultValue == null) {
                param.setRequired(true);
            } else {
                param.setRequired(false);
                param.setDefaultValue(defaultValue);
            }
        }
        return param;
    }

    /**
     * Gets a request representation from a Java method and a controller configuration.
     *
     * @param method The Java method
     * @param configuration The controller configuration
     * @return The request representation
     */
    protected List<WadlRepresentation> getRequestRepresentations(Method method, Configuration configuration) {
        String httpMethod = AnnotationUtils.getHttpMethod(method);
        if (! (httpMethod.equals(HttpMethod.POST) || httpMethod.equals(HttpMethod.PUT))) {
            return Collections.emptyList();
        }
        Class<?> consumerClass = AnnotationUtils.getConsumerFromMethodOrClass(method);
        if (consumerClass == null) {
            consumerClass = configuration.getDefaultConsumer();
        }
        List<String> supportedMediaTypes = AnnotationUtils.getSupportedConsumerMediaTypes(consumerClass);
        return getRepresentations(supportedMediaTypes);
    }

    /**
     * Gets a response representation from a Java method and a controller configuration.
     *
     * @param method The Java method
     * @param configuration The controller configuration
     * @return The response representation
     */
    protected List<WadlRepresentation> getResponseRepresentations(Method method, Configuration configuration) {
        Class<?> producerClass = AnnotationUtils.getProducerFromMethodOrClass(method);
        if (producerClass == null) {
            producerClass = configuration.getDefaultProducer();
        }
        List<String> supportedMediaTypes = AnnotationUtils.getSupportedProducerMediaTypes(producerClass);
        return getRepresentations(supportedMediaTypes);
    }

    /**
     * Gets a list of representations from a list of supported media types.
     *
     * @param supportedMediaTypes The supported media types
     * @return The list of representations
     */
    protected List<WadlRepresentation> getRepresentations(List<String> supportedMediaTypes) {
        Collections.sort(supportedMediaTypes);
        List<WadlRepresentation> representations = new ArrayList<WadlRepresentation>(supportedMediaTypes.size());
        for (String media : supportedMediaTypes) {
            WadlRepresentation representation = new WadlRepresentation();
            representation.setMediaType(media);
            representations.add(representation);
        }
        return representations;
    }

    /**
     * Converts a list of default values into a human readable string.
     *
     * @param defaultValues The list of default values
     * @return The human readable string
     */
    protected String defaultValuesAsString(String[] defaultValues) {
        String defaultValue = "";
        if (defaultValues == null) {
            return null;
        }
        if (defaultValues.length == 1) {
            return defaultValues[0];
        }
        defaultValue += "{";
        for (String value : defaultValues) {
            defaultValue += value + ", ";
        }
        defaultValue = defaultValue.substring(0, defaultValue.length()-2);
        defaultValue += "}";
        return defaultValue;
    }
}
