/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller.constants;

import java.util.HashMap;
import java.util.Map;

/**
 * HTTP status codes with name.
 *
 * @author Christian Bick
 * @since 20.06.11
 */
public enum HttpStatusCode {

    // 1xx
    Continue(100, "Continue"),
    SwitchingProtocols(101, "Switching Protocols"),
    Processing(102, "Processing"),

    // 2xx
    OK(200, "OK"),
    Created(201, "Created"),
    Accepted(202, "Accepted"),
    NonAuthoritativeInformation(203, "Non-Authoritative Information"),
    NoContent(204, "No Content"),
    ResetContent(205, "Reset Content"),
    PartialContent(206, "Partial Content"),
    MultiStatus(207, "Multi-Status"),

    // 3xx
    MultipleChoice(300, "Multiple Choice"),
    MovedPermanently(301, "Moved Permanently"),
    Found(302, "Found"),
    SeeOther(303, "See Other"),
    NotModified(304, "Not Modified"),
    UseProxy (305, "Use Proxy"),
    TemporaryRedirect(307, "Temporary Redirect"),

    // 4xx
    BadRequest(400, "Bad Request"),
    Unauthorized(401, "Unauthorized"),
    Forbidden(403, "Forbidden"),
    NotFound(404, "NotFound"),
    MethodNotAllowed(405, "Method Not Allowed"),
    NotAcceptable(406, "Not Acceptable"),
    ProxyAuthenticationRequired(407, "Proxy Authentication Required"),
    RequestTimeOut(408, "Request Time-out"),
    Conflict(409, "Conflict"),
    Gone(410, "Gone"),
    LengthRequired(411, "411 Length Required"),
    PreconditionFailed(412, "Precondition Failed"),
    RequestEntityTooLarge(413, "RequestEntityTooLarge"),
    RequestURITooLong(414, "Request-URI Too Long"),
    UnsupportedMediaType(415, "Unsupported Media Type"),
    RequestedRangeNotSatisfiable(416, "Requested range not satisfiable"),
    ExpectationFailed(417, "Expectation Failed"),
    ThereAreToManyConnectionsFromYourInternetAddress(421, "There are too many connections from your internet address"),
    UnprocessableEntity(422, "Unprocessable Entity"),
    Locked(423, "Locked"),
    FailedDependency(424, "Failed Dependency"),
    UnorderedCollection(425, "Unordered Collection"),
    UpgradeRequired(426, "Upgrade Required"),

    // 5xx
    InternalServerError(500, "Internal Server Error"),
    NotImplemented(501, "Not Implemented"),
    BadGateway(502, "Bad Gateway"),
    ServiceUnavailable(503, "Service Unavailable"),
    GatewayTimeOut(504, "Gateway Time-out"),
    HTTPVersionNotSupported(505, "HTTP Version not supported"),
    VariantAlsoNegotiates(506, "Variant Also Negotiates"),
    InsufficientStorage(507, "Insufficient Storage"),
    BandwidthLimitExceeded(509, "Bandwidth Limit Exceeded"),
    NotExtended(510, "Not Extended");

    private static Map<Integer, HttpStatusCode> lookUpMap = new HashMap<Integer, HttpStatusCode>();

    static {
        lookUpMap.put(100, Continue);
        lookUpMap.put(101, SwitchingProtocols);
        lookUpMap.put(102, Processing);

        lookUpMap.put(200, OK);
        lookUpMap.put(201, Created);
        lookUpMap.put(202, Accepted);
        lookUpMap.put(203, NonAuthoritativeInformation);
        lookUpMap.put(204, NoContent);
        lookUpMap.put(205, ResetContent);
        lookUpMap.put(206, PartialContent);
        lookUpMap.put(207, MultiStatus);

        lookUpMap.put(300, MultipleChoice);
        lookUpMap.put(301, MovedPermanently);
        lookUpMap.put(302, Found);
        lookUpMap.put(303, SeeOther);
        lookUpMap.put(304, NotModified);
        lookUpMap.put(305, UseProxy);
        lookUpMap.put(307, TemporaryRedirect);

        lookUpMap.put(400, BadRequest);
        lookUpMap.put(401, Unauthorized);
        lookUpMap.put(403, Forbidden);
        lookUpMap.put(404, NotFound);
        lookUpMap.put(405, MethodNotAllowed);
        lookUpMap.put(406, NotAcceptable);
        lookUpMap.put(407, ProxyAuthenticationRequired);
        lookUpMap.put(408, RequestTimeOut);
        lookUpMap.put(409, Conflict);
        lookUpMap.put(410, Gone);
        lookUpMap.put(411, LengthRequired);
        lookUpMap.put(412, PreconditionFailed);
        lookUpMap.put(413, RequestEntityTooLarge);
        lookUpMap.put(414, RequestURITooLong);
        lookUpMap.put(415, UnsupportedMediaType);
        lookUpMap.put(416, RequestedRangeNotSatisfiable);
        lookUpMap.put(417, ExpectationFailed);
        lookUpMap.put(421, ThereAreToManyConnectionsFromYourInternetAddress);
        lookUpMap.put(422, UnprocessableEntity);
        lookUpMap.put(423, Locked);
        lookUpMap.put(424, FailedDependency);
        lookUpMap.put(425, UnorderedCollection);
        lookUpMap.put(426, UpgradeRequired);

        lookUpMap.put(500, InternalServerError);
        lookUpMap.put(501, NotImplemented);
        lookUpMap.put(502, BadGateway);
        lookUpMap.put(503, ServiceUnavailable);
        lookUpMap.put(504, GatewayTimeOut);
        lookUpMap.put(505, HTTPVersionNotSupported);
        lookUpMap.put(506, VariantAlsoNegotiates);
        lookUpMap.put(507, InsufficientStorage);
        lookUpMap.put(509, BandwidthLimitExceeded);
        lookUpMap.put(510, NotExtended);
    }

    private int code;
    private String name;

    private HttpStatusCode(int code, String name) {
        this.code = code;
        this.name = name;
    }

    /**
     * Gets the status code.
     *
     * @return The status code
     */
    public int getCode() {
        return code;
    }

    /**
     * Gets the status name.
     *
     * @return The status name
     */
    public String getName() {
        return name;
    }

    /**
     * Gets the HTTP status code enum for the given status code.
     *
     * @param code The status code
     * @return The status code enum
     */
    public HttpStatusCode getHttpStatusCode(int code) {
        return lookUpMap.get(code);
    }


}
