/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller.allocation;

import net.craftforge.essential.controller.utils.RegExUtils;
import net.craftforge.essential.controller.utils.UriUtils;

import java.lang.reflect.Method;
import java.util.*;
import java.util.regex.Pattern;

/**
 * <p>A resource node is the framework representation of a resource.</p>
 * <p>For a resource '/my/simple/path'</p> the following resource nodes are created:
 * <ul>
 *     <li><b>/</b>: The resource root node (common to all paths)</li>
 *     <li><b>/my</b>: The resource part node 1st level</li>
 *     <li><b>/my/simple</b>: The resource node on 2nd level</li>
 *     <li><b>/my/simple/path</b>: The resource node on 3rd level</li>
 * </ul>
 * <p>
 *     For every resource, a resource node holds information on
 * </p>
 * <ul>
 *     <li>The complete path from the root (e.g. '/my/simple')</li>
 *     <li>The path part (e.g. 'simple')</li>
 *     <li>The level of the path part (e.g. 2) </li>
 *     <li>A pattern matching this path part (just for template parts)</li>
 *     <li>All sub nodes of the current node</li>
 *     <li>The resource class responsible for the path part</li>
 *     <li>The resource methods implementing the HTTP methods for the path part</li>
 * </ul>
 *
 * @author Christian Bick
 * @since 30.07.11
 */
public class ResourceNode {

    private String path;
    private String pathPart;
    private Pattern pattern = null;
    private int level;
    private Map<String, ResourceNode> trivialPattenSubNodes = new HashMap<String, ResourceNode>();
    private List<ResourceNode> noneTrivialPatternSubNodes = new LinkedList<ResourceNode>();
    private Class<?> resourceClass = null;
    private List<Method> resourceMethods = new ArrayList<Method>(4);

    /**
     * Initializes the resource node. Resource nodes are usually created
     * from a path which is split into its parts. To be aware in which
     * level of the path creation process the constructor has been called
     * it is specified, too.
     *
     * @param pathParts The path parts of the original path
     * @param level The current level
     */
    public ResourceNode(String[] pathParts, int level) {
        this.level = level;
        this.path = getPathFromPathParts(pathParts);
        this.pathPart = pathParts[level];
        String regEx = RegExUtils.getRegExFromPathPart(pathPart);
        this.pattern = Pattern.compile(regEx);
    }

    /**
     * Gets the level of the node in the node tree with 0 being
     * the root level.
     *
     * @return The level in the node tree
     */
    public int getLevel() {
        return level;
    }

    /**
     * Gets the complete (virtual) path from root to node.
     *
     * @return The (virtual) path from root to node
     */
    public String getPath() {
        return path;
    }

    /**
     * Gets the path part.
     *
     * @return The path part
     */
    public String getPathPart() {
        return pathPart;
    }

    /**
     * Gets the pattern for matching against variable path parts.
     *
     * @return The pattern
     */
    public Pattern getPattern() {
        return pattern;
    }

    /**
     * Gets the resource class being responsible for this node.
     *
     * @return The responsible resource class
     */
    public Class<?> getResourceClass() {
        return resourceClass;
    }

    /**
     * Gets the resource methods associated to this node.
     *
     * @return The responsible resource methods
     */
    public List<Method> getResourceMethods() {
        return resourceMethods;
    }

    /**
     * Adds a resource method to this node or one of its sub nodes. If no responsible
     * sub node exists then a new sub node will be created along the path parts.
     * If the this operation's maximum level (number of sub parts minus one) is reached
     * then the given resource method is added to this node's resource methods.
     *
     * @param pathParts The path parts
     * @param level The current level
     * @param resourceClass The resource class
     * @param resourceMethod The resource method
     */
    public void addResourceMethod(String[] pathParts, int level, Class<?> resourceClass, Method resourceMethod) {
        if (level == pathParts.length-1) {
            this.resourceClass = resourceClass;
            resourceMethods.add(resourceMethod);
        } else {
            int nextLevel = level+1;
            ResourceNode nextNode = findSubNodeFromVirtualPathPart(pathParts[nextLevel]);
            if (nextNode == null) {
                nextNode = new ResourceNode(pathParts, nextLevel);
                if (nextNode.getPattern().pattern().equals(nextNode.getPathPart())) {
                    // To be more efficient, trivial patterns are putted in a Hash Map
                    trivialPattenSubNodes.put(nextNode.getPathPart(), nextNode);
                } else {
                    // Those which are none-trivial patterns are added to list
                    noneTrivialPatternSubNodes.add(nextNode);
                }
            }
            nextNode.addResourceMethod(pathParts, nextLevel, resourceClass, resourceMethod);
        }
    }

    /**
     * Finds a sub node having the specified virtual path part.
     *
     * @param pathPart The virtual path part.
     * @return The resource node or null of not found
     */
    private ResourceNode findSubNodeFromVirtualPathPart(String pathPart) {
        // Look up the node in the Hash Map first
        ResourceNode simpleNode = trivialPattenSubNodes.get(pathPart);
        if (simpleNode != null) {
            return simpleNode;
        }
        // If the path part is not in the Hash Map, then it might be the same reg ex
        // as a none trivial path part
        for(ResourceNode patternNode : noneTrivialPatternSubNodes) {
            if (patternNode.getPathPart().equals(pathPart)) {
                return patternNode;
            }
        }
        return null;
    }

    /**
     * Finds the sub node matching the last node of a path represented
     * by the path parts.
     *
     * @param pathParts The path parts
     * @return The resource node
     */
    public ResourceNode findResourceNode(String[] pathParts) {
        if (level == pathParts.length-1) {
            return this;
        } else {
            int nextLevel = level+1;
            ResourceNode nextNode = findSubNode(pathParts[nextLevel]);
            if (nextNode == null) {
                return null;
            }
            return nextNode.findResourceNode(pathParts);
        }
    }

    /**
     * Finds all sub nodes along a path represented by the path parts.
     *
     * @param pathParts The path parts
     * @return List of all resource nodes along the path
     */
    public List<ResourceNode> findResourceNodesAlongPath(String[] pathParts) {
        List<ResourceNode> resourceNodes;
        if (level == pathParts.length-1) {
            resourceNodes = new ArrayList<ResourceNode>(level);
        } else {
            int nextLevel = level+1;
            ResourceNode nextNode = findSubNode(pathParts[nextLevel]);
            if (nextNode == null) {
                resourceNodes = new ArrayList<ResourceNode>(level);
            } else {
                resourceNodes = nextNode.findResourceNodesAlongPath(pathParts);
            }
        }
        resourceNodes.add(this);
        return resourceNodes;
    }

    /**
     * Finds all direct and indirect sub nodes including this node
     *
     * @return List of all sub nodes
     */
    public List<ResourceNode> findAllSubNodes() {
        return findAllSubNodes(false);
    }

    /**
     * Finds all direct and indirect sub nodes including this node
     *
     * @return List of all sub nodes
     */
    public List<ResourceNode> findAllSubNodesSkippingRoot() {
        return findAllSubNodes(true);
    }

    /**
     * Finds all direct and indirect sub nodes with the option to skip
     * the base node.
     *
     * @param skip Whether to skip the base node or not
     * @return List of all sub nodes
     */
    private List<ResourceNode> findAllSubNodes(boolean skip) {
        List<ResourceNode> resourceNodes = new LinkedList<ResourceNode>();
        if (! skip) {
            resourceNodes.add(this);
        }
        for (String key : trivialPattenSubNodes.keySet()) {
            ResourceNode subNode = trivialPattenSubNodes.get(key);
            resourceNodes.addAll(subNode.findAllSubNodes(false));
        }
        for (ResourceNode subNode : noneTrivialPatternSubNodes) {
            resourceNodes.addAll(subNode.findAllSubNodes(false));
        }
        return resourceNodes;
    }

    /**
     * Finds a direct sub node matching the given (real) path part.
     *
     * @param pathPart The (real) path part
     * @return The matching resource node or null if none found
     */
    private ResourceNode findSubNode(String pathPart) {
        // Look up the node in the Hash Map first
        ResourceNode simpleNode = trivialPattenSubNodes.get(pathPart);
        if (simpleNode != null) {
            return simpleNode;
        }
        // If the path part is not in the Hash Map, then it might match a pattern
        for(ResourceNode patternNode : noneTrivialPatternSubNodes) {
            if (patternNode.getPattern().matcher(pathPart).matches()) {
                return patternNode;
            }
        }
        return null;
    }

    /**
     * Gets the complete path from a path part concatenating the parts
     * with a '/' separator.
     *
     * @param pathParts The path parts.
     * @return The path
     */
    private String getPathFromPathParts(String[] pathParts) {
        String path = "";
        for (int i=0; i <= level; i++) {
            path += "/" + pathParts[i];
        }
        return UriUtils.standardUri(path);
    }
}
