/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller;

import net.craftforge.essential.controller.constants.Charset;
import net.craftforge.essential.controller.constants.HttpHeader;
import net.craftforge.essential.controller.constants.HttpMethod;
import net.craftforge.essential.controller.constants.MediaType;
import net.craftforge.essential.controller.managers.*;
import net.craftforge.essential.controller.utils.UriUtils;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;

/**
 * Data structure containing all external (request and configuration)
 * information for the phases.
 *
 * @author Christian Bick
 * @since 06.02.2011
 */
public class Setup {

    /**
     * The resource manager
     */
    private ResourceManager resourceManager;

    /**
     * The documentation manager
     */
    private DocumentationManager documentationManager;

    /**
     * The controller configuration
     */
    private Configuration configuration;

    /**
     * The parameter handler
     */
    private ParameterHandler parameterHandler;

    /**
     * The header handler
     */
    private HeaderHandler headerHandler;

    /**
     * The body handler
     */
    private BodyHandler bodyHandler;

    /**
     * The property handler
     */
    private PropertyHandler propertyHandler;

    /**
     * The normalized HTTP request
     */
    private Request request;

    /**
     * The normalized HTTP response
     */
    private Response response;

    /**
     * Constructs the setup.
     *
     * @param resourceManager The resource manager
     * @param documentationManager The documentation manager
     * @param configuration The controller configuration
     * @param request The normalized HTTP request
     * @param response The normalized HTTP response
     */
    public Setup(ResourceManager resourceManager,
                   DocumentationManager documentationManager,
                   Configuration configuration,
                   Request request,
                   Response response) {
        
        this.resourceManager = resourceManager;
        this.documentationManager = documentationManager;
        this.configuration = configuration;
        this.request = request;
        this.response = response;

        this.bodyHandler = new BodyHandler(request.getBodyInputStream());
        this.parameterHandler = new ParameterHandler(getAllParameters());
        this.headerHandler = new HeaderHandler(request.getHeaders());
        this.propertyHandler = new PropertyHandler(configuration);
    }

    /**
     * Gets the parameter handler. The parameter handler provides a convenient
     * access to the HTTP parameters submitted.
     *
     * @return The parameter handler
     */
    public ParameterHandler getParameterHandler() {
        return parameterHandler;
    }

    /**
     * Gets the header handler. The header handler provides a convenient access
     * to the HTTP headers submitted.
     *
     * @return The header handler
     */
    public HeaderHandler getHeaderHandler() {
        return headerHandler;
    }

    /**
     * Gets the body handler. The body handler provides a convenient access
     * to the HTTP body submitted.
     *
     * @return The body handler
     */
    public BodyHandler getBodyHandler() {
        return bodyHandler;
    }

    /**
     * Gets the property handler. The property handler provides a convenient access
     * to the controller configuration properties.
     *
     * @return The property handler
     */
    public PropertyHandler getPropertyHandler() {
        return propertyHandler;
    }

    /**
     * Gets the media type for result serialization. Will use the accepted media type of the request if
     * not overridden in the URL meta part of the URL info part. (e.g. my/path.json)
     *
     * @return The media type
     * @throws ControllerException if the header was not submitted not and no default value is set
     */
    public String getAcceptedMediaType() throws ControllerException {
        String metaDataPart = getRequestUrlMetaPart();
        if (metaDataPart.contains(".xml")) {
            return MediaType.TEXT_XML;
        } else if (metaDataPart.contains(".json")) {
            return MediaType.TEXT_JSON;
        } else {
            return headerHandler.getHeaderValue(HttpHeader.ACCEPT, MediaType.TEXT_XML);
        }
    }

    /**
     * Gets the encoding charset for result serialization. Will use the accepted charset of the request.
     *
     * @return The encoding charset
     * @throws ControllerException if the header was not submitted not and no default value is set
     */
    public String getAcceptedCharset() throws ControllerException {
        return headerHandler.getHeaderValue(HttpHeader.ACCEPT_CHARSET, Charset.UTF8);
    }

    /**
     * Gets the HTTP method for result serialization. Will use the HTTP method of the request if
     * not overridden in the URL meta part of the URL info part. (e.g. my/path.put)
     *
     * @return The HTTP method
     */
    public String getHttpMethod() {
        String metaDataPart = getRequestUrlMetaPart();
        if (metaDataPart.contains(".get")) {
            return HttpMethod.GET;
        } else if (metaDataPart.contains(".post")) {
            return HttpMethod.POST;
        } else if (metaDataPart.contains(".put")) {
            return HttpMethod.PUT;
        } else if (metaDataPart.contains(".delete")) {
            return HttpMethod.DELETE;
        } else if (metaDataPart.contains(".head")) {
            return HttpMethod.HEAD;
        } else if (metaDataPart.contains(".options")) {
            return HttpMethod.OPTIONS;
        } else if (metaDataPart.contains(".trace")) {
            return HttpMethod.TRACE;
        }
        return request.getHttpMethod().toUpperCase();
    }

    /**
     * Gets the resource manager.
     *
     * @return The resource manager
     */
    public ResourceManager getResourceManager() {
        return resourceManager;
    }

    /**
     * Gets the documentation manager.
     *
     * @return The documentation manager
     */
    public DocumentationManager getDocumentationManager() {
        return documentationManager;
    }

    /**
     * Gets the controller configuration.
     *
     * @return The controller configuration
     */
    public Configuration getConfiguration() {
        return configuration;
    }

    /**
     * Gets the request URL resource path by removing the
     * meta data part from the URL info part. This is everything in
     * the URL info part from the beginning until the first occurring dot
     * (excluded). (my/path.put.xml -> my/path)
     *
     * @return The request URL resource part
     */
    public String getRequestUrlResourcePart() {
        String pathInfo = getRequestUrlInfoPart();
        if (pathInfo.contains(".")) {
            pathInfo = pathInfo.substring(0, pathInfo.indexOf("."));
        }
        return UriUtils.standardUri(pathInfo);
    }

    /**
     * Gets the request URL meta part. This is everything in
     * the URL info part from the first occurring dot
     * (included) till the end. (my/path.put.xml -> .put.xml)
     *
     * @return The request URL meta part
     */
    public String getRequestUrlMetaPart() {
        String pathInfo = getRequestUrlInfoPart();
        if (! pathInfo.contains(".")) {
            return "";
        }
        return pathInfo.substring(pathInfo.indexOf(".") - 1, pathInfo.length());
    }

    /**
     * Gets the request URL info part. This is the part that
     * is used by the framework for all allocation, path
     * parameter and meta data related information gathering.
     *
     * @return The request URL info part
     */
    public String getRequestUrlInfoPart() {
        return request.getUrlInfoPart();
    }

    /**
     * Gets the URI query parameters.
     *
     * @return The query parameter map
     */
    public Map<String, String[]> getQueryParameters() {
        return request.getParameters();
    }

    /**
     * Gets the resource parameters.
     *
     * @return The path parameter map
     */
    public Map<String, String[]> getPathParameters() {
        return resourceManager.getPathParameters(getRequestUrlResourcePart());
    }

    /**
     * Gets all query, form and resource parameters.
     *
     * @return The parameter map
     */
    public Map<String, String[]> getAllParameters() {
        Map<String, String[]> allParameters = new HashMap<String, String[]>();
        Map<String, String[]> queryParameters = getQueryParameters();
        if (queryParameters != null) {
             allParameters.putAll(queryParameters);
        }
        Map<String, String[]> pathParameters = getPathParameters();
        if (pathParameters != null) {
             allParameters.putAll(pathParameters);
        }
        return allParameters;
    }

    /**
     * Gets the body input stream. Will return null if HTTP method is not
     * PUT or POST and if the header 'Content-Type' is set to application/x-www-form-urlencoded.
     *
     * @return The input stream of the request body
     */
    public InputStream getRequestBodyInputStream() {
        return request.getBodyInputStream();
    }

    /**
     * Gets the output stream for streaming the serialized response body.
     *
     * @return The output stream
     */
    public OutputStream getResponseBodyOutputStream() {
        return response.getBodyOutputStream();
    }

    /**
     * Gets the normalized HTTP request.
     *
     * @return The HTTP request
     */
    public Request getRequest() {
        return request;
    }

    /**
     * Gets the normalized HTTP response.
     *
     * @return The HTTP response
     */
    public Response getResponse() {
        return response;
    }
}
