/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller;

import net.craftforge.essential.context.basic.BasicRequestImpl;
import net.craftforge.essential.context.basic.BasicResponseImpl;
import net.craftforge.essential.controller.managers.DocumentationManager;
import net.craftforge.essential.controller.managers.ResourceManager;
import net.craftforge.essential.controller.phases.flow.PhaseFlow;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.Map;

/**
 *
 *
 * @author Christian Bick
 * @since 02.12.2010
 */
public class Controller {

    private static final Logger LOGGER = LoggerFactory.getLogger(Controller.class);

    /**
     * Controls instantiation process to ensure that only one controller per package path and
     * configuration (identified by name) is used.
     *
     * @param packagePath The package path
     * @param config The configuration
     * @return The controller responsible for the package path and configuration combination
     */
    public static Controller getInstance(String packagePath, Configuration config) {
        return new Controller(packagePath, config);
    }

    /**
     * Controls instantiation process to ensure that only one controller per package path and
     * default configuration is used.
     *
     * @param packagePath The package path
     * @return The controller responsible for the package path using the default configuration
     */
    public static Controller getInstance(String packagePath) {
        return getInstance(packagePath, new Configuration());
    }

    /**
     * The resource manager
     */
    private ResourceManager resourceManager;

    /**
     * The documentation manager
     */
    private DocumentationManager documentationManager;

    /**
     * The configuration
     */
    private Configuration config;

    /**
     * Constructs a controller being responsible for the package path and configuration.
     *
     * @param packagePath The package path
     * @param config The configuration
     */
    private Controller (String packagePath, Configuration config) {
        LOGGER.debug("[Starting controller initialization] {}", packagePath);
        this.resourceManager = ResourceManager.getInstance(packagePath);
        this.documentationManager = DocumentationManager.getInstance(packagePath);
        this.config = config;
        LOGGER.debug("[Finished controller initialization] {}", packagePath);
    }

    /**
     * Performs a request on a controller using the HTTP method and
     * info part to allocate the resource. Uses the URL info part and
     * request parameters for invoking the resource method. Uses the header information
     * to properly to serialize the result into the output. Uses the output stream
     * for writing back the serialized result.
     *
     * @param httpMethod The HTTP method
     * @param urlInfoPart The URL info part (to identify the resource by this framework)
     * @param requestHeaders The request headers a map (keys = header names, values = header content)
     * @param requestParameters The request parameters
     * @param inputStream The input stream of the request body
     * @param outputStream The output stream of the response body
     * @throws ControllerException Failed to perform the request
     * @return The resulting HTTP status code
     */
    public int perform(String httpMethod,
                        String urlInfoPart,
                        Map<String, String[]> requestHeaders,
                        Map<String, String[]> requestParameters,
                        InputStream inputStream,
                        OutputStream outputStream) throws ControllerException {
        return perform(
                new BasicRequestImpl(httpMethod, urlInfoPart, requestHeaders, requestParameters, inputStream),
                new BasicResponseImpl(outputStream)
        );
    }

    /**
     * Performs a request on the controller, using the response to write back
     * the serialized result.
     *
     * @param request The request
     * @param response The response.
     * @throws ControllerException Failed to perform the request
     * @return The resulting HTTP status code
     */
    public int perform(Request request, Response response) throws ControllerException {

        LOGGER.debug("[Starting request execution] {} {}", request.getHttpMethod(), request.getUrlInfoPart());

        State state = new State();
        Setup setup = new Setup(resourceManager, documentationManager, config, request, response);

        LOGGER.info("[Performing request] {} {}", setup.getHttpMethod(), setup.getRequestUrlResourcePart());

        Class<?> phaseFlowClass = config.getPhaseFlow();
        PhaseFlow flow;
        try {
            flow = (PhaseFlow)phaseFlowClass.newInstance();
        } catch (InstantiationException e) {
            throw new ControllerException("Unable to instantiate phase flow class " + phaseFlowClass.getName(), e);
        } catch (IllegalAccessException e) {
            throw new ControllerException("Unable to instantiate phase flow class " + phaseFlowClass.getName(), e);
        }
        flow.setSetup(setup);
        flow.setState(state);

        Phase phase = flow.getFirstPhase();
        while (phase != null) {
            LOGGER.debug("[Running phase] {}", phase.getClass().getName());
            phase.run();
            phase = flow.getPhaseAfter(phase);
        }

        LOGGER.debug("[Finished request execution] {} {}", request.getHttpMethod(), request.getUrlInfoPart());

        return state.getStatus().getCode();
    }
}
