/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.controller;

import net.craftforge.commons.configuration.PropertiesHolder;
import net.craftforge.essential.controller.phases.flow.StandardPhaseFlow;
import net.craftforge.essential.supply.jaxb.JaxbConsumer;
import net.craftforge.essential.supply.jaxb.JaxbProducer;

import java.util.Map;

/**
 * A controller configuration.
 *
 * @author Christian Bick
 * @since 03.03.2011
 */
public class Configuration extends PropertiesHolder {

    /**
     * The default configuration file
     */
    public static final String DEFAULT_CONFIG_FILE = "essential.properties";

    public static final String PROPERTY_DEFAULT_CONSUMER = "io.consumer";
    public static final String PROPERTY_DEFAULT_PRODUCER = "io.producer";
    public static final String PROPERTY_PHASE_FLOW = "phase.flow";
    public static final String PROPERTY_AUTHENTICATION_HANDLER = "authentication.handler";
    public static final String PROPERTY_XML_DESERIALIZATION = "io.xml.in";
    public static final String PROPERTY_XML_SERIALIZATION = "io.xml.out";
    public static final String PROPERTY_JSON_DESERIALIZATION = "io.json.in";
    public static final String PROPERTY_JSON_SERIALIZATION = "io.json.out";

    public static final Class<?> DEFAULT_CONSUMER = JaxbConsumer.class;
    public static final Class<?> DEFAULT_PRODUCER = JaxbProducer.class;
    public static final Class<?> DEFAULT_PHASE_FLOW = StandardPhaseFlow.class;

    /**
     * Initializes a configuration using the default configuration file.
     */
    public Configuration() {
        this(DEFAULT_CONFIG_FILE);
    }

    /**
     * Initializes a configuration using the given configuration file.
     *
     * @param configFile The configuration file
     */
    public Configuration(String configFile) {
        super("/" + configFile);
    }

    /**
     * Gets the class of the default consumer.
     *
     * @return The class of the default consumer
     */
    public Class<?> getDefaultConsumer() {
        return getClass(PROPERTY_DEFAULT_CONSUMER, DEFAULT_CONSUMER);
    }

    /**
     * <p>Sets the class of the default consumer. The default
     * consumer is used if a resource method or its resource class is not
     * annotated with the @Consumer annotation.</p>
     * <p>
     *     <b>Default:</b> <code>net.craftforge.essential.supply.jaxb.JaxbConsumer</code>
     * </p>
     *
     * @param defaultConsumer The class of the default consumer
     */
    public void setDefaultConsumer(Class<?> defaultConsumer) {
        setProperty(PROPERTY_DEFAULT_CONSUMER, defaultConsumer.getName());
    }

    /**
     * Gets the class of the default producer.
     *
     * @return The class of the default producer.
     */
    public Class<?> getDefaultProducer() {
        return getClass(PROPERTY_DEFAULT_PRODUCER, DEFAULT_PRODUCER);
    }

    /**
     * Sets the class of the default producer. The default
     * producer is used if a resource method or its resource class is not
     * annotated with the @Producer annotation.
     * <p>
     *     <b>Default:</b> <code>net.craftforge.essential.supply.jaxb.JaxbProducer</code>
     * </p>
     *
     * @param defaultProducer The class of the default producer.
     */
    public void setDefaultProducer(Class<?> defaultProducer) {
        setProperty(PROPERTY_DEFAULT_PRODUCER, defaultProducer.getName());
    }

    /**
     * Gets the class of the phase flow defining the controller's phase execution logic.
     *
     * @return The class of the phase flow
     */
    public Class<?> getPhaseFlow() {
        return getClass(PROPERTY_PHASE_FLOW, DEFAULT_PHASE_FLOW);
    }

    /**
     * Sets the class of the phase flow defining the controller's phase execution logic.
     * <p>
     *     <b>Default:</b> <code>net.craftforge.essential.controller.phases.flow.StandardPhaseFlow</code>
     * </p>
     *
     * @param phaseFlow The class of the phase flow
     */
    public void setPhaseFlow(Class<?> phaseFlow) {
        setProperty(PROPERTY_PHASE_FLOW, phaseFlow.getName());
    }

    /**
     * Gets the class of the authentication handler.
     *
     * @return The class of the authentication handler
     */
    public Class<?> getAuthenticationHandler() {
        return getClass(PROPERTY_AUTHENTICATION_HANDLER, null);
    }

    /**
     * Sets the class of the authentication handler. If no authentication handler is
     * specified, the authentication phase will be skipped.
     *
     * @param authenticationHandler The class of the authentication handler
     */
    public void setAuthenticationHandler(Class<?> authenticationHandler) {
        setProperty(PROPERTY_AUTHENTICATION_HANDLER, authenticationHandler.getName());
    }

    /**
     * Gets the XML deserialization properties. Adding properties according to
     * specifications can be used to fine-tune the XML deserialization process.
     *
     * @return The XML deserialization properties as Map
     */
    public Map<String, String> getXmlDeserializationProperties() {
        return getProperties(PROPERTY_XML_DESERIALIZATION);
    }

    /**
     * Sets the XML deserialization properties. Adding properties according to
     * specifications can be used to fine-tune the XML deserialization process.
     *
     * @param xmlDeserializationProperties The XML deserialization properties as Map
     */
    public void setXmlDeserializationProperties(Map<String, String> xmlDeserializationProperties) {
        setProperties(PROPERTY_XML_DESERIALIZATION, xmlDeserializationProperties);
    }

    /**
     * Gets the JSON deserialization properties. Adding properties according to
     * specifications can be used to fine-tune the JSON deserialization process.
     *
     * @return The JSON deserialization properties as Map
     */
    public Map<String, String> getJsonDeserializationProperties() {
        return getProperties(PROPERTY_JSON_DESERIALIZATION);
    }

    /**
     * Sets the JSON deserialization properties. Adding properties according to
     * specifications can be used to fine-tune the JSON deserialization process.
     *
     * @param jsonDeserializationProperties The JSON deserialization properties as Map
     */
    public void setJsonDeserializationProperties(Map<String, String> jsonDeserializationProperties) {
        setProperties(PROPERTY_JSON_DESERIALIZATION, jsonDeserializationProperties);
    }

    /**
     * Gets the XML serialization properties. Adding properties according to
     * specifications can be used to fine-tune the XML serialization process.
     *
     * @return The XML serialization properties as Map
     */
    public Map<String, String> getXmlSerializationProperties() {
        return getProperties(PROPERTY_XML_SERIALIZATION);
    }

    /**
     * Sets the XML serialization properties. Adding properties according to
     * specifications can be used to fine-tune the XML serialization process.
     *
     * @param xmlSerializationProperties The XML serialization properties as Map
     */
    public void setXmlSerializationProperties(Map<String, String> xmlSerializationProperties) {
        setProperties(PROPERTY_XML_SERIALIZATION, xmlSerializationProperties);
    }

    /**
     * Gets the JSON serialization properties. Adding properties according to
     * specifications can be used to fine-tune the JSON serialization process.
     *
     * @return The JSON serialization properties as Map
     */
    public Map<String, String> getJsonSerializationProperties() {
        return getProperties(PROPERTY_JSON_SERIALIZATION);
    }

    /**
     * Sets the JSON serialization properties. Adding properties according to
     * specifications can be used to fine-tune the JSON serialization process.
     *
     * @param jsonSerializationProperties The JSON serialization properties as Map
     */
    public void setJsonSerializationProperties(Map<String, String> jsonSerializationProperties) {
        setProperties(PROPERTY_JSON_SERIALIZATION, jsonSerializationProperties);
    }
}
