/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.context.basic;

import net.craftforge.essential.controller.Response;
import net.craftforge.essential.controller.constants.Charset;
import net.craftforge.essential.controller.constants.HttpStatusCode;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.Map;

/**
 * A basic implementation of a controller response.
 *
 * @author Christian Bick
 * @since 20.06.11
 */
public class BasicResponseImpl implements Response {

    private HttpStatusCode httpStatusCode;
    private Map<String, String[]> headers;
    private OutputStream outputStream;

    /**
     * Creates a basic response.
     *
     * @param outputStream The HTTP response body output stream.
     */
    public BasicResponseImpl(OutputStream outputStream) {
        this.outputStream = outputStream;
    }

    /**
     * {@inheritDoc}
     */
    public void sendHeaders(HttpStatusCode httpStatusCode, Map<String, String[]> headers) {
        this.httpStatusCode = httpStatusCode;
        this.headers = headers;
        BufferedWriter writer;
        try{
            writer = new BufferedWriter(new OutputStreamWriter(outputStream, Charset.UTF8));
            writer.write("HTTP/1.1 " + httpStatusCode.getCode() + " " + httpStatusCode.getName());
            writer.newLine();
            for (String headerName : headers.keySet()) {
                for (String headerValue : headers.get(headerName)) {
                    writer.write(headerName + ": " + headerValue);
                    writer.newLine();
                }
            }
            writer.flush();
        } catch (IOException e) {
            throw new RuntimeException("Unable to send response headers", e);
        }
    }

    /**
     * {@inheritDoc}
     */
    public HttpStatusCode getStatusCode() {
        return httpStatusCode;
    }

    /**
     * {@inheritDoc}
     */
    public Map<String, String[]> getHeaders() {
        return headers;
    }

    /**
     * {@inheritDoc}
     */
    public OutputStream getBodyOutputStream() {
        return outputStream;
    }
}
