/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.client;

import net.craftforge.essential.controller.constants.HttpMethod;
import net.craftforge.essential.controller.utils.HeaderUtils;
import net.craftforge.essential.controller.utils.UriUtils;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicHeader;
import org.apache.http.params.BasicHttpParams;
import org.apache.http.params.HttpParams;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * A network client performs client requests remotely via HTTP.
 *
 * @author Christian Bick
 * @since 06.07.11
 */
public class NetworkClient implements Client {

    private String urlBasePart;

    /**
     * <p>Creates an HTTP client that uses the given URL base part for its
     * requests. The URL base part looks like follows.</p>
     *
     * <p><b>schema:</b> <code>protocol://host:port/context</code></p>
     *
     * <p><b>example:</b> <code>http://api.craftforge.net:8080/core</code></p>
     *
     * <p>
     *     The URL base part is prefixed to every URL info part of a request. Together
     *     with an optional URL query part the complete request URL is assembled from
     *     these three parts.
     * </p>
     * @param urlBasePart The URL base part
     */
    public NetworkClient(String urlBasePart) {
        this.urlBasePart = urlBasePart;
    }

    /**
     * Performs an HTTP request from the data provided by the given client
     * request and retrieves an HTTP response that as provided as client
     * response.
     *
     * @param clientRequest The client request
     * @return The client response
     */
    public ClientResponse performRequest(ClientRequest clientRequest) {
        HttpClient httpClient = new DefaultHttpClient();
        try {
            HttpRequestBase httpRequest = getHttpRequest(clientRequest);
            HttpResponse httpResponse = httpClient.execute(httpRequest);
            return getClientResponse(httpResponse);
        } catch (IOException e) {
            throw new RuntimeException("Failed to perform http request", e);
        } catch (URISyntaxException e) {
            throw new RuntimeException("Invalid URI", e);
        }
    }

    /**
     * Gets an HTTP request from the provided client request.
     *
     * @param request The client request
     * @return The HTTP request
     * @throws URISyntaxException if the request or base URI is not valid
     */
    protected HttpRequestBase getHttpRequest(ClientRequest request) throws URISyntaxException {
        final String httpMethod = request.getHttpMethod();
        HttpRequestBase httpRequest;
        if (httpMethod.equalsIgnoreCase(HttpMethod.POST) || httpMethod.equalsIgnoreCase(HttpMethod.PUT)) {
            HttpEntityEnclosingRequestBase entityRequest = new HttpEntityEnclosingRequestBase() {
                @Override
                public String getMethod() {
                    return httpMethod;
                }
            };
            entityRequest.setEntity(new InputStreamEntity(request.getRequestBody(), -1));
            httpRequest = entityRequest;
        } else {
            httpRequest = new HttpRequestBase() {
                @Override
                public String getMethod() {
                    return httpMethod;
                }
            };
            httpRequest.setParams(getRequestParameters(request.getParameters()));
        }
        String uriString = urlBasePart + "/" + request.getUrlInfoPart();
        uriString = UriUtils.slimUri(uriString);
        httpRequest.setURI(new URI(uriString));
        httpRequest.setHeaders(getRequestHeaders(request.getHeaders()));

        return httpRequest;
    }

    /**
     * Gets the HTTP parameters from the provided parameter map.
     *
     * @param parameters The parameter map
     * @return The HTTP parameters
     */
    protected HttpParams getRequestParameters(Map<String, String[]> parameters) {
        BasicHttpParams httpParameters = new BasicHttpParams();
        for (String paramName : parameters.keySet()) {
            String[] paramValues = parameters.get(paramName);
            if (paramValues.length > 0) {
                httpParameters.setParameter(paramName, paramValues[0]);
            }
        }
        return httpParameters;
    }

    /**
     * Gets the HTTP headers from the provided header map.
     *
     * @param headers The header map
     * @return The HTTP headers
     */
    protected Header[] getRequestHeaders(Map<String, String[]> headers) {
        List<Header> httpHeaders = new ArrayList<Header>(headers.size());
        for (String headerName : headers.keySet()) {
            String[] headerValues = headers.get(headerName);
            for (String headerValue : headerValues) {
                httpHeaders.add(new BasicHeader(headerName, headerValue));
            }
        }
        return httpHeaders.toArray(new Header[httpHeaders.size()]);
    }

    /**
     * Gets the client response from the provided HTTP response.
     *
     * @param httpResponse The http response
     * @return The client response
     * @throws IOException if the response reading failed
     */
    protected ClientResponse getClientResponse(HttpResponse httpResponse) throws IOException {
        HttpEntity entity = httpResponse.getEntity();
        InputStream inputStream = entity == null ? null : entity.getContent();
        return new ClientResponse(
                httpResponse.getStatusLine().getStatusCode(),
                getResponseHeaders(httpResponse.getAllHeaders()),
                inputStream
        );
    }

    /**
     * Gets the response header map from the given HTTP headers.
     *
     * @param headers The response HTTP headers
     * @return The response header map
     */
    protected Map<String, String[]> getResponseHeaders(Header[] headers) {
        Map<String, List<String>> responseHeaders = new HashMap<String, List<String>>(headers.length * 2);
        for (Header header : headers) {
            String headerName = header.getName();
            List<String> headerValueList = responseHeaders.get(headerName);
            if (headerValueList == null)  {
                headerValueList = new ArrayList<String>(1);
            }
            headerValueList.add(header.getValue());
            responseHeaders.put(headerName, headerValueList);
        }
        return HeaderUtils.normalizeHeaderMap(responseHeaders);
    }

}
