/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.client;

import net.craftforge.essential.context.basic.BasicRequestImpl;
import net.craftforge.essential.context.basic.BasicResponseImpl;
import net.craftforge.essential.controller.*;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;

/**
 * A local client performs client requests locally against via a
 * controller.
 *
 * @author Christian Bick
 * @since 06.07.11
 */
public class LocalClient implements Client {

    private Controller controller;

    /**
     * Creates a local client and initializes a controller for resource classes
     * in the given package path.
     *
     * @param packagePath The package path
     */
    public LocalClient(String packagePath) {
        this.controller = Controller.getInstance(packagePath);
    }

    /**
     * Creates a local client and initializes a controller for resource classes
     * in the given package path. Uses the specified configuration to configure
     * the controller.
     *
     * @param packagePath The package path
     * @param config The controller configuration
     */
    public LocalClient(String packagePath, Configuration config) {
        this.controller = Controller.getInstance(packagePath, config);
    }

    /**
     * Performs a local request by delegating the client request to the used
     * controller. Retrieves the controller output and provides it as
     * client response.
     *
     * @param clientRequest The client request
     * @return The client response
     */
    public ClientResponse performRequest(ClientRequest clientRequest) {

        Request request = new BasicRequestImpl(
                clientRequest.getHttpMethod(),
                clientRequest.getUrlInfoPart(),
                clientRequest.getHeaders(),
                clientRequest.getParameters(),
                clientRequest.getRequestBody()
        );

        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        Response response = new BasicResponseImpl(outputStream);

        try {
            this.controller.perform(request, response);
        } catch (ControllerException e) {
            throw new RuntimeException(e);
        }
        return new ClientResponse(
                response.getStatusCode().getCode(),
                response.getHeaders(),
                new ByteArrayInputStream(outputStream.toByteArray())
        );
    }
}
