/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.client;

import net.craftforge.essential.controller.ControllerException;
import net.craftforge.essential.controller.constants.Charset;
import net.craftforge.essential.controller.constants.MediaType;
import net.craftforge.essential.controller.managers.BodyHandler;
import net.craftforge.essential.controller.managers.HeaderHandler;
import net.craftforge.essential.supply.Consumer;

import java.io.InputStream;
import java.util.Map;

/**
 * A client response is used to provide a normalized response output for a client.
 *
 * @author Christian Bick
 * @since 04.07.11
 */
public class ClientResponse {

    private int statusCode;
    private HeaderHandler headerHandler;
    private BodyHandler bodyHandler;

    /**
     * Creates a client response using the given HTTP status code,
     * response headers and response body.
     *
     * @param statusCode The HTTP status code
     * @param headers The HTTP response headers
     * @param bodyInputStream The HTTP response body input stream
     */
    public ClientResponse(int statusCode, Map<String, String[]> headers, InputStream bodyInputStream) {
        this.statusCode = statusCode;
        this.headerHandler = new HeaderHandler(headers);
        this.bodyHandler = new BodyHandler(bodyInputStream);
    }

    /**
     * Gets the HTTP status code.
     *
     * @return The HTTP status code
     */
    public int getStatusCode() {
        return statusCode;
    }

    /**
     * Gets the HTTP response body input stream.
     *
     * @return The HTTP response body input stream
     */
    public InputStream getBodyInputStream() {
        return bodyHandler.getBodyInputStream();
    }

    /**
     * Gets the HTTP response body string. Uses the charset
     * from in the Content-Type header for decoding if specified there.
     * Assumes UTF-8 otherwise.
     *
     * @return The response body string
     */
    public String getBodyContent() {
        String charset = getContentCharset();
        return bodyHandler.getBodyContent(charset);
    }

    /**
     * Gets the HTTP response body as an object of the given class. Uses the media
     * type from the Content-Type header for deligating the deserialization to
     * right consumer method. Uses the charset from the Content-Type header for
     * decoding.
     * Assumes text/xml as media type and UTF-8 as charset if
     * not specified in the Content-Type header.
     *
     * @param consumer The consumer
     * @param clazz The class
     * @return The response body object
     * @throws ControllerException if the deserialization fails
     */
    public Object getBodyObject(Consumer consumer, Class<?> clazz) throws ControllerException {
        String mediaType = getContentMediaType();
        String charset = getContentCharset();
        return bodyHandler.getBodyObject(consumer, clazz, mediaType, charset);
    }

    /**
     * Gets the response body's media type from the
     * Content-Type header. Assumes the media type to
     * be text/xml if it is not specified in the header.
     *
     * @return The response body's media type
     */
    public String getContentMediaType() {
        return headerHandler.getContentMediaType(MediaType.TEXT_XML);
    }

    /**
     * Gets the response body's charset encoding from the
     * Content-Type header. Assumes the charset encoding
     * to be UTF-8 if it is not specified in the header.
     *
     * @return The response body's charset encoding
     */
    public String getContentCharset() {
        return headerHandler.getContentCharset(Charset.UTF8);
    }

    /**
     * Gets the user from the authorization header.
     *
     * @return The authorization user
     */
    public String getAuthorizationUser() {
        return headerHandler.getAuthorizationUser();
    }

    /**
     * Gets the password from the authorization header.
     *
     * @return The authorization password
     */
    public String getAuthorizationPassword() {
        return headerHandler.getAuthorizationPassword();
    }
}
