/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.client;

import net.craftforge.essential.controller.ControllerException;
import net.craftforge.essential.controller.constants.Charset;
import net.craftforge.essential.controller.constants.HttpHeader;
import net.craftforge.essential.controller.utils.HeaderUtils;
import net.craftforge.essential.supply.Producer;
import org.apache.http.HttpHeaders;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

/**
 * A client request is used to provide a normalized request input for a client.
 *
 * @author Christian Bick
 * @since 04.07.11
 */
public class ClientRequest {

    private String httpMethod;
    private String urlInfoPart;
    private Map<String, String[]> headers = new HashMap<String, String[]>();
    private Map<String, String[]> parameters = new HashMap<String, String[]>();
    private InputStream requestBody;

    /**
     * Creates a client request. Uses the specified HTTP method
     * and allocates the resource using the URL info part. Will use
     * default headers and no query parameters.
     *
     * @param httpMethod The HTTP request method
     * @param urlInfoPart The request URL info part
     */
    public ClientRequest(String httpMethod, String urlInfoPart) {
        this.httpMethod = httpMethod;
        this.urlInfoPart = urlInfoPart;
    }

    /**
     * Gets the HTTP request method.
     *
     * @return The HTTP request method
     */
    public String getHttpMethod() {
        return httpMethod;
    }

    /**
     * Gets the request URL info part.
     *
     * @return The URL info part
     */
    public String getUrlInfoPart() {
        return urlInfoPart;
    }

    /**
     * Gets the HTTP request headers.
     *
     * @return The HTTP request headers
     */
    public Map<String, String[]> getHeaders() {
        return headers;
    }

    /**
     * Gets the HTTP request parameters.
     *
     * @return The HTTP request parameters
     */
    public Map<String, String[]> getParameters() {
        return parameters;
    }

    /**
     * Gets the HTTP request body.
     *
     * @return The HTTP request body
     */
    public InputStream getRequestBody() {
        return requestBody;
    }

    /**
     * Sets the HTTP request authorization header.
     *
     * @param username The authorization username
     * @param password The authorization password
     */
    public void setAuthorization(String username, String password) {
        String headerValue = HeaderUtils.getBasicAuthorizationHeader(username, password);
        setHeader(HttpHeader.AUTHORIZATION, headerValue);
    }

    /**
     * Sets the request body for this request and specifies the media type
     * used in the Content-Type header. Uses UTF-8 as charset.
     *
     * @param bodyContent The body content as a string
     * @param mediaType The media type
     */
    public void setBody(String bodyContent, String mediaType) {
        setBody(bodyContent, mediaType, Charset.UTF8);
    }

    /**
     * Sets the request body for this request and specifies the media type
     * and charset used in the Content-Type header.
     *
     * @param bodyContent The body content as a string
     * @param mediaType The media type
     * @param charset The charset
     */
    public void setBody(String bodyContent, String mediaType, String charset) {
        InputStream bodyInputStream;
        try {
            bodyInputStream = new ByteArrayInputStream(bodyContent.getBytes(charset));
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
        setBody(bodyInputStream, mediaType, charset);
    }

    /**
     * Sets the request body input stream for this request and specifies the media type
     * used in the Content-Type header. Uses UTF-8 as charset.
     *
     * @param bodyInputStream The body as an input stream
     * @param mediaType The media type
     */
    public void setBody(InputStream bodyInputStream, String mediaType) {
        setBody(bodyInputStream, mediaType, Charset.UTF8);
    }

    /**
     * Sets the request body input stream for this request and specifies the media type
     * and charset used in the Content-Type header.
     *
     * @param bodyInputStream The body as an input stream
     * @param mediaType The media type
     * @param charset The charset
     */
    public void setBody(InputStream bodyInputStream, String mediaType, String charset) {
        requestBody = bodyInputStream;
        String headerValue = mediaType + "; charset=" + charset;
        setHeader(HttpHeaders.CONTENT_TYPE, headerValue);
    }

    /**
     * Sets the request body for this request and specifies the media type
     * and charset used in the Content-Type header. Uses UTF-8 as charset.
     *
     * @param object The request body as an object
     * @param producer The producer to serialize the object
     * @param mediaType The media type
     * @throws ControllerException if the object serialization process fails
     */
    public void setBody(Object object, Producer producer, String mediaType) throws ControllerException {
        setBody(object, producer, mediaType, Charset.UTF8);
    }

    /**
     * Sets the request body for this request and specifies the media type
     * and charset used in the Content-Type header.
     *
     * @param object The request body as an object
     * @param producer The producer to serialize the object
     * @param mediaType The media type
     * @param charset The charset
     * @throws ControllerException if the object serialization process fails
     */
    public void setBody(Object object, Producer producer, String mediaType, String charset) throws ControllerException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        producer.produce(mediaType, object, outputStream, charset);
        InputStream bodyInputStream = new ByteArrayInputStream(outputStream.toByteArray());
        setBody(bodyInputStream, mediaType, charset);
    }

    /**
     * Sets the header with the given header name to the specified
     * header value. Will overwrite any existing header data with
     * the same header name.
     *
     * @param headerName The header name
     * @param headerValue The header value
     */
    public void setHeader(String headerName, String headerValue) {
        String[] headerValues = new String[] { headerValue };
        setHeader(headerName, headerValues);
    }

    /**
     * Sets the header with the given header name to the specified
     * header values. Will overwrite any existing header data with
     * the same header name.
     *
     * @param headerName The header name
     * @param headerValues The header values
     */
    public void setHeader(String headerName, String[] headerValues) {
        headers.put(headerName, headerValues);
    }

    /**
     * Sets the headers.
     *
     * @param headers The headers
     */
    public void setHeaders(Map<String, String[]> headers) {
        this.headers = headers;
    }

    /**
     * Sets the parameter with the given parameter name to the specified
     * parameter value. Will overwrite any existing parameter data with
     * the same parameter name.
     *
     * @param parameterName The parameter name
     * @param parameterValue The parameter value
     */
    public void setParameter(String parameterName, String parameterValue) {
        String[] parameterValues = new String[] { parameterValue };
        setParameter(parameterName, parameterValues);
    }

    /**
     * Sets the parameter with the given parameter name to the specified
     * parameter values. Will overwrite any existing parameter data with
     * the same parameter name.
     *
     * @param parameterName The parameter name
     * @param parameterValues The parameter values
     */
    public void setParameter(String parameterName, String[] parameterValues) {
        parameters.put(parameterName, parameterValues);
    }

    /**
     * Sets the parameters.
     *
     * @param parameters The parameters
     */
    public void setParameters(Map<String, String[]> parameters) {
        this.parameters = parameters;
    }
}
