/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.commons.io;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import java.io.*;

/**
 * Stream Utils provide algorithms for standard stream operations.
 *
 * @author Christian Bick
 * @since 29.08.11
 */
public class StreamUtils {

    /**
     * Reads a string from an input stream using the given charset
     * for decoding.
     *
     * @param inputStream The input stream
     * @param charset The charset
     * @return The read string
     * @throws IOException if the input stream cannot be read
     */
    public static String readStringFromInputStream(InputStream inputStream, String charset) throws IOException {
        InputStreamReader inputStreamReader = new InputStreamReader(inputStream, charset);
        return readStringFromInputStreamReader(inputStreamReader);
    }

    /**
     * Reads a string from an input stream using the given input stream reader.
     *
     * @param inputStreamReader The input stream reader
     * @return The read string
     * @throws IOException if the input stream cannot be read
     */
    public static String readStringFromInputStreamReader(InputStreamReader inputStreamReader) throws IOException {
        StringBuilder stringBuilder = new StringBuilder();
        int read;
        char[] charBuffer = new char[8192];
        while ((read = inputStreamReader.read(charBuffer)) != -1) {
            stringBuilder.append(charBuffer, 0, read);
        }
        inputStreamReader.close();
        return stringBuilder.toString();
    }

    /**
     * Gets an object from an XML output stream using the given charset.
     *
     * @param clazz The object's class
     * @param outputStream The XML output stream
     * @param charset The charset
     * @return The object
     */
    public static Object getObjectFromXmlOutputStream(Class<?> clazz, ByteArrayOutputStream outputStream, String charset) {
        Object result = null;
        try {
            JAXBContext jaxbContext = JAXBContext.newInstance(clazz);
            javax.xml.bind.Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
            result = unmarshaller.unmarshal(new InputStreamReader(new ByteArrayInputStream(outputStream.toByteArray()), charset));
        } catch (JAXBException e) {
            throw new RuntimeException(e);
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
        return result;
    }

    /**
     * Gets an XML input stream from an object using the given charset.
     *
     * @param object The object
     * @param charset The charset
     * @return The XML input stream
     */
    public static InputStream getXmlInputStreamFromObject(Object object, String charset) {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        try {
            OutputStreamWriter outputStreamWriter = new OutputStreamWriter(outputStream, charset);
            JAXBContext jaxbContext = JAXBContext.newInstance(object.getClass());
            Marshaller marshaller = jaxbContext.createMarshaller();
            marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
            marshaller.marshal(object, outputStreamWriter);
        } catch (JAXBException e) {
            throw new RuntimeException(e);
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e);
        }
        return new ByteArrayInputStream(outputStream.toByteArray());
    }
}
