/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.commons.configuration;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * A properties loader provides a cached version of a properties file.
 *
 * @author Christian Bick
 * @since 30.08.11
 */
public class PropertiesLoader {

    private static Logger LOGGER = LoggerFactory.getLogger(PropertiesLoader.class);

    private static ConcurrentMap<String, PropertiesLoader> instances = new ConcurrentHashMap<String, PropertiesLoader>();

    /**
     * Loads properties from a property resource via the class loader
     * of this class. A copy of the same properties resource is returned for
     * each resource. Caches loaded properties.
     *
     * @param resource The properties resource
     * @return The properties copy
     */
    public static Properties loadProperties(String resource) {
        if (! instances.containsKey(resource)) {
            LOGGER.info("[Property loader initialization] " + resource);
            instances.putIfAbsent(resource, new PropertiesLoader(resource));
        }
        Properties properties = instances.get(resource).getProperties();
        Properties propertiesCopy = new Properties();
        propertiesCopy.putAll(properties);
        return propertiesCopy;
    }

    private Properties properties;

    /**
     * Initializes the property loader from a resource.
     *
     * @param resource The resource
     */
    private PropertiesLoader(String resource) {
        properties = new Properties();
        InputStream is = getClass().getResourceAsStream(resource);
        if (is == null) {
            return;
        }
        try {
            properties.load(is);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Loads the properties from the properties file.
     */
    private void loadProperties() {

    }

    /**
     * Gets the properties
     *
     * @return The properties
     */
    private Properties getProperties() {
        return properties;
    }
}
