/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.commons.configuration;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * A properties holder provides a convenient way of accessing a properties resource.
 *
 * @author Christian Bick
 * @since 17.08.11
 */
public class PropertiesHolder {

    private Properties properties;
    private ConcurrentMap<String, Class<?>> classes = new ConcurrentHashMap<String, Class<?>>();
    private ConcurrentMap<String, Map<String, String>> maps = new ConcurrentHashMap<String, Map<String, String>>();

    /**
     * Initializes the properties holder from a resource.
     *
     * @param resource The resource
     */
    public PropertiesHolder(String resource) {
        properties = PropertiesLoader.loadProperties(resource);
    }

    /**
     * Gets a property from the holder or manually added properties.
     *
     * @param propertyName The property name
     * @return The property value
     */
    public String getProperty(String propertyName) {
        return properties.getProperty(propertyName);
    }

    /**
     * Sets a property of the given name to the given value.
     *
     * @param propertyName The property name
     * @param propertyValue The property value
     */
    public void setProperty(String propertyName, String propertyValue) {
        properties.setProperty(propertyName, propertyValue);
        removeClass(propertyName);
        removeMaps(propertyName);
    }

    /**
     * Sets the properties in the given property map and puts the given
     * property name prefix before to the property name keys.
     *
     * @param propertyNamePrefix The property name prefix
     * @param properties The property map
     */
    public void setProperties(String propertyNamePrefix, Map<String, String> properties) {
        for (String key : properties.keySet()) {
            setProperty(propertyNamePrefix + "." + key, properties.get(key));
        }
    }

    /**
     * <p>Gets a map of properties which match the given property name prefix.</p>
     * <p>
     * <b>Example:</b><br/>
     * <br/>
     * <code>my.property.KEY_ONE = value1<br/>
     * my.property.KEY_TWO = value2</code><br/>
     * <br/>
     * <code>my.property</code> -> { KEY_ONE : value1 ; KEY_TWO : value2 }
     * </p>
     *
     * @param propertyNamePrefix The property name prefix
     * @return The matched map of properties
     */
    public Map<String, String> getProperties(String propertyNamePrefix) {
        String prefix = propertyNamePrefix + ".";
        maps.putIfAbsent(prefix, createMap(prefix));
        return maps.get(prefix);
    }

    /**
     * Gets a class from a property if the property value represents a
     * class path known to class loader.
     *
     * @param propertyName The property name
     * @param defaultClass The default class to return if the property is not set
     * @return The class
     */
    public Class<?> getClass(String propertyName, Class<?> defaultClass) {
        if (!classes.containsKey(propertyName)) {
            Class<?> clazz = createClass(propertyName, defaultClass);
            if (clazz == null) {
                return clazz;
            }
            classes.putIfAbsent(propertyName, clazz);
        }
        return classes.get(propertyName);
    }

    /**
     * Creates a class from a property using a default class if
     * the property is not set.
     *
     * @param propertyName The property name
     * @param defaultClass The default class
     * @return The created class
     */
    protected Class<?> createClass(String propertyName, Class<?> defaultClass) {
        String property = properties.getProperty(propertyName);
        if (property == null) {
            return defaultClass;
        }
        try {
            return Class.forName(property);
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("Failed to load configuration when processing property " + propertyName, e);
        }
    }

    /**
     * Creates a map of properties which match the given property name prefix.
     *
     * @param prefix The property name prefix
     * @return The created matched map of properties
     */
    protected Map<String, String> createMap(String prefix) {
        Map<String, String> newMap = new HashMap<String, String>();
        for (Object key : properties.keySet()) {
            String propertyName = (String)key;
            if (propertyName.startsWith(prefix)) {
                String newKey = propertyName.replace(prefix, "");
                String newValue = properties.getProperty(propertyName);
                newMap.put(newKey, newValue);
            }
        }
        return newMap;
    }

    /**
     * Removes the class associated to the given property name
     * from the caching map.
     *
     * @param propertyName The property name
     */
    protected void removeClass(String propertyName) {
        classes.remove(propertyName);
    }

    /**
     * Removes all maps associated to all possible prefixes
     * of the given property name from the caching map.
     *
     * @param propertyName The property name
     */
    protected void removeMaps(String propertyName) {
        String[] propertyNameParts = propertyName.split(".");
        String currentPrefix = "";
        for (String part : propertyNameParts) {
            if (!currentPrefix.isEmpty()) {
                currentPrefix += ".";
            }
            currentPrefix += part;
            maps.remove(currentPrefix);
        }
    }

}
