package net.consentmanager.cm_sdk_android_v3

import androidx.annotation.Keep
import kotlinx.serialization.*
import kotlinx.serialization.json.*

@Keep
@Serializable
data class CMPConsentModel(
    @SerialName("cmpString") val cmpStringBase64Encoded: String? = null,
    @SerialName("addtlConsent") val googleAdditionalConsent: String? = null,
    @SerialName("consentstring") val consentString: String? = null,
    val gdprApplies: Boolean? = null,
    val googleVendorConsents: Map<String, Boolean>? = null,
    val hasGlobalScope: Boolean? = null,
    val publisherCC: String? = null,
    val regulation: Int? = null,
    val regulationKey: String? = null,
    @SerialName("tcfcompliant") val tcfCompliant: Boolean? = null,
    @SerialName("tcfversion") val tcfVersion: Int? = null,
    val lastButtonEvent: Int? = null,
    @SerialName("tcfcaversion") val tcfcaVersion: Int? = null,
    @SerialName("gppversions") val gppVersions: List<String>? = null,
    @SerialName("uspstring") val uspString: String? = null,
    val vendorsList: List<Vendor> = emptyList(),
    val purposesList: List<Purpose> = emptyList(),
    val purposeLI: Map<String, Boolean>? = null,
    val vendorLI: Map<String, Boolean>? = null,
    val vendorConsents: Map<String, Boolean>? = null,
    val purposeConsents: Map<String, Boolean>? = null,
    val metadata: List<CmpMetadata> = emptyList(),
    val userChoiceExists: Boolean? = null,
    val purModeActive: Boolean? = null,
    val purModeLoggedIn: Boolean? = null,
    val purModeLogic: Int? = null,
    val consentExists: Boolean? = null,
    val consentMode: Map<ConsentType, ConsentStatus>? = null,
    val gppdata: GPPData? = null
) {
    fun hasPurposeConsent(id: String): Boolean =
        purposeConsents?.get(id.lowercase()) == true

    fun hasVendorConsent(id: String): Boolean =
        vendorConsents?.get(id.lowercase()) == true

    fun exportCMPInfo(): String =
        cmpStringBase64Encoded ?: ""

    fun getAllPurposesIDs(): List<String> =
        purposesList.map { it.id }

    fun getEnabledPurposesIDs(): List<String> =
        purposeConsents?.keys?.toList() ?: emptyList()

    fun getDisabledPurposesIDs(): List<String> {
        val allPurposes = getAllPurposesIDs().toSet()
        val enabledPurposes = getEnabledPurposesIDs().toSet()
        return (allPurposes - enabledPurposes).toList()
    }

    fun getAllVendorsIDs(): List<String> =
        vendorsList.map { it.id }

    fun getEnabledVendorsIDs(): List<String> =
        vendorConsents?.keys?.toList() ?: emptyList()

    fun getDisabledVendorsIDs(): List<String> {
        val allVendors = getAllVendorsIDs().toSet()
        val enabledVendors = getEnabledVendorsIDs().toSet()
        return (allVendors - enabledVendors).toList()
    }

    fun hasUserChoice(): Boolean =
        !consentString.isNullOrEmpty()

    companion object {
        private val json = Json { ignoreUnknownKeys = true }

        @OptIn(ExperimentalSerializationApi::class)
        fun fromJson(jsonString: String): CMPConsentModel =
            json.decodeFromString(jsonString)

        @OptIn(ExperimentalSerializationApi::class)
        fun toJson(consentModel: CMPConsentModel): String =
            json.encodeToString(consentModel)

        fun getEmptyConsent(): CMPConsentModel =
            CMPConsentModel()
    }
}

@Keep
@Serializable
data class GPPData(
    val applicableSections: List<Int>? = null,
    val cmpDisplayStatus: String? = null,
    val cmpId: Int? = null,
    val cmpStatus: String? = null,
    val gppString: String? = null,
    val gppVersions: String? = null,
    val parsedSections: Map<String, String>? = null,
    val sectionList: List<String>? = null,
    val signalStatus: String? = null,
    val supportedAPIs: List<String>? = null
)

@Keep
@Serializable
data class Purpose(
    val id: String,
    val name: String
)

@Keep
@Serializable
data class CmpMetadata(
    val name: String,
    val type: String,
    val value: String
)

@Keep
@Serializable
data class Vendor(
    val googleid: Int,
    val iabid: Int,
    val id: String,
    val name: String,
    val purposes: List<String>,
    val systemid: String
)

@Keep
@Serializable
enum class ConsentType {
    @SerialName("analytics_storage") ANALYTICS_STORAGE,
    @SerialName("ad_storage") AD_STORAGE,
    @SerialName("ad_user_data") AD_USER_DATA,
    @SerialName("ad_personalization") AD_PERSONALIZATION
}

@Keep
@Serializable
enum class ConsentStatus {
    GRANTED,
    DENIED
}