package net.codinux.i18n

import kotlin.String

/**
 * @param isoCode Lowercase alpha-2 two-letter or alpha-3 three-letter ISO 639 language code.
 * @param englishName English name of the language.
 * @param localName The endonym as speakers of this language refer themselves to their language.
 * @param defaultRegion Language's default region.
 * @param defaultScript Language's default script (writing system).
 */
enum class Language(
  val isoCode: String,
  val englishName: String,
  val localName: String?,
  val defaultRegion: Region?,
  val defaultScript: Script?,
) {
  Afar("aa", "Afar", null, Region.Ethiopia, Script.Latin),
  Abkhazian("ab", "Abkhazian", "Аԥсшәа", Region.Georgia, Script.Cyrillic),
  Acehnese("ace", "Acehnese", null, Region.Indonesia, Script.Latin),
  Acoli("ach", "Acoli", null, Region.Uganda, Script.Latin),
  Adangme("ada", "Adangme", null, Region.Ghana, Script.Latin),
  Adyghe("ady", "Adyghe", null, Region.Russia, Script.Cyrillic),
  Avestan("ae", "Avestan", null, Region.Iran, Script.Avestan),
  TunisianArabic("aeb", "Tunisian Arabic", null, Region.Tunisia, Script.Arabic),
  Afrikaans("af", "Afrikaans", "Afrikaans", Region.SouthAfrica, Script.Latin),
  Afrihili("afh", "Afrihili", null, Region.Ghana, Script.Latin),
  Aghem("agq", "Aghem", "Aghem", Region.Cameroon, Script.Latin),
  Ainu("ain", "Ainu", null, Region.Japan, Script.Katakana),
  SouthernLevantineArabic("ajp", "Southern Levantine Arabic", null, null, null),
  Akan("ak", "Akan", "Akan", Region.Ghana, Script.Latin),
  Akkadian("akk", "Akkadian", null, Region.Iraq, Script.SumeroAkkadianCuneiform),
  Alabama("akz", "Alabama", null, Region.UnitedStates, Script.Latin),
  Aleut("ale", "Aleut", null, Region.UnitedStates, Script.Latin),
  GhegAlbanian("aln", "Gheg Albanian", null, Region.Kosovo, Script.Latin),
  SouthernAltai("alt", "Southern Altai", null, Region.Russia, Script.Cyrillic),
  Amharic("am", "Amharic", "አማርኛ", Region.Ethiopia, Script.Ethiopic),
  Aragonese("an", "Aragonese", "aragonés", Region.Spain, Script.Latin),
  OldEnglish("ang", "Old English", null, Region.UnitedKingdom, Script.Latin),
  Obolo("ann", "Obolo", "Obolo", Region.Nigeria, Script.Latin),
  Angika("anp", "Angika", null, Region.India, Script.Devanagari),
  LevantineArabic("apc", "Levantine Arabic", "العامية", Region.Syria, Script.Arabic),
  Arabic("ar", "Arabic", "العربية", Region.Egypt, Script.Arabic),
  ModernStandardArabic("ar-001", "Modern Standard Arabic", "العربية الفصحى الحديثة", null, null),
  Aramaic("arc", "Aramaic", null, Region.Iran, Script.ImperialAramaic),
  Mapuche("arn", "Mapuche", "Mapudungun", Region.Chile, Script.Latin),
  Araona("aro", "Araona", null, Region.Bolivia, Script.Latin),
  Arapaho("arp", "Arapaho", null, Region.UnitedStates, Script.Latin),
  AlgerianArabic("arq", "Algerian Arabic", null, Region.Algeria, Script.Arabic),
  NajdiArabic("ars", "Najdi Arabic", null, Region.SaudiArabia, Script.Arabic),
  ArabicNajdi("ars-alt-menu", "Arabic, Najdi", null, null, null),
  Arawak("arw", "Arawak", null, Region.Suriname, Script.Latin),
  MoroccanArabic("ary", "Moroccan Arabic", null, Region.Morocco, Script.Arabic),
  EgyptianArabic("arz", "Egyptian Arabic", null, Region.Egypt, Script.Arabic),
  Assamese("as", "Assamese", "অসমীয়া", Region.India, Script.Bangla),
  Asu("asa", "Asu", "Kipare", Region.Tanzania, Script.Latin),
  AmericanSignLanguage("ase", "American Sign Language", null, Region.UnitedStates, Script.SignWriting),
  Asturian("ast", "Asturian", "asturianu", Region.Spain, Script.Latin),
  Atikamekw("atj", "Atikamekw", null, Region.Canada, Script.Latin),
  Avaric("av", "Avaric", null, Region.Russia, Script.Cyrillic),
  Kotava("avk", "Kotava", null, Region.World, Script.Latin),
  Awadhi("awa", "Awadhi", null, Region.India, Script.Devanagari),
  Aymara("ay", "Aymara", null, Region.Bolivia, Script.Latin),
  Azerbaijani("az", "Azerbaijani", "azərbaycan", Region.Azerbaijan, Script.Latin),
  Azeri("az-alt-short", "Azeri", "azəri", null, null),
  Bashkir("ba", "Bashkir", "башҡорт теле", Region.Russia, Script.Cyrillic),
  Baluchi("bal", "Baluchi", "بلۆچی", Region.Pakistan, Script.Arabic),
  Balinese("ban", "Balinese", null, Region.Indonesia, Script.Latin),
  Bavarian("bar", "Bavarian", null, Region.Austria, Script.Latin),
  Basaa("bas", "Basaa", "Ɓàsàa", Region.Cameroon, Script.Latin),
  Bamun("bax", "Bamun", null, Region.Cameroon, Script.Bamum),
  BatakToba("bbc", "Batak Toba", null, Region.Indonesia, Script.Latin),
  Ghomala("bbj", "Ghomala", null, Region.Cameroon, Script.Latin),
  Belarusian("be", "Belarusian", "беларуская", Region.Belarus, Script.Cyrillic),
  Beja("bej", "Beja", null, Region.Sudan, Script.Arabic),
  Bemba("bem", "Bemba", "Ichibemba", Region.Zambia, Script.Latin),
  Betawi("bew", "Betawi", null, Region.Indonesia, Script.Latin),
  Bena("bez", "Bena", "Hibena", Region.Tanzania, Script.Latin),
  Bafut("bfd", "Bafut", null, Region.Cameroon, Script.Latin),
  Badaga("bfq", "Badaga", null, Region.India, Script.Tamil),
  Bulgarian("bg", "Bulgarian", "български", Region.Bulgaria, Script.Cyrillic),
  Haryanvi("bgc", "Haryanvi", "हरियाणवी", Region.India, Script.Devanagari),
  WesternBalochi("bgn", "Western Balochi", "بلوچی (رخشانی)", Region.Pakistan, Script.Arabic),
  Bhojpuri("bho", "Bhojpuri", "भोजपुरी", Region.India, Script.Devanagari),
  Bislama("bi", "Bislama", null, Region.Vanuatu, Script.Latin),
  Bikol("bik", "Bikol", null, Region.Philippines, Script.Latin),
  Bini("bin", "Bini", null, Region.Nigeria, Script.Latin),
  Banjar("bjn", "Banjar", null, Region.Indonesia, Script.Latin),
  Kom("bkm", "Kom", null, Region.Cameroon, Script.Latin),
  Siksika("bla", "Siksiká", null, Region.Canada, Script.Latin),
  Anii("blo", "Anii", "anii kagɩja", Region.Benin, Script.Latin),
  TaiDam("blt", "Tai Dam", "ꪼꪕꪒꪾ", Region.Vietnam, Script.TaiViet),
  Bambara("bm", "Bambara", "bamanakan", Region.Mali, Script.Latin),
  Bangla("bn", "Bangla", "বাংলা", Region.Bangladesh, Script.Bangla),
  Tibetan("bo", "Tibetan", "བོད་སྐད་", Region.China, Script.Tibetan),
  Bishnupriya("bpy", "Bishnupriya", null, Region.India, Script.Bangla),
  Bakhtiari("bqi", "Bakhtiari", null, Region.Iran, Script.Arabic),
  Breton("br", "Breton", "brezhoneg", Region.France, Script.Latin),
  Braj("bra", "Braj", null, Region.India, Script.Devanagari),
  Brahui("brh", "Brahui", null, Region.Pakistan, Script.Arabic),
  Bodo("brx", "Bodo", "बर’", Region.India, Script.Devanagari),
  Bosnian("bs", "Bosnian", "bosanski", Region.BosniaAndHerzegovina, Script.Latin),
  Akoose("bss", "Akoose", "Akoose", Region.Cameroon, Script.Latin),
  Buriat("bua", "Buriat", null, Region.Russia, Script.Cyrillic),
  Buginese("bug", "Buginese", null, Region.Indonesia, Script.Latin),
  Bulu("bum", "Bulu", null, Region.Cameroon, Script.Latin),
  Blin("byn", "Blin", null, Region.Eritrea, Script.Ethiopic),
  Medumba("byv", "Medumba", null, Region.Cameroon, Script.Latin),
  Catalan("ca", "Catalan", "català", Region.Spain, Script.Latin),
  Caddo("cad", "Caddo", null, Region.UnitedStates, Script.Latin),
  Carib("car", "Carib", null, Region.Venezuela, Script.Latin),
  Cayuga("cay", "Cayuga", null, Region.Canada, Script.Latin),
  Atsam("cch", "Atsam", "Atsam", Region.Nigeria, Script.Latin),
  Chakma("ccp", "Chakma", "𑄌𑄋𑄴𑄟𑄳𑄦", Region.Bangladesh, Script.Chakma),
  Chechen("ce", "Chechen", "нохчийн", Region.Russia, Script.Cyrillic),
  Cebuano("ceb", "Cebuano", "Cebuano", Region.Philippines, Script.Latin),
  Chiga("cgg", "Chiga", "Rukiga", Region.Uganda, Script.Latin),
  Chamorro("ch", "Chamorro", null, Region.Guam, Script.Latin),
  Chibcha("chb", "Chibcha", null, Region.Colombia, Script.Latin),
  Chagatai("chg", "Chagatai", null, Region.Turkmenistan, Script.Arabic),
  Chuukese("chk", "Chuukese", null, Region.Micronesia, Script.Latin),
  Mari("chm", "Mari", null, Region.Russia, Script.Cyrillic),
  ChinookJargon("chn", "Chinook Jargon", null, Region.UnitedStates, Script.Latin),
  Choctaw("cho", "Choctaw", "Chahta", Region.UnitedStates, Script.Latin),
  Chipewyan("chp", "Chipewyan", null, Region.Canada, Script.Latin),
  Cherokee("chr", "Cherokee", "ᏣᎳᎩ", Region.UnitedStates, Script.Cherokee),
  Cheyenne("chy", "Cheyenne", null, Region.UnitedStates, Script.Latin),
  Chickasaw("cic", "Chickasaw", "Chikashshanompaʼ", Region.UnitedStates, Script.Latin),
  CentralKurdish("ckb", "Central Kurdish", "کوردیی ناوەندی", Region.Iraq, Script.Arabic),
  KurdishCentral("ckb-alt-menu", "Kurdish, Central", "کوردیی ناوەندی", null, null),
  KurdishSorani("ckb-alt-variant", "Kurdish, Sorani", "کوردیی ناوەندی", null, null),
  Chilcotin("clc", "Chilcotin", null, Region.Canada, Script.Latin),
  Corsican("co", "Corsican", "corsu", Region.France, Script.Latin),
  Coptic("cop", "Coptic", null, Region.Egypt, Script.Coptic),
  Capiznon("cps", "Capiznon", null, Region.Philippines, Script.Latin),
  Cree("cr", "Cree", null, Region.Canada, Script.UnifiedCanadianAboriginalSyllabics),
  Michif("crg", "Michif", null, Region.Canada, Script.Latin),
  CrimeanTatar("crh", "Crimean Tatar", null, Region.Ukraine, Script.Cyrillic),
  SouthernEastCree("crj", "Southern East Cree", null, Region.Canada, Script.UnifiedCanadianAboriginalSyllabics),
  PlainsCree("crk", "Plains Cree", null, Region.Canada, Script.UnifiedCanadianAboriginalSyllabics),
  NorthernEastCree("crl", "Northern East Cree", null, Region.Canada, Script.UnifiedCanadianAboriginalSyllabics),
  MooseCree("crm", "Moose Cree", null, Region.Canada, Script.UnifiedCanadianAboriginalSyllabics),
  CarolinaAlgonquian("crr", "Carolina Algonquian", null, null, null),
  SeselwaCreoleFrench("crs", "Seselwa Creole French", null, Region.Seychelles, Script.Latin),
  Czech("cs", "Czech", "čeština", Region.Czechia, Script.Latin),
  Kashubian("csb", "Kashubian", null, Region.Poland, Script.Latin),
  SwampyCree("csw", "Swampy Cree", "ᓀᐦᐃᓇᐍᐏᐣ", Region.Canada, Script.UnifiedCanadianAboriginalSyllabics),
  ChurchSlavic("cu", "Church Slavic", null, Region.Russia, Script.Cyrillic),
  Chuvash("cv", "Chuvash", "чӑваш", Region.Russia, Script.Cyrillic),
  WoodsCree("cwd", "Woods Cree", null, null, null),
  Welsh("cy", "Welsh", "Cymraeg", Region.UnitedKingdom, Script.Latin),
  Danish("da", "Danish", "dansk", Region.Denmark, Script.Latin),
  Dakota("dak", "Dakota", null, Region.UnitedStates, Script.Latin),
  Dargwa("dar", "Dargwa", null, Region.Russia, Script.Cyrillic),
  Taita("dav", "Taita", "Kitaita", Region.Kenya, Script.Latin),
  German("de", "German", "Deutsch", Region.Germany, Script.Latin),
  AustrianGerman("de-AT", "Austrian German", "Österreichisches Deutsch", null, null),
  SwissHighGerman("de-CH", "Swiss High German", "Schweizer Hochdeutsch", null, null),
  Delaware("del", "Delaware", null, Region.UnitedStates, Script.Latin),
  Slave("den", "Slave", null, Region.Canada, Script.Latin),
  Dogrib("dgr", "Dogrib", null, Region.Canada, Script.Latin),
  Dinka("din", "Dinka", null, Region.SouthSudan, Script.Latin),
  Zarma("dje", "Zarma", "Zarmaciine", Region.Niger, Script.Latin),
  Dogri("doi", "Dogri", "डोगरी", Region.India, Script.Devanagari),
  LowerSorbian("dsb", "Lower Sorbian", "dolnoserbšćina", Region.Germany, Script.Latin),
  CentralDusun("dtp", "Central Dusun", null, Region.Malaysia, Script.Latin),
  Duala("dua", "Duala", "duálá", Region.Cameroon, Script.Latin),
  MiddleDutch("dum", "Middle Dutch", null, Region.Netherlands, Script.Latin),
  Divehi("dv", "Divehi", null, Region.Maldives, Script.Thaana),
  JolaFonyi("dyo", "Jola-Fonyi", "joola", Region.Senegal, Script.Latin),
  Dyula("dyu", "Dyula", null, Region.BurkinaFaso, Script.Latin),
  Dzongkha("dz", "Dzongkha", "རྫོང་ཁ", Region.Bhutan, Script.Tibetan),
  Dazaga("dzg", "Dazaga", null, Region.Chad, Script.Latin),
  Embu("ebu", "Embu", "Kĩembu", Region.Kenya, Script.Latin),
  Ewe("ee", "Ewe", "eʋegbe", Region.Ghana, Script.Latin),
  Efik("efi", "Efik", null, Region.Nigeria, Script.Latin),
  Emilian("egl", "Emilian", null, Region.Italy, Script.Latin),
  AncientEgyptian("egy", "Ancient Egyptian", null, Region.Egypt, Script.EgyptianHieroglyphs),
  Ekajuk("eka", "Ekajuk", null, Region.Nigeria, Script.Latin),
  Greek("el", "Greek", "Ελληνικά", Region.Greece, Script.Greek),
  Elamite("elx", "Elamite", null, null, null),
  English("en", "English", "English", Region.UnitedStates, Script.Latin),
  AustralianEnglish("en-AU", "Australian English", "Australian English", null, null),
  CanadianEnglish("en-CA", "Canadian English", "Canadian English", null, null),
  BritishEnglish("en-GB", "British English", "British English", null, null),
  UKEnglish("en-GB-alt-short", "UK English", "UK English", null, null),
  AmericanEnglish("en-US", "American English", "American English", null, null),
  USEnglish("en-US-alt-short", "US English", "US English", null, null),
  MiddleEnglish("enm", "Middle English", null, Region.UnitedKingdom, Script.Latin),
  Esperanto("eo", "Esperanto", "Esperanto", Region.World, Script.Latin),
  Spanish("es", "Spanish", "español", Region.Spain, Script.Latin),
  LatinAmericanSpanish("es-419", "Latin American Spanish", "español latinoamericano", null, null),
  EuropeanSpanish("es-ES", "European Spanish", "español de España", null, null),
  MexicanSpanish("es-MX", "Mexican Spanish", "español de México", null, null),
  CentralYupik("esu", "Central Yupik", null, Region.UnitedStates, Script.Latin),
  Estonian("et", "Estonian", "eesti", Region.Estonia, Script.Latin),
  Basque("eu", "Basque", "euskara", Region.Spain, Script.Latin),
  Ewondo("ewo", "Ewondo", "ewondo", Region.Cameroon, Script.Latin),
  Extremaduran("ext", "Extremaduran", null, Region.Spain, Script.Latin),
  Persian("fa", "Persian", "فارسی", Region.Iran, Script.Arabic),
  Dari("fa-AF", "Dari", "دری", null, null),
  Fang("fan", "Fang", null, Region.EquatorialGuinea, Script.Latin),
  Fanti("fat", "Fanti", null, null, null),
  Fula("ff", "Fula", "Pulaar", Region.Senegal, Script.Latin),
  Finnish("fi", "Finnish", "suomi", Region.Finland, Script.Latin),
  Filipino("fil", "Filipino", "Filipino", Region.Philippines, Script.Latin),
  TornedalenFinnish("fit", "Tornedalen Finnish", null, Region.Sweden, Script.Latin),
  Fijian("fj", "Fijian", null, Region.Fiji, Script.Latin),
  Faroese("fo", "Faroese", "føroyskt", Region.FaroeIslands, Script.Latin),
  Fon("fon", "Fon", null, Region.Benin, Script.Latin),
  French("fr", "French", "français", Region.France, Script.Latin),
  CanadianFrench("fr-CA", "Canadian French", "français canadien", null, null),
  SwissFrench("fr-CH", "Swiss French", "français suisse", null, null),
  CajunFrench("frc", "Cajun French", null, Region.UnitedStates, Script.Latin),
  MiddleFrench("frm", "Middle French", null, Region.France, Script.Latin),
  OldFrench("fro", "Old French", null, Region.France, Script.Latin),
  Arpitan("frp", "Arpitan", null, Region.France, Script.Latin),
  NorthernFrisian("frr", "Northern Frisian", "Nordfriisk", Region.Germany, Script.Latin),
  EasternFrisian("frs", "Eastern Frisian", null, Region.Germany, Script.Latin),
  Friulian("fur", "Friulian", "furlan", Region.Italy, Script.Latin),
  WesternFrisian("fy", "Western Frisian", "Frysk", Region.Netherlands, Script.Latin),
  Irish("ga", "Irish", "Gaeilge", Region.Ireland, Script.Latin),
  Ga("gaa", "Ga", "Gã", Region.Ghana, Script.Latin),
  Gagauz("gag", "Gagauz", null, Region.Moldova, Script.Latin),
  GanChinese("gan", "Gan Chinese", null, Region.China, Script.Simplified),
  Gayo("gay", "Gayo", null, Region.Indonesia, Script.Latin),
  Gbaya("gba", "Gbaya", null, Region.CentralAfricanRepublic, Script.Latin),
  ZoroastrianDari("gbz", "Zoroastrian Dari", null, Region.Iran, Script.Arabic),
  ScottishGaelic("gd", "Scottish Gaelic", "Gàidhlig", Region.UnitedKingdom, Script.Latin),
  Geez("gez", "Geez", null, Region.Ethiopia, Script.Ethiopic),
  Gilbertese("gil", "Gilbertese", null, Region.Kiribati, Script.Latin),
  Galician("gl", "Galician", "galego", Region.Spain, Script.Latin),
  Gilaki("glk", "Gilaki", null, Region.Iran, Script.Arabic),
  MiddleHighGerman("gmh", "Middle High German", null, Region.Germany, Script.Latin),
  Guarani("gn", "Guarani", "avañe’ẽ", Region.Paraguay, Script.Latin),
  OldHighGerman("goh", "Old High German", null, Region.Germany, Script.Latin),
  Gondi("gon", "Gondi", null, Region.India, Script.Devanagari),
  Gorontalo("gor", "Gorontalo", null, Region.Indonesia, Script.Latin),
  Gothic("got", "Gothic", null, Region.Ukraine, Script.Gothic),
  Grebo("grb", "Grebo", null, Region.Liberia, Script.Latin),
  AncientGreek("grc", "Ancient Greek", null, Region.Cyprus, Script.Cypriot),
  SwissGerman("gsw", "Swiss German", "Schwiizertüütsch", Region.Switzerland, Script.Latin),
  Gujarati("gu", "Gujarati", "ગુજરાતી", Region.India, Script.Gujarati),
  Wayuu("guc", "Wayuu", null, Region.Colombia, Script.Latin),
  Frafra("gur", "Frafra", null, Region.Ghana, Script.Latin),
  Gusii("guz", "Gusii", "Ekegusii", Region.Kenya, Script.Latin),
  Manx("gv", "Manx", "Gaelg", Region.IsleOfMan, Script.Latin),
  Gwichin("gwi", "Gwichʼin", null, Region.Canada, Script.Latin),
  Hausa("ha", "Hausa", "Hausa", Region.Nigeria, Script.Latin),
  Haida("hai", "Haida", null, Region.Canada, Script.Latin),
  HakkaChinese("hak", "Hakka Chinese", null, Region.China, Script.Simplified),
  Hawaiian("haw", "Hawaiian", "ʻŌlelo Hawaiʻi", Region.UnitedStates, Script.Latin),
  SouthernHaida("hax", "Southern Haida", null, Region.Canada, Script.Latin),
  NorthernHaida("hdn", "Northern Haida", null, null, null),
  Hebrew("he", "Hebrew", "עברית", Region.Israel, Script.Hebrew),
  Hindi("hi", "Hindi", "हिन्दी", Region.India, Script.Devanagari),
  HindiLatin("hi-Latn", "Hindi (Latin)", "Hindi (Latin)", null, null),
  Hinglish("hi-Latn-alt-variant", "Hinglish", "Hinglish", null, null),
  FijiHindi("hif", "Fiji Hindi", null, Region.Fiji, Script.Devanagari),
  Hiligaynon("hil", "Hiligaynon", null, Region.Philippines, Script.Latin),
  Hittite("hit", "Hittite", null, Region.Turkiye, Script.SumeroAkkadianCuneiform),
  Hmong("hmn", "Hmong", null, Region.China, Script.Latin),
  HmongNjua("hnj", "Hmong Njua", "𞄀𞄄𞄰𞄩𞄍𞄜𞄰", Region.UnitedStates, Script.NyiakengPuachueHmong),
  HiriMotu("ho", "Hiri Motu", null, Region.PapuaNewGuinea, Script.Latin),
  Croatian("hr", "Croatian", "hrvatski", Region.Croatia, Script.Latin),
  UpperSorbian("hsb", "Upper Sorbian", "hornjoserbšćina", Region.Germany, Script.Latin),
  XiangChinese("hsn", "Xiang Chinese", null, Region.China, Script.Simplified),
  HaitianCreole("ht", "Haitian Creole", null, Region.Haiti, Script.Latin),
  Hungarian("hu", "Hungarian", "magyar", Region.Hungary, Script.Latin),
  Hupa("hup", "Hupa", null, Region.UnitedStates, Script.Latin),
  Halkomelem("hur", "Halkomelem", null, Region.Canada, Script.Latin),
  Armenian("hy", "Armenian", "հայերեն", Region.Armenia, Script.Armenian),
  Herero("hz", "Herero", null, Region.Namibia, Script.Latin),
  Interlingua("ia", "Interlingua", "interlingua", Region.World, Script.Latin),
  Iban("iba", "Iban", null, Region.Malaysia, Script.Latin),
  Ibibio("ibb", "Ibibio", null, Region.Nigeria, Script.Latin),
  Indonesian("id", "Indonesian", "Indonesia", Region.Indonesia, Script.Latin),
  Interlingue("ie", "Interlingue", "Interlingue", Region.Estonia, Script.Latin),
  Igbo("ig", "Igbo", "Igbo", Region.Nigeria, Script.Latin),
  SichuanYi("ii", "Sichuan Yi", "ꆈꌠꉙ", Region.China, Script.Yi),
  Inupiaq("ik", "Inupiaq", null, Region.UnitedStates, Script.Latin),
  EasternCanadianInuktitut("ike", "Eastern Canadian Inuktitut", null, null, null),
  WesternCanadianInuktitut("ikt", "Western Canadian Inuktitut", null, Region.Canada, Script.Latin),
  Iloko("ilo", "Iloko", null, Region.Philippines, Script.Latin),
  Ingush("inh", "Ingush", null, Region.Russia, Script.Cyrillic),
  Ido("io", "Ido", "Ido", Region.World, Script.Latin),
  Icelandic("is", "Icelandic", "íslenska", Region.Iceland, Script.Latin),
  Italian("it", "Italian", "italiano", Region.Italy, Script.Latin),
  Inuktitut("iu", "Inuktitut", null, Region.Canada, Script.UnifiedCanadianAboriginalSyllabics),
  Ingrian("izh", "Ingrian", null, Region.Russia, Script.Latin),
  Japanese("ja", "Japanese", "日本語", Region.Japan, Script.Japanese),
  JamaicanCreoleEnglish("jam", "Jamaican Creole English", null, Region.Jamaica, Script.Latin),
  Lojban("jbo", "Lojban", "la .lojban.", Region.World, Script.Latin),
  Ngomba("jgo", "Ngomba", "Ndaꞌa", Region.Cameroon, Script.Latin),
  Machame("jmc", "Machame", "Kimachame", Region.Tanzania, Script.Latin),
  JudeoPersian("jpr", "Judeo-Persian", null, Region.Israel, Script.Hebrew),
  JudeoArabic("jrb", "Judeo-Arabic", null, Region.Israel, Script.Hebrew),
  Jutish("jut", "Jutish", null, Region.Denmark, Script.Latin),
  Javanese("jv", "Javanese", "Jawa", Region.Indonesia, Script.Latin),
  Georgian("ka", "Georgian", "ქართული", Region.Georgia, Script.Georgian),
  KaraKalpak("kaa", "Kara-Kalpak", "қарақалпақша", Region.Uzbekistan, Script.Cyrillic),
  Kabyle("kab", "Kabyle", "Taqbaylit", Region.Algeria, Script.Latin),
  Kachin("kac", "Kachin", null, Region.MyanmarBurma, Script.Latin),
  Jju("kaj", "Jju", "Kaje", Region.Nigeria, Script.Latin),
  Kamba("kam", "Kamba", "Kikamba", Region.Kenya, Script.Latin),
  Kawi("kaw", "Kawi", null, Region.Indonesia, Script.Kawi),
  Kabardian("kbd", "Kabardian", null, Region.Russia, Script.Cyrillic),
  Kanembu("kbl", "Kanembu", null, Region.Chad, Script.Latin),
  Tyap("kcg", "Tyap", "Katab", Region.Nigeria, Script.Latin),
  Makonde("kde", "Makonde", "Chimakonde", Region.Tanzania, Script.Latin),
  Kabuverdianu("kea", "Kabuverdianu", "kabuverdianu", Region.CapeVerde, Script.Latin),
  Kenyang("ken", "Kenyang", "Kɛnyaŋ", Region.Cameroon, Script.Latin),
  Koro("kfo", "Koro", null, Region.CoteDIvoire, Script.Latin),
  Kongo("kg", "Kongo", null, Region.Congo_DemocraticRepublic, Script.Latin),
  Kaingang("kgp", "Kaingang", "kanhgág", Region.Brazil, Script.Latin),
  Khasi("kha", "Khasi", null, Region.India, Script.Latin),
  Khotanese("kho", "Khotanese", null, Region.Iran, Script.Brahmi),
  KoyraChiini("khq", "Koyra Chiini", "Koyra ciini", Region.Mali, Script.Latin),
  Khowar("khw", "Khowar", null, Region.Pakistan, Script.Arabic),
  Kikuyu("ki", "Kikuyu", "Gikuyu", Region.Kenya, Script.Latin),
  Kirmanjki("kiu", "Kirmanjki", null, Region.Turkiye, Script.Latin),
  Kuanyama("kj", "Kuanyama", null, Region.Namibia, Script.Latin),
  Kazakh("kk", "Kazakh", "қазақ тілі", Region.Kazakhstan, Script.Cyrillic),
  Kako("kkj", "Kako", "kakɔ", Region.Cameroon, Script.Latin),
  Kalaallisut("kl", "Kalaallisut", "kalaallisut", Region.Greenland, Script.Latin),
  Kalenjin("kln", "Kalenjin", "Kalenjin", Region.Kenya, Script.Latin),
  Khmer("km", "Khmer", "ខ្មែរ", Region.Cambodia, Script.Khmer),
  Kimbundu("kmb", "Kimbundu", null, Region.Angola, Script.Latin),
  Kannada("kn", "Kannada", "ಕನ್ನಡ", Region.India, Script.Kannada),
  Korean("ko", "Korean", "한국어", Region.SouthKorea, Script.Korean),
  KomiPermyak("koi", "Komi-Permyak", null, Region.Russia, Script.Cyrillic),
  Konkani("kok", "Konkani", "कोंकणी", Region.India, Script.Devanagari),
  Kosraean("kos", "Kosraean", null, Region.Micronesia, Script.Latin),
  Kpelle("kpe", "Kpelle", "Kpɛlɛɛ", Region.Liberia, Script.Latin),
  Kanuri("kr", "Kanuri", null, Region.Nigeria, Script.Latin),
  KarachayBalkar("krc", "Karachay-Balkar", null, Region.Russia, Script.Cyrillic),
  Krio("kri", "Krio", null, Region.SierraLeone, Script.Latin),
  KinarayA("krj", "Kinaray-a", null, Region.Philippines, Script.Latin),
  Karelian("krl", "Karelian", null, Region.Russia, Script.Latin),
  Kurukh("kru", "Kurukh", null, Region.India, Script.Devanagari),
  Kashmiri("ks", "Kashmiri", "کٲشُر", Region.India, Script.Arabic),
  Shambala("ksb", "Shambala", "Kishambaa", Region.Tanzania, Script.Latin),
  Bafia("ksf", "Bafia", "rikpa", Region.Cameroon, Script.Latin),
  Colognian("ksh", "Colognian", "Kölsch", Region.Germany, Script.Latin),
  Kurdish("ku", "Kurdish", "kurdî (kurmancî)", Region.Turkiye, Script.Latin),
  Kumyk("kum", "Kumyk", null, Region.Russia, Script.Cyrillic),
  Kutenai("kut", "Kutenai", null, Region.Canada, Script.Latin),
  Komi("kv", "Komi", null, Region.Russia, Script.Cyrillic),
  Cornish("kw", "Cornish", "kernewek", Region.UnitedKingdom, Script.Latin),
  Kwakwala("kwk", "Kwakʼwala", null, Region.Canada, Script.Latin),
  Kuvi("kxv", "Kuvi", "kuvi", Region.India, Script.Latin),
  Kyrgyz("ky", "Kyrgyz", "кыргызча", Region.Kyrgyzstan, Script.Cyrillic),
  Kirghiz("ky-alt-variant", "Kirghiz", null, null, null),
  Latin("la", "Latin", null, Region.VaticanCity, Script.Latin),
  Ladino("lad", "Ladino", null, Region.Israel, Script.Hebrew),
  Langi("lag", "Langi", "Kɨlaangi", Region.Tanzania, Script.Latin),
  WesternPanjabi("lah", "Western Panjabi", null, Region.Pakistan, Script.Arabic),
  Lamba("lam", "Lamba", null, Region.Zambia, Script.Latin),
  Luxembourgish("lb", "Luxembourgish", "Lëtzebuergesch", Region.Luxembourg, Script.Latin),
  Lezghian("lez", "Lezghian", null, Region.Russia, Script.Cyrillic),
  LinguaFrancaNova("lfn", "Lingua Franca Nova", null, Region.World, Script.Latin),
  Ganda("lg", "Ganda", "Luganda", Region.Uganda, Script.Latin),
  Limburgish("li", "Limburgish", null, Region.Netherlands, Script.Latin),
  Ligurian("lij", "Ligurian", "ligure", Region.Italy, Script.Latin),
  Lillooet("lil", "Lillooet", null, Region.Canada, Script.Latin),
  Livonian("liv", "Livonian", null, Region.Latvia, Script.Latin),
  Lakota("lkt", "Lakota", "Lakȟólʼiyapi", Region.UnitedStates, Script.Latin),
  Ladin("lld", "Ladin", "ladin", Region.Italy, Script.Latin),
  Lombard("lmo", "Lombard", "Lombard", Region.Italy, Script.Latin),
  Lingala("ln", "Lingala", "lingála", Region.Congo_DemocraticRepublic, Script.Latin),
  Lao("lo", "Lao", "ລາວ", Region.Laos, Script.Lao),
  Mongo("lol", "Mongo", null, Region.Congo_DemocraticRepublic, Script.Latin),
  LouisianaCreole("lou", "Louisiana Creole", null, Region.UnitedStates, Script.Latin),
  Lozi("loz", "Lozi", null, Region.Zambia, Script.Latin),
  NorthernLuri("lrc", "Northern Luri", "لۊری شومالی", Region.Iran, Script.Arabic),
  Saamia("lsm", "Saamia", null, Region.Uganda, Script.Latin),
  Lithuanian("lt", "Lithuanian", "lietuvių", Region.Lithuania, Script.Latin),
  Latgalian("ltg", "Latgalian", null, Region.Latvia, Script.Latin),
  LubaKatanga("lu", "Luba-Katanga", "Tshiluba", Region.Congo_DemocraticRepublic, Script.Latin),
  LubaLulua("lua", "Luba-Lulua", null, Region.Congo_DemocraticRepublic, Script.Latin),
  Luiseno("lui", "Luiseno", null, Region.UnitedStates, Script.Latin),
  Lunda("lun", "Lunda", null, Region.Zambia, Script.Latin),
  Luo("luo", "Luo", "Dholuo", Region.Kenya, Script.Latin),
  Mizo("lus", "Mizo", null, Region.India, Script.Latin),
  Luyia("luy", "Luyia", "Luluhia", Region.Kenya, Script.Latin),
  Latvian("lv", "Latvian", "latviešu", Region.Latvia, Script.Latin),
  LiteraryChinese("lzh", "Literary Chinese", null, Region.China, Script.Simplified),
  Laz("lzz", "Laz", null, Region.Turkiye, Script.Latin),
  Madurese("mad", "Madurese", null, Region.Indonesia, Script.Latin),
  Mafa("maf", "Mafa", null, Region.Cameroon, Script.Latin),
  Magahi("mag", "Magahi", null, Region.India, Script.Devanagari),
  Maithili("mai", "Maithili", "मैथिली", Region.India, Script.Devanagari),
  Makasar("mak", "Makasar", null, Region.Indonesia, Script.Latin),
  Mandingo("man", "Mandingo", null, Region.Gambia, Script.Latin),
  Masai("mas", "Masai", "Maa", Region.Kenya, Script.Latin),
  Maba("mde", "Maba", null, Region.Chad, Script.Arabic),
  Moksha("mdf", "Moksha", "мокшень кяль", Region.Russia, Script.Cyrillic),
  Mandar("mdr", "Mandar", null, Region.Indonesia, Script.Latin),
  Mende("men", "Mende", null, Region.SierraLeone, Script.Latin),
  Meru("mer", "Meru", "Kĩmĩrũ", Region.Kenya, Script.Latin),
  Morisyen("mfe", "Morisyen", "kreol morisien", Region.Mauritius, Script.Latin),
  Malagasy("mg", "Malagasy", "Malagasy", Region.Madagascar, Script.Latin),
  MiddleIrish("mga", "Middle Irish", null, Region.Ireland, Script.GaelicLatin),
  MakhuwaMeetto("mgh", "Makhuwa-Meetto", "Makua", Region.Mozambique, Script.Latin),
  Meta("mgo", "Metaʼ", "metaʼ", Region.Cameroon, Script.Latin),
  Marshallese("mh", "Marshallese", null, Region.MarshallIslands, Script.Latin),
  Mocheno("mhn", "Mòcheno", null, Region.Italy, Script.Latin),
  Maori("mi", "Māori", "Māori", Region.NewZealand, Script.Latin),
  Mikmaw("mic", "Mi'kmaw", "Lʼnuiʼsuti", Region.Canada, Script.Latin),
  Minangkabau("min", "Minangkabau", null, Region.Indonesia, Script.Latin),
  Macedonian("mk", "Macedonian", "македонски", Region.NorthMacedonia, Script.Cyrillic),
  Malayalam("ml", "Malayalam", "മലയാളം", Region.India, Script.Malayalam),
  Mongolian("mn", "Mongolian", "монгол", Region.Mongolia, Script.Cyrillic),
  Manchu("mnc", "Manchu", null, Region.China, Script.Mongolian),
  Manipuri("mni", "Manipuri", "মৈতৈলোন্", Region.India, Script.Bangla),
  InnuAimun("moe", "Innu-aimun", null, Region.Canada, Script.Latin),
  Mohawk("moh", "Mohawk", "Kanienʼkéha", Region.Canada, Script.Latin),
  Mossi("mos", "Mossi", null, Region.BurkinaFaso, Script.Latin),
  Marathi("mr", "Marathi", "मराठी", Region.India, Script.Devanagari),
  WesternMari("mrj", "Western Mari", null, Region.Russia, Script.Cyrillic),
  Malay("ms", "Malay", "Melayu", Region.Malaysia, Script.Latin),
  Maltese("mt", "Maltese", "Malti", Region.Malta, Script.Latin),
  Mundang("mua", "Mundang", "MUNDAŊ", Region.Cameroon, Script.Latin),
  MultipleLanguages("mul", "Multiple languages", null, null, null),
  Muscogee("mus", "Muscogee", "Mvskoke", Region.UnitedStates, Script.Latin),
  Mvskoke("mus-alt-official", "Mvskoke", null, null, null),
  Mirandese("mwl", "Mirandese", null, Region.Portugal, Script.Latin),
  Marwari("mwr", "Marwari", null, Region.India, Script.Devanagari),
  Mentawai("mwv", "Mentawai", null, Region.Indonesia, Script.Latin),
  Burmese("my", "Burmese", "မြန်မာ", Region.MyanmarBurma, Script.Myanmar),
  MyanmarLanguage("my-alt-variant", "Myanmar Language", null, null, null),
  Myene("mye", "Myene", null, Region.Gabon, Script.Latin),
  Erzya("myv", "Erzya", "эрзянь кель", Region.Russia, Script.Cyrillic),
  Mazanderani("mzn", "Mazanderani", "مازرونی", Region.Iran, Script.Arabic),
  Nauru("na", "Nauru", null, Region.Nauru, Script.Latin),
  MinNanChinese("nan", "Min Nan Chinese", null, Region.China, Script.Simplified),
  Neapolitan("nap", "Neapolitan", null, Region.Italy, Script.Latin),
  Nama("naq", "Nama", "Khoekhoegowab", Region.Namibia, Script.Latin),
  NorwegianBokmal("nb", "Norwegian Bokmål", "norsk bokmål", Region.Norway, Script.Latin),
  NorthNdebele("nd", "North Ndebele", "isiNdebele", Region.Zimbabwe, Script.Latin),
  LowGerman("nds", "Low German", "Neddersass’sch", Region.Germany, Script.Latin),
  LowSaxon("nds-NL", "Low Saxon", null, null, null),
  Nepali("ne", "Nepali", "नेपाली", Region.Nepal, Script.Devanagari),
  Newari("new", "Newari", null, Region.Nepal, Script.Devanagari),
  Ndonga("ng", "Ndonga", null, Region.Namibia, Script.Latin),
  Nias("nia", "Nias", null, Region.Indonesia, Script.Latin),
  Niuean("niu", "Niuean", null, Region.Niue, Script.Latin),
  AoNaga("njo", "Ao Naga", null, Region.India, Script.Latin),
  Dutch("nl", "Dutch", "Nederlands", Region.Netherlands, Script.Latin),
  Flemish("nl-BE", "Flemish", "Vlaams", null, null),
  Kwasio("nmg", "Kwasio", null, Region.Cameroon, Script.Latin),
  NorwegianNynorsk("nn", "Norwegian Nynorsk", "norsk nynorsk", Region.Norway, Script.Latin),
  Ngiemboon("nnh", "Ngiemboon", "Shwóŋò ngiembɔɔn", Region.Cameroon, Script.Latin),
  Norwegian("no", "Norwegian", "norsk", Region.Norway, Script.Latin),
  Nogai("nog", "Nogai", null, Region.Russia, Script.Cyrillic),
  OldNorse("non", "Old Norse", null, Region.Sweden, Script.Runic),
  Novial("nov", "Novial", null, Region.World, Script.Latin),
  NKo("nqo", "N’Ko", "ߒߞߏ", Region.Guinea, Script.NKo),
  SouthNdebele("nr", "South Ndebele", null, Region.SouthAfrica, Script.Latin),
  NorthernSotho("nso", "Northern Sotho", "Sesotho sa Leboa", Region.SouthAfrica, Script.Latin),
  Nuer("nus", "Nuer", "Thok Nath", Region.SouthSudan, Script.Latin),
  Navajo("nv", "Navajo", "Diné Bizaad", Region.UnitedStates, Script.Latin),
  ClassicalNewari("nwc", "Classical Newari", null, Region.Nepal, Script.Newa),
  Nyanja("ny", "Nyanja", "Nyanja", Region.Malawi, Script.Latin),
  Nyamwezi("nym", "Nyamwezi", null, Region.Tanzania, Script.Latin),
  Nyankole("nyn", "Nyankole", "Runyankore", Region.Uganda, Script.Latin),
  Nyoro("nyo", "Nyoro", null, Region.Uganda, Script.Latin),
  Nzima("nzi", "Nzima", null, Region.Ghana, Script.Latin),
  Occitan("oc", "Occitan", "occitan", Region.France, Script.Latin),
  Ojibwa("oj", "Ojibwa", null, Region.Canada, Script.UnifiedCanadianAboriginalSyllabics),
  NorthwesternOjibwa("ojb", "Northwestern Ojibwa", null, Region.Canada, Script.Latin),
  CentralOjibwa("ojc", "Central Ojibwa", null, Region.Canada, Script.Latin),
  EasternOjibwa("ojg", "Eastern Ojibwa", null, null, null),
  OjiCree("ojs", "Oji-Cree", null, Region.Canada, Script.UnifiedCanadianAboriginalSyllabics),
  WesternOjibwa("ojw", "Western Ojibwa", null, Region.Canada, Script.Latin),
  Okanagan("oka", "Okanagan", null, Region.Canada, Script.Latin),
  Oromo("om", "Oromo", "Oromoo", Region.Ethiopia, Script.Latin),
  Odia("or", "Odia", "ଓଡ଼ିଆ", Region.India, Script.Odia),
  Ossetic("os", "Ossetic", "ирон", Region.Georgia, Script.Cyrillic),
  Osage("osa", "Osage", "𐓏𐓘𐓻𐓘𐓻𐓟", Region.UnitedStates, Script.Osage),
  OttomanTurkish("ota", "Ottoman Turkish", null, Region.Turkiye, Script.Arabic),
  Punjabi("pa", "Punjabi", "ਪੰਜਾਬੀ", Region.India, Script.Gurmukhi),
  Pangasinan("pag", "Pangasinan", null, Region.Philippines, Script.Latin),
  Pahlavi("pal", "Pahlavi", null, Region.Iran, Script.InscriptionalPahlavi),
  Pampanga("pam", "Pampanga", null, Region.Philippines, Script.Latin),
  Papiamento("pap", "Papiamento", "Papiamentu", Region.Curacao, Script.Latin),
  Palauan("pau", "Palauan", null, Region.Palau, Script.Latin),
  Picard("pcd", "Picard", null, Region.France, Script.Latin),
  NigerianPidgin("pcm", "Nigerian Pidgin", "Naijíriá Píjin", Region.Nigeria, Script.Latin),
  PennsylvaniaGerman("pdc", "Pennsylvania German", null, Region.UnitedStates, Script.Latin),
  Plautdietsch("pdt", "Plautdietsch", null, Region.Canada, Script.Latin),
  OldPersian("peo", "Old Persian", null, Region.Iran, Script.OldPersian),
  PalatineGerman("pfl", "Palatine German", null, Region.Germany, Script.Latin),
  Phoenician("phn", "Phoenician", null, Region.Lebanon, Script.Phoenician),
  Pali("pi", "Pali", null, Region.India, Script.Sinhala),
  Pijin("pis", "Pijin", "Pijin", Region.SolomonIslands, Script.Latin),
  Polish("pl", "Polish", "polski", Region.Poland, Script.Latin),
  Piedmontese("pms", "Piedmontese", null, Region.Italy, Script.Latin),
  Pontic("pnt", "Pontic", null, Region.Greece, Script.Greek),
  Pohnpeian("pon", "Pohnpeian", null, Region.Micronesia, Script.Latin),
  MaliseetPassamaquoddy("pqm", "Maliseet-Passamaquoddy", null, Region.Canada, Script.Latin),
  Prussian("prg", "Prussian", "prūsiskan", Region.Poland, Script.Latin),
  OldProvencal("pro", "Old Provençal", null, Region.France, Script.Latin),
  Pashto("ps", "Pashto", "پښتو", Region.Afghanistan, Script.Arabic),
  Pushto("ps-alt-variant", "Pushto", null, null, null),
  Portuguese("pt", "Portuguese", "português", Region.Brazil, Script.Latin),
  BrazilianPortuguese("pt-BR", "Brazilian Portuguese", null, null, null),
  EuropeanPortuguese("pt-PT", "European Portuguese", "português europeu", null, null),
  Quechua("qu", "Quechua", "Runasimi", Region.Peru, Script.Latin),
  Kiche("quc", "Kʼicheʼ", "Kʼicheʼ", Region.Guatemala, Script.Latin),
  ChimborazoHighlandQuichua("qug", "Chimborazo Highland Quichua", null, Region.Ecuador, Script.Latin),
  Rajasthani("raj", "Rajasthani", "राजस्थानी", Region.India, Script.Devanagari),
  Rapanui("rap", "Rapanui", null, Region.Chile, Script.Latin),
  Rarotongan("rar", "Rarotongan", null, Region.CookIslands, Script.Latin),
  Romagnol("rgn", "Romagnol", null, Region.Italy, Script.Latin),
  Rohingya("rhg", "Rohingya", "𐴌𐴗𐴥𐴝𐴙𐴚𐴒𐴙𐴝", Region.MyanmarBurma, Script.Hanifi),
  Riffian("rif", "Riffian", "Tarifit", Region.Morocco, Script.Latin),
  Romansh("rm", "Romansh", "rumantsch", Region.Switzerland, Script.Latin),
  Rundi("rn", "Rundi", "Ikirundi", Region.Burundi, Script.Latin),
  Romanian("ro", "Romanian", "română", Region.Romania, Script.Latin),
  Moldavian("ro-MD", "Moldavian", null, null, null),
  Rombo("rof", "Rombo", "Kihorombo", Region.Tanzania, Script.Latin),
  Romany("rom", "Romany", null, Region.Romania, Script.Latin),
  Rotuman("rtm", "Rotuman", null, Region.Fiji, Script.Latin),
  Russian("ru", "Russian", "русский", Region.Russia, Script.Cyrillic),
  Rusyn("rue", "Rusyn", null, Region.Ukraine, Script.Cyrillic),
  Roviana("rug", "Roviana", null, Region.SolomonIslands, Script.Latin),
  Aromanian("rup", "Aromanian", null, Region.Romania, Script.Latin),
  Kinyarwanda("rw", "Kinyarwanda", "Ikinyarwanda", Region.Rwanda, Script.Latin),
  Rwa("rwk", "Rwa", "Kiruwa", Region.Tanzania, Script.Latin),
  Sanskrit("sa", "Sanskrit", "संस्कृत भाषा", Region.India, Script.Devanagari),
  Sandawe("sad", "Sandawe", null, Region.Tanzania, Script.Latin),
  Yakut("sah", "Yakut", "саха тыла", Region.Russia, Script.Cyrillic),
  SamaritanAramaic("sam", "Samaritan Aramaic", null, Region.Palestine, Script.Samaritan),
  Samburu("saq", "Samburu", "Kisampur", Region.Kenya, Script.Latin),
  Sasak("sas", "Sasak", null, Region.Indonesia, Script.Latin),
  Santali("sat", "Santali", "ᱥᱟᱱᱛᱟᱲᱤ", Region.India, Script.OlChiki),
  Saurashtra("saz", "Saurashtra", null, Region.India, Script.Saurashtra),
  Ngambay("sba", "Ngambay", null, Region.Chad, Script.Latin),
  Sangu("sbp", "Sangu", "Ishisangu", Region.Tanzania, Script.Latin),
  Sardinian("sc", "Sardinian", "sardu", Region.Italy, Script.Latin),
  Sicilian("scn", "Sicilian", "sicilianu", Region.Italy, Script.Latin),
  Scots("sco", "Scots", null, Region.UnitedKingdom, Script.Latin),
  Sindhi("sd", "Sindhi", "سنڌي", Region.Pakistan, Script.Arabic),
  SassareseSardinian("sdc", "Sassarese Sardinian", null, Region.Italy, Script.Latin),
  SouthernKurdish("sdh", "Southern Kurdish", "کوردی خوارگ", Region.Iran, Script.Arabic),
  NorthernSami("se", "Northern Sami", "davvisámegiella", Region.Norway, Script.Latin),
  SamiNorthern("se-alt-menu", "Sami, Northern", null, null, null),
  Seneca("see", "Seneca", null, Region.UnitedStates, Script.Latin),
  Sena("seh", "Sena", "sena", Region.Mozambique, Script.Latin),
  Seri("sei", "Seri", null, Region.Mexico, Script.Latin),
  Selkup("sel", "Selkup", null, Region.Russia, Script.Cyrillic),
  KoyraboroSenni("ses", "Koyraboro Senni", "Koyraboro senni", Region.Mali, Script.Latin),
  Sango("sg", "Sango", "Sängö", Region.CentralAfricanRepublic, Script.Latin),
  OldIrish("sga", "Old Irish", null, Region.Ireland, Script.Ogham),
  Samogitian("sgs", "Samogitian", null, Region.Lithuania, Script.Latin),
  SerboCroatian("sh", "Serbo-Croatian", null, null, null),
  Tachelhit("shi", "Tachelhit", "ⵜⴰⵛⵍⵃⵉⵜ", Region.Morocco, Script.Tifinagh),
  Shan("shn", "Shan", "တႆး", Region.MyanmarBurma, Script.Myanmar),
  ChadianArabic("shu", "Chadian Arabic", null, Region.Chad, Script.Arabic),
  Sinhala("si", "Sinhala", "සිංහල", Region.SriLanka, Script.Sinhala),
  Sidamo("sid", "Sidamo", null, Region.Ethiopia, Script.Latin),
  Slovak("sk", "Slovak", "slovenčina", Region.Slovakia, Script.Latin),
  Saraiki("skr", "Saraiki", "سرائیکی", Region.Pakistan, Script.Arabic),
  Slovenian("sl", "Slovenian", "slovenščina", Region.Slovenia, Script.Latin),
  SouthernLushootseed("slh", "Southern Lushootseed", null, Region.UnitedStates, Script.Latin),
  LowerSilesian("sli", "Lower Silesian", null, Region.Poland, Script.Latin),
  Selayar("sly", "Selayar", null, Region.Indonesia, Script.Latin),
  Samoan("sm", "Samoan", null, Region.Samoa, Script.Latin),
  SouthernSami("sma", "Southern Sami", "Åarjelsaemien gïele", Region.Sweden, Script.Latin),
  SamiSouthern("sma-alt-menu", "Sami, Southern", null, null, null),
  LuleSami("smj", "Lule Sami", "julevsámegiella", Region.Sweden, Script.Latin),
  SamiLule("smj-alt-menu", "Sami, Lule", null, null, null),
  InariSami("smn", "Inari Sami", "anarâškielâ", Region.Finland, Script.Latin),
  SamiInari("smn-alt-menu", "Sami, Inari", null, null, null),
  SkoltSami("sms", "Skolt Sami", "sääʹmǩiõll", Region.Finland, Script.Latin),
  SamiSkolt("sms-alt-menu", "Sami, Skolt", null, null, null),
  Shona("sn", "Shona", "chiShona", Region.Zimbabwe, Script.Latin),
  Soninke("snk", "Soninke", null, Region.Mali, Script.Latin),
  Somali("so", "Somali", "Soomaali", Region.Somalia, Script.Latin),
  Sogdien("sog", "Sogdien", null, Region.Uzbekistan, Script.Sogdian),
  Albanian("sq", "Albanian", "shqip", Region.Albania, Script.Latin),
  Serbian("sr", "Serbian", "српски", Region.Serbia, Script.Cyrillic),
  Montenegrin("sr-ME", "Montenegrin", null, null, null),
  SrananTongo("srn", "Sranan Tongo", null, Region.Suriname, Script.Latin),
  Serer("srr", "Serer", null, Region.Senegal, Script.Latin),
  Swati("ss", "Swati", "siSwati", Region.SouthAfrica, Script.Latin),
  Saho("ssy", "Saho", "Saho", Region.Eritrea, Script.Latin),
  SouthernSotho("st", "Southern Sotho", "Sesotho", Region.SouthAfrica, Script.Latin),
  SaterlandFrisian("stq", "Saterland Frisian", null, Region.Germany, Script.Latin),
  StraitsSalish("str", "Straits Salish", null, Region.Canada, Script.Latin),
  Sundanese("su", "Sundanese", "Basa Sunda", Region.Indonesia, Script.Latin),
  Sukuma("suk", "Sukuma", null, Region.Tanzania, Script.Latin),
  Susu("sus", "Susu", null, Region.Guinea, Script.Latin),
  Sumerian("sux", "Sumerian", null, null, null),
  Swedish("sv", "Swedish", "svenska", Region.Sweden, Script.Latin),
  Swahili("sw", "Swahili", "Kiswahili", Region.Tanzania, Script.Latin),
  CongoSwahili("sw-CD", "Congo Swahili", null, null, null),
  Comorian("swb", "Comorian", null, Region.Mayotte, Script.Arabic),
  ClassicalSyriac("syc", "Classical Syriac", null, Region.Turkiye, Script.Syriac),
  Syriac("syr", "Syriac", "ܣܘܪܝܝܐ", Region.Iraq, Script.Syriac),
  Silesian("szl", "Silesian", "ślōnski", Region.Poland, Script.Latin),
  Tamil("ta", "Tamil", "தமிழ்", Region.India, Script.Tamil),
  SouthernTutchone("tce", "Southern Tutchone", null, Region.Canada, Script.Latin),
  Tulu("tcy", "Tulu", null, Region.India, Script.Kannada),
  Telugu("te", "Telugu", "తెలుగు", Region.India, Script.Telugu),
  Timne("tem", "Timne", null, Region.SierraLeone, Script.Latin),
  Teso("teo", "Teso", "Kiteso", Region.Uganda, Script.Latin),
  Tereno("ter", "Tereno", null, Region.Brazil, Script.Latin),
  Tetum("tet", "Tetum", null, Region.TimorLeste, Script.Latin),
  Tajik("tg", "Tajik", "тоҷикӣ", Region.Tajikistan, Script.Cyrillic),
  Tagish("tgx", "Tagish", null, Region.Canada, Script.Latin),
  Thai("th", "Thai", "ไทย", Region.Thailand, Script.Thai),
  Tahltan("tht", "Tahltan", null, Region.Canada, Script.Latin),
  Tigrinya("ti", "Tigrinya", "ትግርኛ", Region.Ethiopia, Script.Ethiopic),
  Tigre("tig", "Tigre", null, Region.Eritrea, Script.Ethiopic),
  Tiv("tiv", "Tiv", null, Region.Nigeria, Script.Latin),
  Turkmen("tk", "Turkmen", "türkmen dili", Region.Turkmenistan, Script.Latin),
  Tokelau("tkl", "Tokelau", null, Region.Tokelau, Script.Latin),
  Tsakhur("tkr", "Tsakhur", null, Region.Azerbaijan, Script.Latin),
  Tagalog("tl", "Tagalog", null, Region.Philippines, Script.Latin),
  Klingon("tlh", "Klingon", null, null, null),
  Tlingit("tli", "Tlingit", null, Region.UnitedStates, Script.Latin),
  Talysh("tly", "Talysh", null, Region.Azerbaijan, Script.Latin),
  Tamashek("tmh", "Tamashek", null, Region.Niger, Script.Latin),
  Tswana("tn", "Tswana", "Setswana", Region.SouthAfrica, Script.Latin),
  Tongan("to", "Tongan", "lea fakatonga", Region.Tonga, Script.Latin),
  NyasaTonga("tog", "Nyasa Tonga", null, Region.Malawi, Script.Latin),
  TokiPona("tok", "Toki Pona", "toki pona", Region.World, Script.Latin),
  TokPisin("tpi", "Tok Pisin", "Tok Pisin", Region.PapuaNewGuinea, Script.Latin),
  Turkish("tr", "Turkish", "Türkçe", Region.Turkiye, Script.Latin),
  Turoyo("tru", "Turoyo", null, Region.Turkiye, Script.Latin),
  Taroko("trv", "Taroko", "patas Taroko", Region.Taiwan, Script.Latin),
  Torwali("trw", "Torwali", "توروالی", Region.Pakistan, Script.Arabic),
  Tsonga("ts", "Tsonga", null, Region.SouthAfrica, Script.Latin),
  Tsakonian("tsd", "Tsakonian", null, Region.Greece, Script.Greek),
  Tsimshian("tsi", "Tsimshian", null, Region.Canada, Script.Latin),
  Tatar("tt", "Tatar", "татар", Region.Russia, Script.Cyrillic),
  NorthernTutchone("ttm", "Northern Tutchone", null, Region.Canada, Script.Latin),
  MuslimTat("ttt", "Muslim Tat", null, Region.Azerbaijan, Script.Latin),
  Tumbuka("tum", "Tumbuka", null, Region.Malawi, Script.Latin),
  Tuvalu("tvl", "Tuvalu", null, Region.Tuvalu, Script.Latin),
  Twi("tw", "Twi", null, null, null),
  Tasawaq("twq", "Tasawaq", "Tasawaq senni", Region.Niger, Script.Latin),
  Tahitian("ty", "Tahitian", null, Region.FrenchPolynesia, Script.Latin),
  Tuvinian("tyv", "Tuvinian", null, Region.Russia, Script.Cyrillic),
  CentralAtlasTamazight("tzm", "Central Atlas Tamazight", "Tamaziɣt n laṭlaṣ", Region.Morocco, Script.Latin),
  Udmurt("udm", "Udmurt", null, Region.Russia, Script.Cyrillic),
  Uyghur("ug", "Uyghur", "ئۇيغۇرچە", Region.China, Script.Arabic),
  Uighur("ug-alt-variant", "Uighur", null, null, null),
  Ugaritic("uga", "Ugaritic", null, Region.Syria, Script.Ugaritic),
  Ukrainian("uk", "Ukrainian", "українська", Region.Ukraine, Script.Cyrillic),
  Umbundu("umb", "Umbundu", null, Region.Angola, Script.Latin),
  UnknownLanguage("und", "Unknown language", null, Region.UnitedStates, Script.Latin),
  Urdu("ur", "Urdu", "اردو", Region.Pakistan, Script.Arabic),
  Uzbek("uz", "Uzbek", "o‘zbek", Region.Uzbekistan, Script.Latin),
  Vai("vai", "Vai", "ꕙꔤ", Region.Liberia, Script.Vai),
  Venda("ve", "Venda", null, Region.SouthAfrica, Script.Latin),
  Venetian("vec", "Venetian", "veneto", Region.Italy, Script.Latin),
  Veps("vep", "Veps", null, Region.Russia, Script.Latin),
  Vietnamese("vi", "Vietnamese", "Tiếng Việt", Region.Vietnam, Script.Latin),
  WestFlemish("vls", "West Flemish", null, Region.Belgium, Script.Latin),
  MainFranconian("vmf", "Main-Franconian", null, Region.Germany, Script.Latin),
  Makhuwa("vmw", "Makhuwa", "emakhuwa", Region.Mozambique, Script.Latin),
  Volapuk("vo", "Volapük", null, Region.World, Script.Latin),
  Votic("vot", "Votic", null, Region.Russia, Script.Latin),
  Voro("vro", "Võro", null, Region.Estonia, Script.Latin),
  Vunjo("vun", "Vunjo", "Kyivunjo", Region.Tanzania, Script.Latin),
  Walloon("wa", "Walloon", "walon", Region.Belgium, Script.Latin),
  Walser("wae", "Walser", "Walser", Region.Switzerland, Script.Latin),
  Wolaytta("wal", "Wolaytta", null, Region.Ethiopia, Script.Ethiopic),
  Waray("war", "Waray", null, Region.Philippines, Script.Latin),
  Washo("was", "Washo", null, Region.UnitedStates, Script.Latin),
  Warlpiri("wbp", "Warlpiri", "Warlpiri", Region.Australia, Script.Latin),
  Wolof("wo", "Wolof", "Wolof", Region.Senegal, Script.Latin),
  WuChinese("wuu", "Wu Chinese", null, Region.China, Script.Simplified),
  Kalmyk("xal", "Kalmyk", null, Region.Russia, Script.Cyrillic),
  Xhosa("xh", "Xhosa", "IsiXhosa", Region.SouthAfrica, Script.Latin),
  Mingrelian("xmf", "Mingrelian", null, Region.Georgia, Script.Georgian),
  Kangri("xnr", "Kangri", "कांगड़ी", Region.India, Script.Devanagari),
  Soga("xog", "Soga", "Olusoga", Region.Uganda, Script.Latin),
  Yao("yao", "Yao", null, Region.Mozambique, Script.Latin),
  Yapese("yap", "Yapese", null, Region.Micronesia, Script.Latin),
  Yangben("yav", "Yangben", "nuasue", Region.Cameroon, Script.Latin),
  Yemba("ybb", "Yemba", null, Region.Cameroon, Script.Latin),
  Yiddish("yi", "Yiddish", "ייִדיש", Region.Ukraine, Script.Hebrew),
  Yoruba("yo", "Yoruba", "Èdè Yorùbá", Region.Nigeria, Script.Latin),
  Nheengatu("yrl", "Nheengatu", "nheẽgatu", Region.Brazil, Script.Latin),
  Cantonese("yue", "Cantonese", "粵語", Region.HongKong, Script.Traditional),
  ChineseCantonese("yue-alt-menu", "Chinese, Cantonese", "中文 (粵語)", null, null),
  Zhuang("za", "Zhuang", "Vahcuengh", Region.China, Script.Latin),
  Zapotec("zap", "Zapotec", null, Region.Mexico, Script.Latin),
  Blissymbols("zbl", "Blissymbols", null, null, null),
  Zeelandic("zea", "Zeelandic", null, Region.Netherlands, Script.Latin),
  Zenaga("zen", "Zenaga", null, Region.Mauritania, Script.Tifinagh),
  StandardMoroccanTamazight("zgh", "Standard Moroccan Tamazight", "ⵜⴰⵎⴰⵣⵉⵖⵜ", Region.Morocco, Script.Tifinagh),
  Chinese("zh", "Chinese", "中文", Region.China, Script.Simplified),
  SimplifiedChinese("zh-Hans", "Simplified Chinese", "简体中文", null, null),
  SimplifiedMandarinChinese("zh-Hans-alt-long", "Simplified Mandarin Chinese", "现代标准汉语（简体）", null, null),
  TraditionalChinese("zh-Hant", "Traditional Chinese", "繁體中文", null, null),
  TraditionalMandarinChinese("zh-Hant-alt-long", "Traditional Mandarin Chinese", "繁體中文", null, null),
  MandarinChinese("zh-alt-long", "Mandarin Chinese", null, null, null),
  ChineseMandarin("zh-alt-menu", "Chinese, Mandarin", "普通话", null, null),
  Zulu("zu", "Zulu", "isiZulu", Region.SouthAfrica, Script.Latin),
  Zuni("zun", "Zuni", null, Region.UnitedStates, Script.Latin),
  NoLinguisticContent("zxx", "No linguistic content", null, null, null),
  Zaza("zza", "Zaza", null, Region.Turkiye, Script.Latin),
}
