package net.codinux.i18n

import kotlin.Int
import kotlin.String
import kotlin.collections.List

/**
 * Please note, the list includes not only countries but also:
 * - Geographical regions such as [World], [EasternAfrica] or [Antarctica],
 * - Supranational unions (alliances) such as the [UnitedNations], [EuropeanUnion] or [Eurozone],
 * - Sub-national territories of countries such as [SaintHelena] or [SouthGeorgiaAndSouthSandwichIslands].
 *
 * @param code Either alpha-2 tor alpha-3 ISO code or numeric UN M.49 code.
 * @param alpha2Code alpha-2 two-letter country codes are "most prominently for the Internet's country code top-level
 * domains (with a few exceptions). They are also used as country identifiers extending the postal
 * code when appropriate within the international postal system for paper mail."
 *
 * User-assigned codes are: AA, QM to QZ, XA to XZ, and ZZ.
 * These can be freely used and will never be part of the standard.
 * @param alpha3Code alpha-3 three-letter country codes "allow a better visual association between the codes and the
 * country names than the alpha-2 codes. They represent countries, dependent territories, and special
 * areas of geographical interest. (...) They are used most prominently in ISO/IEC 7501-1 for
 * machine-readable passports."
 *
 * User-assigned codes are: AAA to AAZ, QMA to QZZ, XAA to XZZ, and ZZA to ZZZ.
 * These can be freely used and will never be part of the standard.
 * @param numericCode ISO 3166-1 `numeric` codes are three-digit country codes that originate from
 * [UN M.49](https://en.wikipedia.org/wiki/UN_M.49) standard, with the advantage of script (writing system)
 * independence, and hence useful for people or systems using non-Latin scripts such as Arabic or Chinese.
 *
 * The UN M.49 contains also codes for geographical and political regions like a continent and therefore
 * allow for hierarchical mapping of regions.
 *
 * User-assigned codes range from 900 to 999. These are reserved for users to add custom geographical names 
 * in their applications and will never be used by the ISO 3166 standard.
 * @param numericCodeAsString The value of [numericCode] as String, padded with zero to three digits.
 * @param englishName English name of the country or region.
 * @param variantName Optional a variant of the English name of the country or region (if available).
 * @param defaultLanguage Region's default language.
 * @param defaultScript Region's default script (writing system).
 * @param isContainedIn A list of regions this region is contained in.
 * @param contains A list of region this region contains.
 */
enum class Region(
  val code: String,
  val alpha2Code: String?,
  val alpha3Code: String?,
  val numericCode: Int?,
  val numericCodeAsString: String?,
  val englishName: String?,
  val variantName: String?,
  val defaultLanguage: Language?,
  val defaultScript: Script?,
  val isContainedIn: List<String>,
  val contains: List<String>,
) {
  World("001", null, null, 1, "001", "world", null, null, null, listOf(), listOf("019", "002", "150", "142", "009")),
  Africa("002", null, null, 2, "002", "Africa", null, null, null, listOf("001"), listOf("015", "011", "017", "014", "018")),
  NorthAmerica("003", null, null, 3, "003", "North America", null, null, null, listOf(), listOf("021", "013", "029")),
  SouthAmerica("005", null, null, 5, "005", "South America", null, null, null, listOf("019", "419"), listOf("AR", "BO", "BR", "BV", "CL", "CO", "EC", "FK", "GF", "GS", "GY", "PE", "PY", "SR", "UY", "VE")),
  Oceania("009", null, null, 9, "009", "Oceania", null, null, null, listOf("001"), listOf("053", "054", "057", "061", "QO")),
  WesternAfrica("011", null, null, 11, "011", "Western Africa", null, null, null, listOf("002", "202"), listOf("BF", "BJ", "CI", "CV", "GH", "GM", "GN", "GW", "LR", "ML", "MR", "NE", "NG", "SH", "SL", "SN", "TG")),
  CentralAmerica("013", null, null, 13, "013", "Central America", null, null, null, listOf("003", "019", "419"), listOf("BZ", "CR", "GT", "HN", "MX", "NI", "PA", "SV")),
  EasternAfrica("014", null, null, 14, "014", "Eastern Africa", null, null, null, listOf("002", "202"), listOf("BI", "DJ", "ER", "ET", "IO", "KE", "KM", "MG", "MU", "MW", "MZ", "RE", "RW", "SC", "SO", "SS", "TF", "TZ", "UG", "YT", "ZM", "ZW")),
  NorthernAfrica("015", null, null, 15, "015", "Northern Africa", null, null, null, listOf("002"), listOf("DZ", "EG", "EH", "LY", "MA", "SD", "TN", "EA", "IC")),
  MiddleAfrica("017", null, null, 17, "017", "Middle Africa", null, null, null, listOf("002", "202"), listOf("AO", "CD", "CF", "CG", "CM", "GA", "GQ", "ST", "TD")),
  SouthernAfrica("018", null, null, 18, "018", "Southern Africa", null, null, null, listOf("002", "202"), listOf("BW", "LS", "NA", "SZ", "ZA")),
  Americas("019", null, null, 19, "019", "Americas", null, null, null, listOf("001"), listOf("021", "013", "029", "005")),
  NorthernAmerica("021", null, null, 21, "021", "Northern America", null, null, null, listOf("003", "019"), listOf("BM", "CA", "GL", "PM", "US")),
  Caribbean("029", null, null, 29, "029", "Caribbean", null, null, null, listOf("003", "019", "419"), listOf("AG", "AI", "AW", "BB", "BL", "BQ", "BS", "CU", "CW", "DM", "DO", "GD", "GP", "HT", "JM", "KN", "KY", "LC", "MF", "MQ", "MS", "PR", "SX", "TC", "TT", "VC", "VG", "VI")),
  EasternAsia("030", null, null, 30, "030", "Eastern Asia", null, null, null, listOf("142"), listOf("CN", "HK", "JP", "KP", "KR", "MN", "MO", "TW")),
  SouthernAsia("034", null, null, 34, "034", "Southern Asia", null, null, null, listOf("142"), listOf("AF", "BD", "BT", "IN", "IR", "LK", "MV", "NP", "PK")),
  SoutheastAsia("035", null, null, 35, "035", "Southeast Asia", null, null, null, listOf("142"), listOf("BN", "ID", "KH", "LA", "MM", "MY", "PH", "SG", "TH", "TL", "VN")),
  SouthernEurope("039", null, null, 39, "039", "Southern Europe", null, null, null, listOf("150"), listOf("AD", "AL", "BA", "ES", "GI", "GR", "HR", "IT", "ME", "MK", "MT", "RS", "PT", "SI", "SM", "VA", "XK")),
  Australasia("053", null, null, 53, "053", "Australasia", null, null, null, listOf("009"), listOf("AU", "CC", "CX", "HM", "NF", "NZ")),
  Melanesia("054", null, null, 54, "054", "Melanesia", null, null, null, listOf("009"), listOf("FJ", "NC", "PG", "SB", "VU")),
  MicronesianRegion("057", null, null, 57, "057", "Micronesian Region", null, null, null, listOf("009"), listOf("FM", "GU", "KI", "MH", "MP", "NR", "PW", "UM")),
  Polynesia("061", null, null, 61, "061", "Polynesia", null, null, null, listOf("009"), listOf("AS", "CK", "NU", "PF", "PN", "TK", "TO", "TV", "WF", "WS")),
  Asia("142", null, null, 142, "142", "Asia", null, null, null, listOf("001"), listOf("145", "143", "030", "034", "035")),
  CentralAsia("143", null, null, 143, "143", "Central Asia", null, null, null, listOf("142"), listOf("TM", "TJ", "KG", "KZ", "UZ")),
  WesternAsia("145", null, null, 145, "145", "Western Asia", null, null, null, listOf("142"), listOf("AE", "AM", "AZ", "BH", "CY", "GE", "IL", "IQ", "JO", "KW", "LB", "OM", "PS", "QA", "SA", "SY", "TR", "YE")),
  Europe("150", null, null, 150, "150", "Europe", null, null, null, listOf("001"), listOf("154", "155", "151", "039")),
  EasternEurope("151", null, null, 151, "151", "Eastern Europe", null, null, null, listOf("150"), listOf("BG", "BY", "CZ", "HU", "MD", "PL", "RO", "RU", "SK", "UA")),
  NorthernEurope("154", null, null, 154, "154", "Northern Europe", null, null, null, listOf("150"), listOf("GG", "IM", "JE", "AX", "DK", "EE", "FI", "FO", "GB", "IE", "IS", "LT", "LV", "NO", "SE", "SJ", "CQ")),
  WesternEurope("155", null, null, 155, "155", "Western Europe", null, null, null, listOf("150"), listOf("AT", "BE", "CH", "DE", "FR", "LI", "LU", "MC", "NL")),
  SubSaharanAfrica("202", null, null, 202, "202", "Sub-Saharan Africa", null, null, null, listOf(), listOf("011", "017", "014", "018")),
  LatinAmerica("419", null, null, 419, "419", "Latin America", null, null, null, listOf(), listOf("013", "029", "005")),
  AA("AA", "AA", "AAA", 958, "958", null, null, null, null, listOf(), listOf()),
  AN("AN", "AN", "ANT", 530, "530", null, null, null, null, listOf(), listOf()),
  Afghanistan("AF", "AF", "AFG", 4, "004", "Afghanistan", null, Language.Persian, Script.Arabic, listOf("034", "UN"), listOf()),
  Albania("AL", "AL", "ALB", 8, "008", "Albania", null, Language.Albanian, Script.Latin, listOf("039", "UN"), listOf()),
  Algeria("DZ", "DZ", "DZA", 12, "012", "Algeria", null, Language.Arabic, Script.Arabic, listOf("015", "UN"), listOf()),
  AmericanSamoa("AS", "AS", "ASM", 16, "016", "American Samoa", null, Language.Samoan, Script.Latin, listOf("061"), listOf()),
  Andorra("AD", "AD", "AND", 20, "020", "Andorra", null, Language.Catalan, Script.Latin, listOf("039", "UN"), listOf()),
  Angola("AO", "AO", "AGO", 24, "024", "Angola", null, Language.Portuguese, Script.Latin, listOf("017", "UN"), listOf()),
  Anguilla("AI", "AI", "AIA", 660, "660", "Anguilla", null, null, null, listOf("029"), listOf()),
  Antarctica("AQ", "AQ", "ATA", 10, "010", "Antarctica", null, null, null, listOf("QO"), listOf()),
  AntiguaAndBarbuda("AG", "AG", "ATG", 28, "028", "Antigua & Barbuda", null, null, null, listOf("029", "UN"), listOf()),
  Argentina("AR", "AR", "ARG", 32, "032", "Argentina", null, Language.Spanish, Script.Latin, listOf("005", "UN"), listOf()),
  Armenia("AM", "AM", "ARM", 51, "051", "Armenia", null, Language.Armenian, Script.Armenian, listOf("145", "UN"), listOf()),
  Aruba("AW", "AW", "ABW", 533, "533", "Aruba", null, Language.Dutch, Script.Latin, listOf("029"), listOf()),
  AscensionIsland("AC", "AC", "ASC", 654, "654", "Ascension Island", null, null, null, listOf("QO"), listOf()),
  Australia("AU", "AU", "AUS", 36, "036", "Australia", null, null, null, listOf("053", "UN"), listOf()),
  Austria("AT", "AT", "AUT", 40, "040", "Austria", null, Language.German, Script.Latin, listOf("155", "EU", "EZ", "UN"), listOf()),
  Azerbaijan("AZ", "AZ", "AZE", 31, "031", "Azerbaijan", null, Language.Azerbaijani, Script.Latin, listOf("145", "UN"), listOf()),
  BU("BU", "BU", "BUR", 104, "104", null, null, null, null, listOf(), listOf()),
  Bahamas("BS", "BS", "BHS", 44, "044", "Bahamas", null, null, null, listOf("029", "UN"), listOf()),
  Bahrain("BH", "BH", "BHR", 48, "048", "Bahrain", null, Language.Arabic, Script.Arabic, listOf("145", "UN"), listOf()),
  Bangladesh("BD", "BD", "BGD", 50, "050", "Bangladesh", null, Language.Bangla, Script.Bangla, listOf("034", "UN"), listOf()),
  Barbados("BB", "BB", "BRB", 52, "052", "Barbados", null, null, null, listOf("029", "UN"), listOf()),
  Belarus("BY", "BY", "BLR", 112, "112", "Belarus", null, Language.Belarusian, Script.Cyrillic, listOf("151", "UN"), listOf()),
  Belgium("BE", "BE", "BEL", 56, "056", "Belgium", null, Language.Dutch, Script.Latin, listOf("155", "EU", "EZ", "UN"), listOf()),
  Belize("BZ", "BZ", "BLZ", 84, "084", "Belize", null, null, null, listOf("013", "UN"), listOf()),
  Benin("BJ", "BJ", "BEN", 204, "204", "Benin", null, Language.French, Script.Latin, listOf("011", "UN"), listOf()),
  Bermuda("BM", "BM", "BMU", 60, "060", "Bermuda", null, null, null, listOf("021"), listOf()),
  Bhutan("BT", "BT", "BTN", 64, "064", "Bhutan", null, Language.Dzongkha, Script.Tibetan, listOf("034", "UN"), listOf()),
  Bolivia("BO", "BO", "BOL", 68, "068", "Bolivia", null, Language.Spanish, Script.Latin, listOf("005", "UN"), listOf()),
  BosniaAndHerzegovina("BA", "BA", "BIH", 70, "070", "Bosnia & Herzegovina", "Bosnia", Language.Bosnian, Script.Latin, listOf("039", "UN"), listOf()),
  Botswana("BW", "BW", "BWA", 72, "072", "Botswana", null, null, null, listOf("018", "UN"), listOf()),
  BouvetIsland("BV", "BV", "BVT", 74, "074", "Bouvet Island", null, null, null, listOf("005"), listOf()),
  Brazil("BR", "BR", "BRA", 76, "076", "Brazil", null, Language.Portuguese, Script.Latin, listOf("005", "UN"), listOf()),
  BritishIndianOceanTerritory("IO", "IO", "IOT", 86, "086", "British Indian Ocean Territory", null, null, null, listOf("014"), listOf()),
  BritishVirginIslands("VG", "VG", "VGB", 92, "092", "British Virgin Islands", null, null, null, listOf("029"), listOf()),
  Brunei("BN", "BN", "BRN", 96, "096", "Brunei", null, Language.Malay, Script.Latin, listOf("035", "UN"), listOf()),
  Bulgaria("BG", "BG", "BGR", 100, "100", "Bulgaria", null, Language.Bulgarian, Script.Cyrillic, listOf("151", "EU", "UN"), listOf()),
  BurkinaFaso("BF", "BF", "BFA", 854, "854", "Burkina Faso", null, Language.French, Script.Latin, listOf("011", "UN"), listOf()),
  Burundi("BI", "BI", "BDI", 108, "108", "Burundi", null, Language.Rundi, Script.Latin, listOf("014", "UN"), listOf()),
  CS("CS", "CS", "SCG", 891, "891", null, null, null, null, listOf(), listOf()),
  Cambodia("KH", "KH", "KHM", 116, "116", "Cambodia", null, Language.Khmer, Script.Khmer, listOf("035", "UN"), listOf()),
  Cameroon("CM", "CM", "CMR", 120, "120", "Cameroon", null, Language.French, Script.Latin, listOf("017", "UN"), listOf()),
  Canada("CA", "CA", "CAN", 124, "124", "Canada", null, null, null, listOf("021", "UN"), listOf()),
  CanaryIslands("IC", "IC", null, null, null, "Canary Islands", null, Language.Spanish, Script.Latin, listOf("015"), listOf()),
  CapeVerde("CV", "CV", "CPV", 132, "132", "Cape Verde", "Cabo Verde", Language.Portuguese, Script.Latin, listOf("011", "UN"), listOf()),
  CaribbeanNetherlands("BQ", "BQ", "BES", 535, "535", "Caribbean Netherlands", null, Language.Papiamento, Script.Latin, listOf("029"), listOf()),
  CaymanIslands("KY", "KY", "CYM", 136, "136", "Cayman Islands", null, null, null, listOf("029"), listOf()),
  CentralAfricanRepublic("CF", "CF", "CAF", 140, "140", "Central African Republic", null, Language.French, Script.Latin, listOf("017", "UN"), listOf()),
  CeutaAndMelilla("EA", "EA", null, null, null, "Ceuta & Melilla", null, Language.Spanish, Script.Latin, listOf("015"), listOf()),
  Chad("TD", "TD", "TCD", 148, "148", "Chad", null, Language.French, Script.Latin, listOf("017", "UN"), listOf()),
  Chile("CL", "CL", "CHL", 152, "152", "Chile", null, Language.Spanish, Script.Latin, listOf("005", "UN"), listOf()),
  China("CN", "CN", "CHN", 156, "156", "China", null, Language.Chinese, Script.Simplified, listOf("030", "UN"), listOf()),
  ChristmasIsland("CX", "CX", "CXR", 162, "162", "Christmas Island", null, null, null, listOf("053"), listOf()),
  ClippertonIsland("CP", "CP", "CPT", null, null, "Clipperton Island", null, null, null, listOf("QO"), listOf()),
  CocosKeelingIslands("CC", "CC", "CCK", 166, "166", "Cocos (Keeling) Islands", null, Language.Malay, Script.Arabic, listOf("053"), listOf()),
  Colombia("CO", "CO", "COL", 170, "170", "Colombia", null, Language.Spanish, Script.Latin, listOf("005", "UN"), listOf()),
  Comoros("KM", "KM", "COM", 174, "174", "Comoros", null, Language.Arabic, Script.Arabic, listOf("014", "UN"), listOf()),
  Congo("CG", "CG", "COG", 178, "178", "Congo - Brazzaville", "Congo (Republic)", Language.French, Script.Latin, listOf("017", "UN"), listOf()),
  Congo_DemocraticRepublic("CD", "CD", "COD", 180, "180", "Congo - Kinshasa", "Congo (DRC)", Language.Swahili, Script.Latin, listOf("017", "UN"), listOf()),
  CookIslands("CK", "CK", "COK", 184, "184", "Cook Islands", null, null, null, listOf("061"), listOf()),
  CostaRica("CR", "CR", "CRI", 188, "188", "Costa Rica", null, Language.Spanish, Script.Latin, listOf("013", "UN"), listOf()),
  Croatia("HR", "HR", "HRV", 191, "191", "Croatia", null, Language.Croatian, Script.Latin, listOf("039", "EU", "UN"), listOf()),
  Cuba("CU", "CU", "CUB", 192, "192", "Cuba", null, Language.Spanish, Script.Latin, listOf("029", "UN"), listOf()),
  Curacao("CW", "CW", "CUW", 531, "531", "Curaçao", null, Language.Papiamento, Script.Latin, listOf("029"), listOf()),
  Cyprus("CY", "CY", "CYP", 196, "196", "Cyprus", null, Language.Greek, Script.Greek, listOf("145", "EU", "EZ", "UN"), listOf()),
  Czechia("CZ", "CZ", "CZE", 203, "203", "Czechia", "Czech Republic", Language.Czech, Script.Latin, listOf("151", "EU", "UN"), listOf()),
  CoteDIvoire("CI", "CI", "CIV", 384, "384", "Côte d’Ivoire", "Ivory Coast", Language.French, Script.Latin, listOf("011", "UN"), listOf()),
  DD("DD", "DD", "DDR", 278, "278", null, null, null, null, listOf(), listOf()),
  Denmark("DK", "DK", "DNK", 208, "208", "Denmark", null, Language.Danish, Script.Latin, listOf("154", "EU", "UN"), listOf()),
  DiegoGarcia("DG", "DG", "DGA", null, null, "Diego Garcia", null, null, null, listOf("QO"), listOf()),
  Djibouti("DJ", "DJ", "DJI", 262, "262", "Djibouti", null, Language.Afar, Script.Latin, listOf("014", "UN"), listOf()),
  Dominica("DM", "DM", "DMA", 212, "212", "Dominica", null, null, null, listOf("029", "UN"), listOf()),
  DominicanRepublic("DO", "DO", "DOM", 214, "214", "Dominican Republic", null, Language.Spanish, Script.Latin, listOf("029", "UN"), listOf()),
  Ecuador("EC", "EC", "ECU", 218, "218", "Ecuador", null, Language.Spanish, Script.Latin, listOf("005", "UN"), listOf()),
  Egypt("EG", "EG", "EGY", 818, "818", "Egypt", null, Language.Arabic, Script.Arabic, listOf("015", "UN"), listOf()),
  ElSalvador("SV", "SV", "SLV", 222, "222", "El Salvador", null, Language.Spanish, Script.Latin, listOf("013", "UN"), listOf()),
  EquatorialGuinea("GQ", "GQ", "GNQ", 226, "226", "Equatorial Guinea", null, Language.Spanish, Script.Latin, listOf("017", "UN"), listOf()),
  Eritrea("ER", "ER", "ERI", 232, "232", "Eritrea", null, Language.Tigrinya, Script.Ethiopic, listOf("014", "UN"), listOf()),
  Estonia("EE", "EE", "EST", 233, "233", "Estonia", null, Language.Estonian, Script.Latin, listOf("154", "EU", "EZ", "UN"), listOf()),
  Eswatini("SZ", "SZ", "SWZ", 748, "748", "Eswatini", "Swaziland", null, null, listOf("018", "UN"), listOf()),
  Ethiopia("ET", "ET", "ETH", 231, "231", "Ethiopia", null, Language.Amharic, Script.Ethiopic, listOf("014", "UN"), listOf()),
  EuropeanUnion("EU", "EU", "QUU", 967, "967", "European Union", null, null, null, listOf(), listOf("AT", "BE", "CY", "CZ", "DE", "DK", "EE", "ES", "FI", "FR", "GR", "HR", "HU", "IE", "IT", "LT", "LU", "LV", "MT", "NL", "PL", "PT", "SE", "SI", "SK", "BG", "RO")),
  Eurozone("EZ", "EZ", null, null, null, "Eurozone", null, null, null, listOf(), listOf("AT", "BE", "CY", "DE", "EE", "ES", "FI", "FR", "GR", "IE", "IT", "LT", "LU", "LV", "MT", "NL", "PT", "SI", "SK")),
  FX("FX", "FX", "FXX", 249, "249", null, null, null, null, listOf(), listOf()),
  FalklandIslands("FK", "FK", "FLK", 238, "238", "Falkland Islands", "Falkland Islands (Islas Malvinas)", null, null, listOf("005"), listOf()),
  FaroeIslands("FO", "FO", "FRO", 234, "234", "Faroe Islands", null, Language.Faroese, Script.Latin, listOf("154"), listOf()),
  Fiji("FJ", "FJ", "FJI", 242, "242", "Fiji", null, null, null, listOf("054", "UN"), listOf()),
  Finland("FI", "FI", "FIN", 246, "246", "Finland", null, Language.Finnish, Script.Latin, listOf("154", "EU", "EZ", "UN"), listOf()),
  France("FR", "FR", "FRA", 250, "250", "France", null, Language.French, Script.Latin, listOf("155", "EU", "EZ", "UN"), listOf()),
  FrenchGuiana("GF", "GF", "GUF", 254, "254", "French Guiana", null, Language.French, Script.Latin, listOf("005"), listOf()),
  FrenchPolynesia("PF", "PF", "PYF", 258, "258", "French Polynesia", null, Language.French, Script.Latin, listOf("061"), listOf()),
  FrenchSouthernTerritories("TF", "TF", "ATF", 260, "260", "French Southern Territories", null, Language.French, Script.Latin, listOf("014"), listOf()),
  Gabon("GA", "GA", "GAB", 266, "266", "Gabon", null, Language.French, Script.Latin, listOf("017", "UN"), listOf()),
  Gambia("GM", "GM", "GMB", 270, "270", "Gambia", null, null, null, listOf("011", "UN"), listOf()),
  Georgia("GE", "GE", "GEO", 268, "268", "Georgia", null, Language.Georgian, Script.Georgian, listOf("145", "UN"), listOf()),
  Germany("DE", "DE", "DEU", 276, "276", "Germany", null, Language.German, Script.Latin, listOf("155", "EU", "EZ", "UN"), listOf()),
  Ghana("GH", "GH", "GHA", 288, "288", "Ghana", null, Language.Akan, Script.Latin, listOf("011", "UN"), listOf()),
  Gibraltar("GI", "GI", "GIB", 292, "292", "Gibraltar", null, null, null, listOf("039"), listOf()),
  Greece("GR", "GR", "GRC", 300, "300", "Greece", null, Language.Greek, Script.Greek, listOf("039", "EU", "EZ", "UN"), listOf()),
  Greenland("GL", "GL", "GRL", 304, "304", "Greenland", null, Language.Kalaallisut, Script.Latin, listOf("021"), listOf()),
  Grenada("GD", "GD", "GRD", 308, "308", "Grenada", null, null, null, listOf("029", "UN"), listOf()),
  Guadeloupe("GP", "GP", "GLP", 312, "312", "Guadeloupe", null, Language.French, Script.Latin, listOf("029"), listOf()),
  Guam("GU", "GU", "GUM", 316, "316", "Guam", null, null, null, listOf("057"), listOf()),
  Guatemala("GT", "GT", "GTM", 320, "320", "Guatemala", null, Language.Spanish, Script.Latin, listOf("013", "UN"), listOf()),
  Guernsey("GG", "GG", "GGY", 831, "831", "Guernsey", null, null, null, listOf("154"), listOf()),
  Guinea("GN", "GN", "GIN", 324, "324", "Guinea", null, Language.French, Script.Latin, listOf("011", "UN"), listOf()),
  GuineaBissau("GW", "GW", "GNB", 624, "624", "Guinea-Bissau", null, Language.Portuguese, Script.Latin, listOf("011", "UN"), listOf()),
  Guyana("GY", "GY", "GUY", 328, "328", "Guyana", null, null, null, listOf("005", "UN"), listOf()),
  Haiti("HT", "HT", "HTI", 332, "332", "Haiti", null, Language.HaitianCreole, Script.Latin, listOf("029", "UN"), listOf()),
  HeardAndMcDonaldIslands("HM", "HM", "HMD", 334, "334", "Heard & McDonald Islands", null, null, null, listOf("053"), listOf()),
  Honduras("HN", "HN", "HND", 340, "340", "Honduras", null, Language.Spanish, Script.Latin, listOf("013", "UN"), listOf()),
  HongKong("HK", "HK", "HKG", 344, "344", "Hong Kong SAR China", "Hong Kong", Language.Chinese, Script.Traditional, listOf("030"), listOf()),
  Hungary("HU", "HU", "HUN", 348, "348", "Hungary", null, Language.Hungarian, Script.Latin, listOf("151", "EU", "UN"), listOf()),
  Iceland("IS", "IS", "ISL", 352, "352", "Iceland", null, Language.Icelandic, Script.Latin, listOf("154", "UN"), listOf()),
  India("IN", "IN", "IND", 356, "356", "India", null, Language.Hindi, Script.Devanagari, listOf("034", "UN"), listOf()),
  Indonesia("ID", "ID", "IDN", 360, "360", "Indonesia", null, Language.Indonesian, Script.Latin, listOf("035", "UN"), listOf()),
  Iran("IR", "IR", "IRN", 364, "364", "Iran", null, Language.Persian, Script.Arabic, listOf("034", "UN"), listOf()),
  Iraq("IQ", "IQ", "IRQ", 368, "368", "Iraq", null, Language.Arabic, Script.Arabic, listOf("145", "UN"), listOf()),
  Ireland("IE", "IE", "IRL", 372, "372", "Ireland", null, null, null, listOf("154", "EU", "EZ", "UN"), listOf()),
  IsleOfMan("IM", "IM", "IMN", 833, "833", "Isle of Man", null, null, null, listOf("154"), listOf()),
  Israel("IL", "IL", "ISR", 376, "376", "Israel", null, Language.Hebrew, Script.Hebrew, listOf("145", "UN"), listOf()),
  Italy("IT", "IT", "ITA", 380, "380", "Italy", null, Language.Italian, Script.Latin, listOf("039", "EU", "EZ", "UN"), listOf()),
  Jamaica("JM", "JM", "JAM", 388, "388", "Jamaica", null, null, null, listOf("029", "UN"), listOf()),
  Japan("JP", "JP", "JPN", 392, "392", "Japan", null, Language.Japanese, Script.Japanese, listOf("030", "UN"), listOf()),
  Jersey("JE", "JE", "JEY", 832, "832", "Jersey", null, null, null, listOf("154"), listOf()),
  Jordan("JO", "JO", "JOR", 400, "400", "Jordan", null, Language.Arabic, Script.Arabic, listOf("145", "UN"), listOf()),
  Kazakhstan("KZ", "KZ", "KAZ", 398, "398", "Kazakhstan", null, Language.Russian, Script.Cyrillic, listOf("143", "UN"), listOf()),
  Kenya("KE", "KE", "KEN", 404, "404", "Kenya", null, Language.Swahili, Script.Latin, listOf("014", "UN"), listOf()),
  Kiribati("KI", "KI", "KIR", 296, "296", "Kiribati", null, null, null, listOf("057", "UN"), listOf()),
  Kosovo("XK", "XK", "XKK", 983, "983", "Kosovo", null, Language.Albanian, Script.Latin, listOf("039"), listOf()),
  Kuwait("KW", "KW", "KWT", 414, "414", "Kuwait", null, Language.Arabic, Script.Arabic, listOf("145", "UN"), listOf()),
  Kyrgyzstan("KG", "KG", "KGZ", 417, "417", "Kyrgyzstan", null, Language.Kyrgyz, Script.Cyrillic, listOf("143", "UN"), listOf()),
  Laos("LA", "LA", "LAO", 418, "418", "Laos", null, Language.Lao, Script.Lao, listOf("035", "UN"), listOf()),
  Latvia("LV", "LV", "LVA", 428, "428", "Latvia", null, Language.Latvian, Script.Latin, listOf("154", "EU", "EZ", "UN"), listOf()),
  Lebanon("LB", "LB", "LBN", 422, "422", "Lebanon", null, Language.Arabic, Script.Arabic, listOf("145", "UN"), listOf()),
  Lesotho("LS", "LS", "LSO", 426, "426", "Lesotho", null, Language.SouthernSotho, Script.Latin, listOf("018", "UN"), listOf()),
  Liberia("LR", "LR", "LBR", 430, "430", "Liberia", null, null, null, listOf("011", "UN"), listOf()),
  Libya("LY", "LY", "LBY", 434, "434", "Libya", null, Language.Arabic, Script.Arabic, listOf("015", "UN"), listOf()),
  Liechtenstein("LI", "LI", "LIE", 438, "438", "Liechtenstein", null, Language.German, Script.Latin, listOf("155", "UN"), listOf()),
  Lithuania("LT", "LT", "LTU", 440, "440", "Lithuania", null, Language.Lithuanian, Script.Latin, listOf("154", "EU", "EZ", "UN"), listOf()),
  Luxembourg("LU", "LU", "LUX", 442, "442", "Luxembourg", null, Language.French, Script.Latin, listOf("155", "EU", "EZ", "UN"), listOf()),
  Macao("MO", "MO", "MAC", 446, "446", "Macao SAR China", "Macao", Language.Chinese, Script.Traditional, listOf("030"), listOf()),
  Madagascar("MG", "MG", "MDG", 450, "450", "Madagascar", null, Language.Malagasy, Script.Latin, listOf("014", "UN"), listOf()),
  Malawi("MW", "MW", "MWI", 454, "454", "Malawi", null, null, null, listOf("014", "UN"), listOf()),
  Malaysia("MY", "MY", "MYS", 458, "458", "Malaysia", null, Language.Malay, Script.Latin, listOf("035", "UN"), listOf()),
  Maldives("MV", "MV", "MDV", 462, "462", "Maldives", null, Language.Divehi, Script.Thaana, listOf("034", "UN"), listOf()),
  Mali("ML", "ML", "MLI", 466, "466", "Mali", null, Language.Bambara, Script.Latin, listOf("011", "UN"), listOf()),
  Malta("MT", "MT", "MLT", 470, "470", "Malta", null, Language.Maltese, Script.Latin, listOf("039", "EU", "EZ", "UN"), listOf()),
  MarshallIslands("MH", "MH", "MHL", 584, "584", "Marshall Islands", null, null, null, listOf("057", "UN"), listOf()),
  Martinique("MQ", "MQ", "MTQ", 474, "474", "Martinique", null, Language.French, Script.Latin, listOf("029"), listOf()),
  Mauritania("MR", "MR", "MRT", 478, "478", "Mauritania", null, Language.Arabic, Script.Arabic, listOf("011", "UN"), listOf()),
  Mauritius("MU", "MU", "MUS", 480, "480", "Mauritius", null, Language.Morisyen, Script.Latin, listOf("014", "UN"), listOf()),
  Mayotte("YT", "YT", "MYT", 175, "175", "Mayotte", null, Language.French, Script.Latin, listOf("014"), listOf()),
  Mexico("MX", "MX", "MEX", 484, "484", "Mexico", null, Language.Spanish, Script.Latin, listOf("013", "UN"), listOf()),
  Micronesia("FM", "FM", "FSM", 583, "583", "Micronesia", null, null, null, listOf("057", "UN"), listOf()),
  Moldova("MD", "MD", "MDA", 498, "498", "Moldova", null, Language.Romanian, Script.Latin, listOf("151", "UN"), listOf()),
  Monaco("MC", "MC", "MCO", 492, "492", "Monaco", null, Language.French, Script.Latin, listOf("155", "UN"), listOf()),
  Mongolia("MN", "MN", "MNG", 496, "496", "Mongolia", null, Language.Mongolian, Script.Cyrillic, listOf("030", "UN"), listOf()),
  Montenegro("ME", "ME", "MNE", 499, "499", "Montenegro", null, Language.Serbian, Script.Latin, listOf("039", "UN"), listOf()),
  Montserrat("MS", "MS", "MSR", 500, "500", "Montserrat", null, null, null, listOf("029"), listOf()),
  Morocco("MA", "MA", "MAR", 504, "504", "Morocco", null, Language.Arabic, Script.Arabic, listOf("015", "UN"), listOf()),
  Mozambique("MZ", "MZ", "MOZ", 508, "508", "Mozambique", null, Language.Portuguese, Script.Latin, listOf("014", "UN"), listOf()),
  MyanmarBurma("MM", "MM", "MMR", 104, "104", "Myanmar (Burma)", "Myanmar", Language.Burmese, Script.Myanmar, listOf("035", "UN"), listOf()),
  NT("NT", "NT", "NTZ", 536, "536", null, null, null, null, listOf(), listOf()),
  Namibia("NA", "NA", "NAM", 516, "516", "Namibia", null, Language.Afrikaans, Script.Latin, listOf("018", "UN"), listOf()),
  Nauru("NR", "NR", "NRU", 520, "520", "Nauru", null, null, null, listOf("057", "UN"), listOf()),
  Nepal("NP", "NP", "NPL", 524, "524", "Nepal", null, Language.Nepali, Script.Devanagari, listOf("034", "UN"), listOf()),
  Netherlands("NL", "NL", "NLD", 528, "528", "Netherlands", null, Language.Dutch, Script.Latin, listOf("155", "EU", "EZ", "UN"), listOf()),
  NewCaledonia("NC", "NC", "NCL", 540, "540", "New Caledonia", null, Language.French, Script.Latin, listOf("054"), listOf()),
  NewZealand("NZ", "NZ", "NZL", 554, "554", "New Zealand", "Aotearoa New Zealand", null, null, listOf("053", "UN"), listOf()),
  Nicaragua("NI", "NI", "NIC", 558, "558", "Nicaragua", null, Language.Spanish, Script.Latin, listOf("013", "UN"), listOf()),
  Niger("NE", "NE", "NER", 562, "562", "Niger", null, Language.Hausa, Script.Latin, listOf("011", "UN"), listOf()),
  Nigeria("NG", "NG", "NGA", 566, "566", "Nigeria", null, null, null, listOf("011", "UN"), listOf()),
  Niue("NU", "NU", "NIU", 570, "570", "Niue", null, null, null, listOf("061"), listOf()),
  NorfolkIsland("NF", "NF", "NFK", 574, "574", "Norfolk Island", null, null, null, listOf("053"), listOf()),
  NorthKorea("KP", "KP", "PRK", 408, "408", "North Korea", null, Language.Korean, Script.Korean, listOf("030", "UN"), listOf()),
  NorthMacedonia("MK", "MK", "MKD", 807, "807", "North Macedonia", null, Language.Macedonian, Script.Cyrillic, listOf("039", "UN"), listOf()),
  NorthernMarianaIslands("MP", "MP", "MNP", 580, "580", "Northern Mariana Islands", null, null, null, listOf("057"), listOf()),
  Norway("NO", "NO", "NOR", 578, "578", "Norway", null, Language.NorwegianBokmal, Script.Latin, listOf("154", "UN"), listOf()),
  Oman("OM", "OM", "OMN", 512, "512", "Oman", null, Language.Arabic, Script.Arabic, listOf("145", "UN"), listOf()),
  OutlyingOceania("QO", "QO", "QOO", 961, "961", "Outlying Oceania", null, null, null, listOf("009"), listOf("AQ", "AC", "CP", "DG", "TA")),
  Pakistan("PK", "PK", "PAK", 586, "586", "Pakistan", null, Language.Urdu, Script.Arabic, listOf("034", "UN"), listOf()),
  Palau("PW", "PW", "PLW", 585, "585", "Palau", null, Language.Palauan, Script.Latin, listOf("057", "UN"), listOf()),
  Palestine("PS", "PS", "PSE", 275, "275", "Palestinian Territories", "Palestine", Language.Arabic, Script.Arabic, listOf("145"), listOf()),
  Panama("PA", "PA", "PAN", 591, "591", "Panama", null, Language.Spanish, Script.Latin, listOf("013", "UN"), listOf()),
  PapuaNewGuinea("PG", "PG", "PNG", 598, "598", "Papua New Guinea", null, Language.TokPisin, Script.Latin, listOf("054", "UN"), listOf()),
  Paraguay("PY", "PY", "PRY", 600, "600", "Paraguay", null, Language.Guarani, Script.Latin, listOf("005", "UN"), listOf()),
  Peru("PE", "PE", "PER", 604, "604", "Peru", null, Language.Spanish, Script.Latin, listOf("005", "UN"), listOf()),
  Philippines("PH", "PH", "PHL", 608, "608", "Philippines", null, Language.Filipino, Script.Latin, listOf("035", "UN"), listOf()),
  PitcairnIslands("PN", "PN", "PCN", 612, "612", "Pitcairn Islands", null, null, null, listOf("061"), listOf()),
  Poland("PL", "PL", "POL", 616, "616", "Poland", null, Language.Polish, Script.Latin, listOf("151", "EU", "UN"), listOf()),
  Portugal("PT", "PT", "PRT", 620, "620", "Portugal", null, Language.Portuguese, Script.Latin, listOf("039", "EU", "EZ", "UN"), listOf()),
  PseudoAccents("XA", "XA", "XAA", 973, "973", "Pseudo-Accents", null, null, null, listOf(), listOf()),
  PseudoBidi("XB", "XB", "XBB", 974, "974", "Pseudo-Bidi", null, null, null, listOf(), listOf()),
  PuertoRico("PR", "PR", "PRI", 630, "630", "Puerto Rico", null, Language.Spanish, Script.Latin, listOf("029"), listOf()),
  QM("QM", "QM", "QMM", 959, "959", null, null, null, null, listOf(), listOf()),
  QN("QN", "QN", "QNN", 960, "960", null, null, null, null, listOf(), listOf()),
  QP("QP", "QP", "QPP", 962, "962", null, null, null, null, listOf(), listOf()),
  QQ("QQ", "QQ", "QQQ", 963, "963", null, null, null, null, listOf(), listOf()),
  QR("QR", "QR", "QRR", 964, "964", null, null, null, null, listOf(), listOf()),
  QS("QS", "QS", "QSS", 965, "965", null, null, null, null, listOf(), listOf()),
  QT("QT", "QT", "QTT", 966, "966", null, null, null, null, listOf(), listOf()),
  QU("QU", "QU", "QUU", 967, "967", null, null, null, null, listOf(), listOf()),
  QV("QV", "QV", "QVV", 968, "968", null, null, null, null, listOf(), listOf()),
  QW("QW", "QW", "QWW", 969, "969", null, null, null, null, listOf(), listOf()),
  QX("QX", "QX", "QXX", 970, "970", null, null, null, null, listOf(), listOf()),
  QY("QY", "QY", "QYY", 971, "971", null, null, null, null, listOf(), listOf()),
  QZ("QZ", "QZ", "QZZ", 972, "972", null, null, null, null, listOf(), listOf()),
  Qatar("QA", "QA", "QAT", 634, "634", "Qatar", null, Language.Arabic, Script.Arabic, listOf("145", "UN"), listOf()),
  Romania("RO", "RO", "ROU", 642, "642", "Romania", null, Language.Romanian, Script.Latin, listOf("151", "EU", "UN"), listOf()),
  Russia("RU", "RU", "RUS", 643, "643", "Russia", null, Language.Russian, Script.Cyrillic, listOf("151", "UN"), listOf()),
  Rwanda("RW", "RW", "RWA", 646, "646", "Rwanda", null, Language.Kinyarwanda, Script.Latin, listOf("014", "UN"), listOf()),
  Reunion("RE", "RE", "REU", 638, "638", "Réunion", null, Language.French, Script.Latin, listOf("014"), listOf()),
  SU("SU", "SU", "SUN", 810, "810", null, null, null, null, listOf(), listOf()),
  SaintBarthelemy("BL", "BL", "BLM", 652, "652", "St. Barthélemy", null, Language.French, Script.Latin, listOf("029"), listOf()),
  SaintHelena("SH", "SH", "SHN", 654, "654", "St. Helena", null, null, null, listOf("011"), listOf()),
  SaintKittsAndNevis("KN", "KN", "KNA", 659, "659", "St. Kitts & Nevis", null, null, null, listOf("029", "UN"), listOf()),
  SaintLucia("LC", "LC", "LCA", 662, "662", "St. Lucia", null, null, null, listOf("029", "UN"), listOf()),
  SaintMartin("MF", "MF", "MAF", 663, "663", "St. Martin", null, Language.French, Script.Latin, listOf("029"), listOf()),
  SaintPierreAndMiquelon("PM", "PM", "SPM", 666, "666", "St. Pierre & Miquelon", null, Language.French, Script.Latin, listOf("021"), listOf()),
  SaintVincentAndGrenadines("VC", "VC", "VCT", 670, "670", "St. Vincent & Grenadines", null, null, null, listOf("029", "UN"), listOf()),
  Samoa("WS", "WS", "WSM", 882, "882", "Samoa", null, Language.Samoan, Script.Latin, listOf("061", "UN"), listOf()),
  SanMarino("SM", "SM", "SMR", 674, "674", "San Marino", null, Language.Italian, Script.Latin, listOf("039", "UN"), listOf()),
  Sark("CQ", "CQ", null, null, null, "Sark", null, null, null, listOf("154"), listOf()),
  SaudiArabia("SA", "SA", "SAU", 682, "682", "Saudi Arabia", null, Language.Arabic, Script.Arabic, listOf("145", "UN"), listOf()),
  Senegal("SN", "SN", "SEN", 686, "686", "Senegal", null, Language.French, Script.Latin, listOf("011", "UN"), listOf()),
  Serbia("RS", "RS", "SRB", 688, "688", "Serbia", null, Language.Serbian, Script.Cyrillic, listOf("039", "UN"), listOf()),
  Seychelles("SC", "SC", "SYC", 690, "690", "Seychelles", null, Language.French, Script.Latin, listOf("014", "UN"), listOf()),
  SierraLeone("SL", "SL", "SLE", 694, "694", "Sierra Leone", null, Language.Krio, Script.Latin, listOf("011", "UN"), listOf()),
  Singapore("SG", "SG", "SGP", 702, "702", "Singapore", null, null, null, listOf("035", "UN"), listOf()),
  SintMaarten("SX", "SX", "SXM", 534, "534", "Sint Maarten", null, null, null, listOf("029"), listOf()),
  Slovakia("SK", "SK", "SVK", 703, "703", "Slovakia", null, Language.Slovak, Script.Latin, listOf("151", "EU", "EZ", "UN"), listOf()),
  Slovenia("SI", "SI", "SVN", 705, "705", "Slovenia", null, Language.Slovenian, Script.Latin, listOf("039", "EU", "EZ", "UN"), listOf()),
  SolomonIslands("SB", "SB", "SLB", 90, "090", "Solomon Islands", null, null, null, listOf("054", "UN"), listOf()),
  Somalia("SO", "SO", "SOM", 706, "706", "Somalia", null, Language.Somali, Script.Latin, listOf("014", "UN"), listOf()),
  SouthAfrica("ZA", "ZA", "ZAF", 710, "710", "South Africa", null, null, null, listOf("018", "UN"), listOf()),
  SouthGeorgiaAndSouthSandwichIslands("GS", "GS", "SGS", 239, "239", "South Georgia & South Sandwich Islands", null, null, null, listOf("005"), listOf()),
  SouthKorea("KR", "KR", "KOR", 410, "410", "South Korea", null, Language.Korean, Script.Korean, listOf("030", "UN"), listOf()),
  SouthSudan("SS", "SS", "SSD", 728, "728", "South Sudan", null, Language.Arabic, Script.Arabic, listOf("014", "UN"), listOf()),
  Spain("ES", "ES", "ESP", 724, "724", "Spain", null, Language.Spanish, Script.Latin, listOf("039", "EU", "EZ", "UN"), listOf()),
  SriLanka("LK", "LK", "LKA", 144, "144", "Sri Lanka", null, Language.Sinhala, Script.Sinhala, listOf("034", "UN"), listOf()),
  Sudan("SD", "SD", "SDN", 729, "729", "Sudan", null, Language.Arabic, Script.Arabic, listOf("015", "UN"), listOf()),
  Suriname("SR", "SR", "SUR", 740, "740", "Suriname", null, Language.Dutch, Script.Latin, listOf("005", "UN"), listOf()),
  SvalbardAndJanMayen("SJ", "SJ", "SJM", 744, "744", "Svalbard & Jan Mayen", null, Language.NorwegianBokmal, Script.Latin, listOf("154"), listOf()),
  Sweden("SE", "SE", "SWE", 752, "752", "Sweden", null, Language.Swedish, Script.Latin, listOf("154", "EU", "UN"), listOf()),
  Switzerland("CH", "CH", "CHE", 756, "756", "Switzerland", null, Language.German, Script.Latin, listOf("155", "UN"), listOf()),
  Syria("SY", "SY", "SYR", 760, "760", "Syria", null, Language.Arabic, Script.Arabic, listOf("145", "UN"), listOf()),
  SaoTomeAndPrincipe("ST", "ST", "STP", 678, "678", "São Tomé & Príncipe", null, Language.Portuguese, Script.Latin, listOf("017", "UN"), listOf()),
  TP("TP", "TP", "TMP", 626, "626", null, null, null, null, listOf(), listOf()),
  Taiwan("TW", "TW", "TWN", 158, "158", "Taiwan", null, Language.Chinese, Script.Traditional, listOf("030"), listOf()),
  Tajikistan("TJ", "TJ", "TJK", 762, "762", "Tajikistan", null, Language.Tajik, Script.Cyrillic, listOf("143", "UN"), listOf()),
  Tanzania("TZ", "TZ", "TZA", 834, "834", "Tanzania", null, Language.Swahili, Script.Latin, listOf("014", "UN"), listOf()),
  Thailand("TH", "TH", "THA", 764, "764", "Thailand", null, Language.Thai, Script.Thai, listOf("035", "UN"), listOf()),
  TimorLeste("TL", "TL", "TLS", 626, "626", "Timor-Leste", "East Timor", Language.Portuguese, Script.Latin, listOf("035", "UN"), listOf()),
  Togo("TG", "TG", "TGO", 768, "768", "Togo", null, Language.French, Script.Latin, listOf("011", "UN"), listOf()),
  Tokelau("TK", "TK", "TKL", 772, "772", "Tokelau", null, Language.Tokelau, Script.Latin, listOf("061"), listOf()),
  Tonga("TO", "TO", "TON", 776, "776", "Tonga", null, Language.Tongan, Script.Latin, listOf("061", "UN"), listOf()),
  TrinidadAndTobago("TT", "TT", "TTO", 780, "780", "Trinidad & Tobago", null, null, null, listOf("029", "UN"), listOf()),
  TristanDaCunha("TA", "TA", "TAA", 654, "654", "Tristan da Cunha", null, null, null, listOf("QO"), listOf()),
  Tunisia("TN", "TN", "TUN", 788, "788", "Tunisia", null, Language.Arabic, Script.Arabic, listOf("015", "UN"), listOf()),
  Turkmenistan("TM", "TM", "TKM", 795, "795", "Turkmenistan", null, Language.Turkmen, Script.Latin, listOf("143", "UN"), listOf()),
  TurksAndCaicosIslands("TC", "TC", "TCA", 796, "796", "Turks & Caicos Islands", null, null, null, listOf("029"), listOf()),
  Tuvalu("TV", "TV", "TUV", 798, "798", "Tuvalu", null, Language.Tuvalu, Script.Latin, listOf("061", "UN"), listOf()),
  Turkiye("TR", "TR", "TUR", 792, "792", "Türkiye", "Turkey", Language.Turkish, Script.Latin, listOf("145", "UN"), listOf()),
  USOutlyingIslands("UM", "UM", "UMI", 581, "581", "U.S. Outlying Islands", null, null, null, listOf("057"), listOf()),
  USVirginIslands("VI", "VI", "VIR", 850, "850", "U.S. Virgin Islands", null, null, null, listOf("029"), listOf()),
  Uganda("UG", "UG", "UGA", 800, "800", "Uganda", null, Language.Swahili, Script.Latin, listOf("014", "UN"), listOf()),
  Ukraine("UA", "UA", "UKR", 804, "804", "Ukraine", null, Language.Ukrainian, Script.Cyrillic, listOf("151", "UN"), listOf()),
  UnitedArabEmirates("AE", "AE", "ARE", 784, "784", "United Arab Emirates", null, Language.Arabic, Script.Arabic, listOf("145", "UN"), listOf()),
  UnitedKingdom("GB", "GB", "GBR", 826, "826", "United Kingdom", "UK", null, null, listOf("154", "UN"), listOf()),
  UnitedNations("UN", "UN", null, null, null, "United Nations", "UN", null, null, listOf(), listOf("AD", "AE", "AF", "AG", "AL", "AM", "AO", "AR", "AT", "AU", "AZ", "BA", "BB", "BD", "BE", "BF", "BG", "BH", "BI", "BJ", "BN", "BO", "BR", "BS", "BT", "BW", "BY", "BZ", "CA", "CD", "CF", "CG", "CH", "CI", "CL", "CM", "CN", "CO", "CR", "CU", "CV", "CY", "CZ", "DE", "DJ", "DK", "DM", "DO", "DZ", "EC", "EE", "EG", "ER", "ES", "ET", "FI", "FJ", "FM", "FR", "GA", "GB", "GD", "GE", "GH", "GM", "GN", "GQ", "GR", "GT", "GW", "GY", "HN", "HR", "HT", "HU", "ID", "IE", "IL", "IN", "IQ", "IR", "IS", "IT", "JM", "JO", "JP", "KE", "KG", "KH", "KI", "KM", "KN", "KP", "KR", "KW", "KZ", "LA", "LB", "LC", "LI", "LK", "LR", "LS", "LT", "LU", "LV", "LY", "MA", "MC", "MD", "ME", "MG", "MH", "MK", "ML", "MM", "MN", "MR", "MT", "MU", "MV", "MX", "MW", "MY", "MZ", "NA", "NE", "NG", "NI", "NL", "NO", "NR", "NP", "NZ", "OM", "PA", "PE", "PG", "PH", "PK", "PL", "PT", "PW", "PY", "QA", "RO", "RS", "RU", "RW", "SA", "SB", "SC", "SD", "SE", "SG", "SI", "SK", "SL", "SM", "SN", "SO", "SR", "SS", "ST", "SV", "SY", "SZ", "TD", "TG", "TH", "TJ", "TL", "TM", "TN", "TO", "TR", "TT", "TV", "TZ", "UA", "UG", "US", "UY", "UZ", "VC", "VE", "VN", "VU", "WS", "YE", "ZA", "ZM", "ZW")),
  UnitedStates("US", "US", "USA", 840, "840", "United States", "US", null, null, listOf("021", "UN"), listOf()),
  UnknownRegion("ZZ", "ZZ", "ZZZ", 999, "999", "Unknown Region", null, null, null, listOf(), listOf()),
  Uruguay("UY", "UY", "URY", 858, "858", "Uruguay", null, Language.Spanish, Script.Latin, listOf("005", "UN"), listOf()),
  Uzbekistan("UZ", "UZ", "UZB", 860, "860", "Uzbekistan", null, Language.Uzbek, Script.Latin, listOf("143", "UN"), listOf()),
  Vanuatu("VU", "VU", "VUT", 548, "548", "Vanuatu", null, Language.Bislama, Script.Latin, listOf("054", "UN"), listOf()),
  VaticanCity("VA", "VA", "VAT", 336, "336", "Vatican City", null, Language.Italian, Script.Latin, listOf("039"), listOf()),
  Venezuela("VE", "VE", "VEN", 862, "862", "Venezuela", null, Language.Spanish, Script.Latin, listOf("005", "UN"), listOf()),
  Vietnam("VN", "VN", "VNM", 704, "704", "Vietnam", null, Language.Vietnamese, Script.Latin, listOf("035", "UN"), listOf()),
  WallisAndFutuna("WF", "WF", "WLF", 876, "876", "Wallis & Futuna", null, Language.French, Script.Latin, listOf("061"), listOf()),
  WesternSahara("EH", "EH", "ESH", 732, "732", "Western Sahara", null, Language.Arabic, Script.Arabic, listOf("015"), listOf()),
  XC("XC", "XC", "XCC", 975, "975", null, null, null, null, listOf(), listOf()),
  XD("XD", "XD", "XDD", 976, "976", null, null, null, null, listOf(), listOf()),
  XE("XE", "XE", "XEE", 977, "977", null, null, null, null, listOf(), listOf()),
  XF("XF", "XF", "XFF", 978, "978", null, null, null, null, listOf(), listOf()),
  XG("XG", "XG", "XGG", 979, "979", null, null, null, null, listOf(), listOf()),
  XH("XH", "XH", "XHH", 980, "980", null, null, null, null, listOf(), listOf()),
  XI("XI", "XI", "XII", 981, "981", null, null, null, null, listOf(), listOf()),
  XJ("XJ", "XJ", "XJJ", 982, "982", null, null, null, null, listOf(), listOf()),
  XL("XL", "XL", "XLL", 984, "984", null, null, null, null, listOf(), listOf()),
  XM("XM", "XM", "XMM", 985, "985", null, null, null, null, listOf(), listOf()),
  XN("XN", "XN", "XNN", 986, "986", null, null, null, null, listOf(), listOf()),
  XO("XO", "XO", "XOO", 987, "987", null, null, null, null, listOf(), listOf()),
  XP("XP", "XP", "XPP", 988, "988", null, null, null, null, listOf(), listOf()),
  XQ("XQ", "XQ", "XQQ", 989, "989", null, null, null, null, listOf(), listOf()),
  XR("XR", "XR", "XRR", 990, "990", null, null, null, null, listOf(), listOf()),
  XS("XS", "XS", "XSS", 991, "991", null, null, null, null, listOf(), listOf()),
  XT("XT", "XT", "XTT", 992, "992", null, null, null, null, listOf(), listOf()),
  XU("XU", "XU", "XUU", 993, "993", null, null, null, null, listOf(), listOf()),
  XV("XV", "XV", "XVV", 994, "994", null, null, null, null, listOf(), listOf()),
  XW("XW", "XW", "XWW", 995, "995", null, null, null, null, listOf(), listOf()),
  XX("XX", "XX", "XXX", 996, "996", null, null, null, null, listOf(), listOf()),
  XY("XY", "XY", "XYY", 997, "997", null, null, null, null, listOf(), listOf()),
  XZ("XZ", "XZ", "XZZ", 998, "998", null, null, null, null, listOf(), listOf()),
  YD("YD", "YD", "YMD", 720, "720", null, null, null, null, listOf(), listOf()),
  YU("YU", "YU", "YUG", 891, "891", null, null, null, null, listOf(), listOf()),
  Yemen("YE", "YE", "YEM", 887, "887", "Yemen", null, Language.Arabic, Script.Arabic, listOf("145", "UN"), listOf()),
  ZR("ZR", "ZR", "ZAR", 180, "180", null, null, null, null, listOf(), listOf()),
  Zambia("ZM", "ZM", "ZMB", 894, "894", "Zambia", null, Language.Bemba, Script.Latin, listOf("014", "UN"), listOf()),
  Zimbabwe("ZW", "ZW", "ZWE", 716, "716", "Zimbabwe", null, Language.Shona, Script.Latin, listOf("014", "UN"), listOf()),
  AlandIslands("AX", "AX", "ALA", 248, "248", "Åland Islands", null, Language.Swedish, Script.Latin, listOf("154"), listOf()),
}
