package net.codinux.i18n

import kotlin.String

/**
 * Defines all Numbering Systems known to CLDR.
 *
 * @param code The lowercase code of this numbering system.
 * @param type The type of this numbering system (numeric or algorithmic).
 * @param digits The digits that this numbering system has.
 */
enum class NumberingSystem(
  val code: String,
  val type: NumberingSystemType,
  val digits: String?,
  val rules: String?,
  val englishName: String?,
) {
  AdlamDigits("adlm", NumberingSystemType.Numeric, "𞥐𞥑𞥒𞥓𞥔𞥕𞥖𞥗𞥘𞥙", null, "Adlam Digits"),
  AhomDigits("ahom", NumberingSystemType.Numeric, "𑜰𑜱𑜲𑜳𑜴𑜵𑜶𑜷𑜸𑜹", null, "Ahom Digits"),
  ArabicIndicDigits("arab", NumberingSystemType.Numeric, "٠١٢٣٤٥٦٧٨٩", null, "Arabic-Indic Digits"),
  ArmenianLowercaseNumerals("armnlow", NumberingSystemType.Algorithmic, null, "armenian-lower", "Armenian Lowercase Numerals"),
  ArmenianNumerals("armn", NumberingSystemType.Algorithmic, null, "armenian-upper", "Armenian Numerals"),
  BalineseDigits("bali", NumberingSystemType.Numeric, "᭐᭑᭒᭓᭔᭕᭖᭗᭘᭙", null, "Balinese Digits"),
  BanglaDigits("beng", NumberingSystemType.Numeric, "০১২৩৪৫৬৭৮৯", null, "Bangla Digits"),
  BhaiksukiDigits("bhks", NumberingSystemType.Numeric, "𑱐𑱑𑱒𑱓𑱔𑱕𑱖𑱗𑱘𑱙", null, "Bhaiksuki Digits"),
  BrahmiDigits("brah", NumberingSystemType.Numeric, "𑁦𑁧𑁨𑁩𑁪𑁫𑁬𑁭𑁮𑁯", null, "Brahmi Digits"),
  ChakmaDigits("cakm", NumberingSystemType.Numeric, "𑄶𑄷𑄸𑄹𑄺𑄻𑄼𑄽𑄾𑄿", null, "Chakma Digits"),
  ChamDigits("cham", NumberingSystemType.Numeric, "꩐꩑꩒꩓꩔꩕꩖꩗꩘꩙", null, "Cham Digits"),
  ChineseCalendarDayOfMonthNumerals("hanidays", NumberingSystemType.Algorithmic, null, "zh/SpelloutRules/spellout-numbering-days", "Chinese Calendar Day-of-Month Numerals"),
  ChineseDecimalNumerals("hanidec", NumberingSystemType.Numeric, "〇一二三四五六七八九", null, "Chinese Decimal Numerals"),
  CyrillicNumerals("cyrl", NumberingSystemType.Algorithmic, null, "cyrillic-lower", "Cyrillic Numerals"),
  DevanagariDigits("deva", NumberingSystemType.Numeric, "०१२३४५६७८९", null, "Devanagari Digits"),
  DivesAkuruDigits("diak", NumberingSystemType.Numeric, "𑥐𑥑𑥒𑥓𑥔𑥕𑥖𑥗𑥘𑥙", null, "Dives Akuru Digits"),
  EthiopicNumerals("ethi", NumberingSystemType.Algorithmic, null, "ethiopic", "Ethiopic Numerals"),
  ExtendedArabicIndicDigits("arabext", NumberingSystemType.Numeric, "۰۱۲۳۴۵۶۷۸۹", null, "Extended Arabic-Indic Digits"),
  FullWidthDigits("fullwide", NumberingSystemType.Numeric, "０１２３４５６７８９", null, "Full-Width Digits"),
  GarayDigits("gara", NumberingSystemType.Numeric, "𐵀𐵁𐵂𐵃𐵄𐵅𐵆𐵇𐵈𐵉", null, "Garay Digits"),
  GeorgianNumerals("geor", NumberingSystemType.Algorithmic, null, "georgian", "Georgian Numerals"),
  GreekLowercaseNumerals("greklow", NumberingSystemType.Algorithmic, null, "greek-lower", "Greek Lowercase Numerals"),
  GreekNumerals("grek", NumberingSystemType.Algorithmic, null, "greek-upper", "Greek Numerals"),
  GujaratiDigits("gujr", NumberingSystemType.Numeric, "૦૧૨૩૪૫૬૭૮૯", null, "Gujarati Digits"),
  GunjalaGondiDigits("gong", NumberingSystemType.Numeric, "𑶠𑶡𑶢𑶣𑶤𑶥𑶦𑶧𑶨𑶩", null, "Gunjala Gondi digits"),
  GurmukhiDigits("guru", NumberingSystemType.Numeric, "੦੧੨੩੪੫੬੭੮੯", null, "Gurmukhi Digits"),
  GurungKhemaDigits("gukh", NumberingSystemType.Numeric, "𖄰𖄱𖄲𖄳𖄴𖄵𖄶𖄷𖄸𖄹", null, "Gurung Khema Digits"),
  HanifiRohingyaDigits("rohg", NumberingSystemType.Numeric, "𐴰𐴱𐴲𐴳𐴴𐴵𐴶𐴷𐴸𐴹", null, "Hanifi Rohingya digits"),
  HebrewNumerals("hebr", NumberingSystemType.Algorithmic, null, "hebrew", "Hebrew Numerals"),
  JapaneseCalendarGannenYearNumerals("jpanyear", NumberingSystemType.Algorithmic, null, "ja/SpelloutRules/spellout-numbering-year-latn", "Japanese Calendar Gannen Year Numerals"),
  JapaneseFinancialNumerals("jpanfin", NumberingSystemType.Algorithmic, null, "ja/SpelloutRules/spellout-cardinal-financial", "Japanese Financial Numerals"),
  JapaneseNumerals("jpan", NumberingSystemType.Algorithmic, null, "ja/SpelloutRules/spellout-cardinal", "Japanese Numerals"),
  JavaneseDigits("java", NumberingSystemType.Numeric, "꧐꧑꧒꧓꧔꧕꧖꧗꧘꧙", null, "Javanese Digits"),
  KannadaDigits("knda", NumberingSystemType.Numeric, "೦೧೨೩೪೫೬೭೮೯", null, "Kannada Digits"),
  KawiDigits("kawi", NumberingSystemType.Numeric, "𑽐𑽑𑽒𑽓𑽔𑽕𑽖𑽗𑽘𑽙", null, "Kawi Digits"),
  KayahLiDigits("kali", NumberingSystemType.Numeric, "꤀꤁꤂꤃꤄꤅꤆꤇꤈꤉", null, "Kayah Li Digits"),
  KhmerDigits("khmr", NumberingSystemType.Numeric, "០១២៣៤៥៦៧៨៩", null, "Khmer Digits"),
  KhudawadiDigits("sind", NumberingSystemType.Numeric, "𑋰𑋱𑋲𑋳𑋴𑋵𑋶𑋷𑋸𑋹", null, "Khudawadi Digits"),
  KiratRaiDigits("krai", NumberingSystemType.Numeric, "𖵰𖵱𖵲𖵳𖵴𖵵𖵶𖵷𖵸𖵹", null, "Kirat Rai Digits"),
  LaoDigits("laoo", NumberingSystemType.Numeric, "໐໑໒໓໔໕໖໗໘໙", null, "Lao Digits"),
  LepchaDigits("lepc", NumberingSystemType.Numeric, "᱀᱁᱂᱃᱄᱅᱆᱇᱈᱉", null, "Lepcha Digits"),
  LimbuDigits("limb", NumberingSystemType.Numeric, "᥆᥇᥈᥉᥊᥋᥌᥍᥎᥏", null, "Limbu Digits"),
  MalayalamDigits("mlym", NumberingSystemType.Numeric, "൦൧൨൩൪൫൬൭൮൯", null, "Malayalam Digits"),
  MasaramGondiDigits("gonm", NumberingSystemType.Numeric, "𑵐𑵑𑵒𑵓𑵔𑵕𑵖𑵗𑵘𑵙", null, "Masaram Gondi digits"),
  MathematicalBoldDigits("mathbold", NumberingSystemType.Numeric, "𝟎𝟏𝟐𝟑𝟒𝟓𝟔𝟕𝟖𝟗", null, "Mathematical Bold Digits"),
  MathematicalDoubleStruckDigits("mathdbl", NumberingSystemType.Numeric, "𝟘𝟙𝟚𝟛𝟜𝟝𝟞𝟟𝟠𝟡", null, "Mathematical Double-Struck Digits"),
  MathematicalMonospaceDigits("mathmono", NumberingSystemType.Numeric, "𝟶𝟷𝟸𝟹𝟺𝟻𝟼𝟽𝟾𝟿", null, "Mathematical Monospace Digits"),
  MathematicalSansSerifBoldDigits("mathsanb", NumberingSystemType.Numeric, "𝟬𝟭𝟮𝟯𝟰𝟱𝟲𝟳𝟴𝟵", null, "Mathematical Sans-Serif Bold Digits"),
  MathematicalSansSerifDigits("mathsans", NumberingSystemType.Numeric, "𝟢𝟣𝟤𝟥𝟦𝟧𝟨𝟩𝟪𝟫", null, "Mathematical Sans-Serif Digits"),
  MeeteiMayekDigits("mtei", NumberingSystemType.Numeric, "꯰꯱꯲꯳꯴꯵꯶꯷꯸꯹", null, "Meetei Mayek Digits"),
  ModiDigits("modi", NumberingSystemType.Numeric, "𑙐𑙑𑙒𑙓𑙔𑙕𑙖𑙗𑙘𑙙", null, "Modi Digits"),
  MongolianDigits("mong", NumberingSystemType.Numeric, "᠐᠑᠒᠓᠔᠕᠖᠗᠘᠙", null, "Mongolian Digits"),
  MroDigits("mroo", NumberingSystemType.Numeric, "𖩠𖩡𖩢𖩣𖩤𖩥𖩦𖩧𖩨𖩩", null, "Mro Digits"),
  MyanmarDigits("mymr", NumberingSystemType.Numeric, "၀၁၂၃၄၅၆၇၈၉", null, "Myanmar Digits"),
  MyanmarEasternPwoKarenDigits("mymrepka", NumberingSystemType.Numeric, "𑛚𑛛𑛜𑛝𑛞𑛟𑛠𑛡𑛢𑛣", null, "Myanmar Eastern Pwo Karen Digits"),
  MyanmarPaoDigits("mymrpao", NumberingSystemType.Numeric, "𑛐𑛑𑛒𑛓𑛔𑛕𑛖𑛗𑛘𑛙", null, "Myanmar Pao Digits"),
  MyanmarShanDigits("mymrshan", NumberingSystemType.Numeric, "႐႑႒႓႔႕႖႗႘႙", null, "Myanmar Shan Digits"),
  MyanmarTaiLaingDigits("mymrtlng", NumberingSystemType.Numeric, "꧰꧱꧲꧳꧴꧵꧶꧷꧸꧹", null, "Myanmar Tai Laing Digits"),
  NagMundariDigits("nagm", NumberingSystemType.Numeric, "𞓰𞓱𞓲𞓳𞓴𞓵𞓶𞓷𞓸𞓹", null, "Nag Mundari Digits"),
  NewTaiLueDigits("talu", NumberingSystemType.Numeric, "᧐᧑᧒᧓᧔᧕᧖᧗᧘᧙", null, "New Tai Lue Digits"),
  NewaDigits("newa", NumberingSystemType.Numeric, "𑑐𑑑𑑒𑑓𑑔𑑕𑑖𑑗𑑘𑑙", null, "Newa Digits"),
  NyiakengPuachueHmongDigits("hmnp", NumberingSystemType.Numeric, "𞅀𞅁𞅂𞅃𞅄𞅅𞅆𞅇𞅈𞅉", null, "Nyiakeng Puachue Hmong Digits"),
  NKoDigits("nkoo", NumberingSystemType.Numeric, "߀߁߂߃߄߅߆߇߈߉", null, "N’Ko Digits"),
  OdiaDigits("orya", NumberingSystemType.Numeric, "୦୧୨୩୪୫୬୭୮୯", null, "Odia Digits"),
  OlChikiDigits("olck", NumberingSystemType.Numeric, "᱐᱑᱒᱓᱔᱕᱖᱗᱘᱙", null, "Ol Chiki Digits"),
  OlOnalDigits("onao", NumberingSystemType.Numeric, "𞗱𞗲𞗳𞗴𞗵𞗶𞗷𞗸𞗹𞗺", null, "Ol Onal Digits"),
  OsmanyaDigits("osma", NumberingSystemType.Numeric, "𐒠𐒡𐒢𐒣𐒤𐒥𐒦𐒧𐒨𐒩", null, "Osmanya Digits"),
  OutlinedDigits("outlined", NumberingSystemType.Numeric, "𜳰𜳱𜳲𜳳𜳴𜳵𜳶𜳷𜳸𜳹", null, "Outlined Digits"),
  PahawhHmongDigits("hmng", NumberingSystemType.Numeric, "𖭐𖭑𖭒𖭓𖭔𖭕𖭖𖭗𖭘𖭙", null, "Pahawh Hmong Digits"),
  RomanLowercaseNumerals("romanlow", NumberingSystemType.Algorithmic, null, "roman-lower", "Roman Lowercase Numerals"),
  RomanNumerals("roman", NumberingSystemType.Algorithmic, null, "roman-upper", "Roman Numerals"),
  SaurashtraDigits("saur", NumberingSystemType.Numeric, "꣐꣑꣒꣓꣔꣕꣖꣗꣘꣙", null, "Saurashtra Digits"),
  SegmentedDigits("segment", NumberingSystemType.Numeric, "🯰🯱🯲🯳🯴🯵🯶🯷🯸🯹", null, "Segmented Digits"),
  SharadaDigits("shrd", NumberingSystemType.Numeric, "𑇐𑇑𑇒𑇓𑇔𑇕𑇖𑇗𑇘𑇙", null, "Sharada Digits"),
  SimplifiedChineseFinancialNumerals("hansfin", NumberingSystemType.Algorithmic, null, "zh/SpelloutRules/spellout-cardinal-financial", "Simplified Chinese Financial Numerals"),
  SimplifiedChineseNumerals("hans", NumberingSystemType.Algorithmic, null, "zh/SpelloutRules/spellout-cardinal", "Simplified Chinese Numerals"),
  SinhalaLithDigits("sinh", NumberingSystemType.Numeric, "෦෧෨෩෪෫෬෭෮෯", null, "Sinhala Lith Digits"),
  SoraSompengDigits("sora", NumberingSystemType.Numeric, "𑃰𑃱𑃲𑃳𑃴𑃵𑃶𑃷𑃸𑃹", null, "Sora Sompeng Digits"),
  SundaneseDigits("sund", NumberingSystemType.Numeric, "᮰᮱᮲᮳᮴᮵᮶᮷᮸᮹", null, "Sundanese Digits"),
  SunuwarDigits("sunu", NumberingSystemType.Numeric, "𑯰𑯱𑯲𑯳𑯴𑯵𑯶𑯷𑯸𑯹", null, "Sunuwar Digits"),
  TaiThamHoraDigits("lana", NumberingSystemType.Numeric, "᪀᪁᪂᪃᪄᪅᪆᪇᪈᪉", null, "Tai Tham Hora Digits"),
  TaiThamThamDigits("lanatham", NumberingSystemType.Numeric, "᪐᪑᪒᪓᪔᪕᪖᪗᪘᪙", null, "Tai Tham Tham Digits"),
  TakriDigits("takr", NumberingSystemType.Numeric, "𑛀𑛁𑛂𑛃𑛄𑛅𑛆𑛇𑛈𑛉", null, "Takri Digits"),
  TamilDigits("tamldec", NumberingSystemType.Numeric, "௦௧௨௩௪௫௬௭௮௯", null, "Tamil Digits"),
  TangsaDigits("tnsa", NumberingSystemType.Numeric, "𖫀𖫁𖫂𖫃𖫄𖫅𖫆𖫇𖫈𖫉", null, "Tangsa Digits"),
  TeluguDigits("telu", NumberingSystemType.Numeric, "౦౧౨౩౪౫౬౭౮౯", null, "Telugu Digits"),
  ThaiDigits("thai", NumberingSystemType.Numeric, "๐๑๒๓๔๕๖๗๘๙", null, "Thai Digits"),
  TibetanDigits("tibt", NumberingSystemType.Numeric, "༠༡༢༣༤༥༦༧༨༩", null, "Tibetan Digits"),
  TirhutaDigits("tirh", NumberingSystemType.Numeric, "𑓐𑓑𑓒𑓓𑓔𑓕𑓖𑓗𑓘𑓙", null, "Tirhuta Digits"),
  TraditionalChineseFinancialNumerals("hantfin", NumberingSystemType.Algorithmic, null, "zh_Hant/SpelloutRules/spellout-cardinal-financial", "Traditional Chinese Financial Numerals"),
  TraditionalChineseNumerals("hant", NumberingSystemType.Algorithmic, null, "zh_Hant/SpelloutRules/spellout-cardinal", "Traditional Chinese Numerals"),
  TraditionalTamilNumerals("taml", NumberingSystemType.Algorithmic, null, "tamil", "Traditional Tamil Numerals"),
  VaiDigits("vaii", NumberingSystemType.Numeric, "꘠꘡꘢꘣꘤꘥꘦꘧꘨꘩", null, "Vai Digits"),
  WanchoDigits("wcho", NumberingSystemType.Numeric, "𞋰𞋱𞋲𞋳𞋴𞋵𞋶𞋷𞋸𞋹", null, "Wancho Digits"),
  WarangCitiDigits("wara", NumberingSystemType.Numeric, "𑣠𑣡𑣢𑣣𑣤𑣥𑣦𑣧𑣨𑣩", null, "Warang Citi Digits"),
  WesternDigits("latn", NumberingSystemType.Numeric, "0123456789", null, "Western Digits"),
  ;

  companion object {
    fun forCode(code: String): NumberingSystem = NumberingSystem.entries.first { it.code == code }
  }
}
