package net.codinux.i18n

import kotlin.Int
import kotlin.String
import kotlin.collections.List

/**
 * @param code Either alpha-2 tor alpha-3 ISO code or numeric UN M.49 code.
 * @param alpha2Code alpha-2 two-letter country codes are "most prominently for the Internet's country code top-level
 * domains (with a few exceptions). They are also used as country identifiers extending the postal
 * code when appropriate within the international postal system for paper mail."
 *
 * User-assigned codes are: AA, QM to QZ, XA to XZ, and ZZ.
 * These can be freely used and will never be part of the standard.
 * @param alpha3Code alpha-3 three-letter country codes "allow a better visual association between the codes and the
 * country names than the alpha-2 codes. They represent countries, dependent territories, and special
 * areas of geographical interest. (...) They are used most prominently in ISO/IEC 7501-1 for
 * machine-readable passports."
 *
 * User-assigned codes are: AAA to AAZ, QMA to QZZ, XAA to XZZ, and ZZA to ZZZ.
 * These can be freely used and will never be part of the standard.
 * @param numericCode ISO 3166-1 `numeric` codes are three-digit country codes that originate from
 * [UN M.49](https://en.wikipedia.org/wiki/UN_M.49) standard, with the advantage of script (writing system)
 * independence, and hence useful for people or systems using non-Latin scripts such as Arabic or Chinese.
 *
 * The UN M.49 contains also codes for geographical and political regions like a continent and therefore
 * allow for hierarchical mapping of regions.
 *
 * User-assigned codes range from 900 to 999. These are reserved for users to add custom geographical names 
 * in their applications and will never be used by the ISO 3166 standard.
 * @param numericCodeAsString The value of [numericCode] as String, padded with zero to three digits.
 * @param englishName English name of the country or region.
 * @param variantName Optional a variant of the English name of the country or region (if available).
 * @param isContainedIn A list of regions this region is contained in.
 * @param contains A list of region this region contains.
 */
enum class Region(
  val code: String,
  val alpha2Code: String?,
  val alpha3Code: String?,
  val numericCode: Int?,
  val numericCodeAsString: String?,
  val englishName: String,
  val variantName: String?,
  val isContainedIn: List<String>,
  val contains: List<String>,
) {
  World("001", null, null, 1, "001", "world", null, listOf(), listOf("019", "002", "150", "142", "009")),
  Africa("002", null, null, 2, "002", "Africa", null, listOf("001"), listOf("015", "011", "017", "014", "018")),
  NorthAmerica("003", null, null, 3, "003", "North America", null, listOf("019-status-grouping"), listOf("021", "013", "029")),
  SouthAmerica("005", null, null, 5, "005", "South America", null, listOf("019", "419"), listOf("AR", "BO", "BR", "BV", "CL", "CO", "EC", "FK", "GF", "GS", "GY", "PE", "PY", "SR", "UY", "VE")),
  Oceania("009", null, null, 9, "009", "Oceania", null, listOf("001"), listOf("053", "054", "057", "061", "QO")),
  WesternAfrica("011", null, null, 11, "011", "Western Africa", null, listOf("002", "202"), listOf("BF", "BJ", "CI", "CV", "GH", "GM", "GN", "GW", "LR", "ML", "MR", "NE", "NG", "SH", "SL", "SN", "TG")),
  CentralAmerica("013", null, null, 13, "013", "Central America", null, listOf("003", "019", "419"), listOf("BZ", "CR", "GT", "HN", "MX", "NI", "PA", "SV")),
  EasternAfrica("014", null, null, 14, "014", "Eastern Africa", null, listOf("002", "202"), listOf("BI", "DJ", "ER", "ET", "IO", "KE", "KM", "MG", "MU", "MW", "MZ", "RE", "RW", "SC", "SO", "SS", "TF", "TZ", "UG", "YT", "ZM", "ZW")),
  NorthernAfrica("015", null, null, 15, "015", "Northern Africa", null, listOf("002"), listOf("DZ", "EG", "EH", "LY", "MA", "SD", "TN", "EA", "IC")),
  MiddleAfrica("017", null, null, 17, "017", "Middle Africa", null, listOf("002", "202"), listOf("AO", "CD", "CF", "CG", "CM", "GA", "GQ", "ST", "TD")),
  SouthernAfrica("018", null, null, 18, "018", "Southern Africa", null, listOf("002", "202"), listOf("BW", "LS", "NA", "SZ", "ZA")),
  Americas("019", null, null, 19, "019", "Americas", null, listOf("001"), listOf("021", "013", "029", "005")),
  NorthernAmerica("021", null, null, 21, "021", "Northern America", null, listOf("003", "019"), listOf("BM", "CA", "GL", "PM", "US")),
  Caribbean("029", null, null, 29, "029", "Caribbean", null, listOf("003", "019", "419"), listOf("AG", "AI", "AW", "BB", "BL", "BQ", "BS", "CU", "CW", "DM", "DO", "GD", "GP", "HT", "JM", "KN", "KY", "LC", "MF", "MQ", "MS", "PR", "SX", "TC", "TT", "VC", "VG", "VI")),
  EasternAsia("030", null, null, 30, "030", "Eastern Asia", null, listOf("142"), listOf("CN", "HK", "JP", "KP", "KR", "MN", "MO", "TW")),
  SouthernAsia("034", null, null, 34, "034", "Southern Asia", null, listOf("142"), listOf("AF", "BD", "BT", "IN", "IR", "LK", "MV", "NP", "PK")),
  SoutheastAsia("035", null, null, 35, "035", "Southeast Asia", null, listOf("142"), listOf("BN", "ID", "KH", "LA", "MM", "MY", "PH", "SG", "TH", "TL", "VN")),
  SouthernEurope("039", null, null, 39, "039", "Southern Europe", null, listOf("150"), listOf("AD", "AL", "BA", "ES", "GI", "GR", "HR", "IT", "ME", "MK", "MT", "RS", "PT", "SI", "SM", "VA", "XK")),
  Australasia("053", null, null, 53, "053", "Australasia", null, listOf("009"), listOf("AU", "CC", "CX", "HM", "NF", "NZ")),
  Melanesia("054", null, null, 54, "054", "Melanesia", null, listOf("009"), listOf("FJ", "NC", "PG", "SB", "VU")),
  MicronesianRegion("057", null, null, 57, "057", "Micronesian Region", null, listOf("009"), listOf("FM", "GU", "KI", "MH", "MP", "NR", "PW", "UM")),
  Polynesia("061", null, null, 61, "061", "Polynesia", null, listOf("009"), listOf("AS", "CK", "NU", "PF", "PN", "TK", "TO", "TV", "WF", "WS")),
  Asia("142", null, null, 142, "142", "Asia", null, listOf("001"), listOf("145", "143", "030", "034", "035")),
  CentralAsia("143", null, null, 143, "143", "Central Asia", null, listOf("142"), listOf("TM", "TJ", "KG", "KZ", "UZ")),
  WesternAsia("145", null, null, 145, "145", "Western Asia", null, listOf("142"), listOf("AE", "AM", "AZ", "BH", "CY", "GE", "IL", "IQ", "JO", "KW", "LB", "OM", "PS", "QA", "SA", "SY", "TR", "YE")),
  Europe("150", null, null, 150, "150", "Europe", null, listOf("001"), listOf("154", "155", "151", "039")),
  EasternEurope("151", null, null, 151, "151", "Eastern Europe", null, listOf("150"), listOf("BG", "BY", "CZ", "HU", "MD", "PL", "RO", "RU", "SK", "UA")),
  NorthernEurope("154", null, null, 154, "154", "Northern Europe", null, listOf("150"), listOf("GG", "IM", "JE", "AX", "DK", "EE", "FI", "FO", "GB", "IE", "IS", "LT", "LV", "NO", "SE", "SJ", "CQ")),
  WesternEurope("155", null, null, 155, "155", "Western Europe", null, listOf("150"), listOf("AT", "BE", "CH", "DE", "FR", "LI", "LU", "MC", "NL")),
  SubSaharanAfrica("202", null, null, 202, "202", "Sub-Saharan Africa", null, listOf("002-status-grouping"), listOf("011", "017", "014", "018")),
  LatinAmerica("419", null, null, 419, "419", "Latin America", null, listOf("019-status-grouping"), listOf("013", "029", "005")),
  AscensionIsland("AC", "AC", "ASC", 654, "654", "Ascension Island", null, listOf("QO"), listOf()),
  Andorra("AD", "AD", "AND", 20, "020", "Andorra", null, listOf("039", "UN"), listOf()),
  UnitedArabEmirates("AE", "AE", "ARE", 784, "784", "United Arab Emirates", null, listOf("145", "UN"), listOf()),
  Afghanistan("AF", "AF", "AFG", 4, "004", "Afghanistan", null, listOf("034", "UN"), listOf()),
  AntiguaAndBarbuda("AG", "AG", "ATG", 28, "028", "Antigua & Barbuda", null, listOf("029", "UN"), listOf()),
  Anguilla("AI", "AI", "AIA", 660, "660", "Anguilla", null, listOf("029"), listOf()),
  Albania("AL", "AL", "ALB", 8, "008", "Albania", null, listOf("039", "UN"), listOf()),
  Armenia("AM", "AM", "ARM", 51, "051", "Armenia", null, listOf("145", "UN"), listOf()),
  Angola("AO", "AO", "AGO", 24, "024", "Angola", null, listOf("017", "UN"), listOf()),
  Antarctica("AQ", "AQ", "ATA", 10, "010", "Antarctica", null, listOf("QO"), listOf()),
  Argentina("AR", "AR", "ARG", 32, "032", "Argentina", null, listOf("005", "UN"), listOf()),
  AmericanSamoa("AS", "AS", "ASM", 16, "016", "American Samoa", null, listOf("061"), listOf()),
  Austria("AT", "AT", "AUT", 40, "040", "Austria", null, listOf("155", "EU", "EZ", "UN"), listOf()),
  Australia("AU", "AU", "AUS", 36, "036", "Australia", null, listOf("053", "UN"), listOf()),
  Aruba("AW", "AW", "ABW", 533, "533", "Aruba", null, listOf("029"), listOf()),
  AlandIslands("AX", "AX", "ALA", 248, "248", "Åland Islands", null, listOf("154"), listOf()),
  Azerbaijan("AZ", "AZ", "AZE", 31, "031", "Azerbaijan", null, listOf("145", "UN"), listOf()),
  BosniaAndHerzegovina("BA", "BA", "BIH", 70, "070", "Bosnia & Herzegovina", "Bosnia", listOf("039", "UN"), listOf()),
  Barbados("BB", "BB", "BRB", 52, "052", "Barbados", null, listOf("029", "UN"), listOf()),
  Bangladesh("BD", "BD", "BGD", 50, "050", "Bangladesh", null, listOf("034", "UN"), listOf()),
  Belgium("BE", "BE", "BEL", 56, "056", "Belgium", null, listOf("155", "EU", "EZ", "UN"), listOf()),
  BurkinaFaso("BF", "BF", "BFA", 854, "854", "Burkina Faso", null, listOf("011", "UN"), listOf()),
  Bulgaria("BG", "BG", "BGR", 100, "100", "Bulgaria", null, listOf("151", "EU", "UN"), listOf()),
  Bahrain("BH", "BH", "BHR", 48, "048", "Bahrain", null, listOf("145", "UN"), listOf()),
  Burundi("BI", "BI", "BDI", 108, "108", "Burundi", null, listOf("014", "UN"), listOf()),
  Benin("BJ", "BJ", "BEN", 204, "204", "Benin", null, listOf("011", "UN"), listOf()),
  SaintBarthelemy("BL", "BL", "BLM", 652, "652", "St. Barthélemy", null, listOf("029"), listOf()),
  Bermuda("BM", "BM", "BMU", 60, "060", "Bermuda", null, listOf("021"), listOf()),
  Brunei("BN", "BN", "BRN", 96, "096", "Brunei", null, listOf("035", "UN"), listOf()),
  Bolivia("BO", "BO", "BOL", 68, "068", "Bolivia", null, listOf("005", "UN"), listOf()),
  CaribbeanNetherlands("BQ", "BQ", "BES", 535, "535", "Caribbean Netherlands", null, listOf("029"), listOf()),
  Brazil("BR", "BR", "BRA", 76, "076", "Brazil", null, listOf("005", "UN"), listOf()),
  Bahamas("BS", "BS", "BHS", 44, "044", "Bahamas", null, listOf("029", "UN"), listOf()),
  Bhutan("BT", "BT", "BTN", 64, "064", "Bhutan", null, listOf("034", "UN"), listOf()),
  BouvetIsland("BV", "BV", "BVT", 74, "074", "Bouvet Island", null, listOf("005"), listOf()),
  Botswana("BW", "BW", "BWA", 72, "072", "Botswana", null, listOf("018", "UN"), listOf()),
  Belarus("BY", "BY", "BLR", 112, "112", "Belarus", null, listOf("151", "UN"), listOf()),
  Belize("BZ", "BZ", "BLZ", 84, "084", "Belize", null, listOf("013", "UN"), listOf()),
  Canada("CA", "CA", "CAN", 124, "124", "Canada", null, listOf("021", "UN"), listOf()),
  CocosKeelingIslands("CC", "CC", "CCK", 166, "166", "Cocos (Keeling) Islands", null, listOf("053"), listOf()),
  Congo_DemocraticRepublic("CD", "CD", "COD", 180, "180", "Congo - Kinshasa", "Congo (DRC)", listOf("017", "UN"), listOf()),
  CentralAfricanRepublic("CF", "CF", "CAF", 140, "140", "Central African Republic", null, listOf("017", "UN"), listOf()),
  Congo("CG", "CG", "COG", 178, "178", "Congo - Brazzaville", "Congo (Republic)", listOf("017", "UN"), listOf()),
  Switzerland("CH", "CH", "CHE", 756, "756", "Switzerland", null, listOf("155", "UN"), listOf()),
  CoteDIvoire("CI", "CI", "CIV", 384, "384", "Côte d’Ivoire", "Ivory Coast", listOf("011", "UN"), listOf()),
  CookIslands("CK", "CK", "COK", 184, "184", "Cook Islands", null, listOf("061"), listOf()),
  Chile("CL", "CL", "CHL", 152, "152", "Chile", null, listOf("005", "UN"), listOf()),
  Cameroon("CM", "CM", "CMR", 120, "120", "Cameroon", null, listOf("017", "UN"), listOf()),
  China("CN", "CN", "CHN", 156, "156", "China", null, listOf("030", "UN"), listOf()),
  Colombia("CO", "CO", "COL", 170, "170", "Colombia", null, listOf("005", "UN"), listOf()),
  ClippertonIsland("CP", "CP", "CPT", null, null, "Clipperton Island", null, listOf("QO"), listOf()),
  Sark("CQ", "CQ", null, null, null, "Sark", null, listOf("154"), listOf()),
  CostaRica("CR", "CR", "CRI", 188, "188", "Costa Rica", null, listOf("013", "UN"), listOf()),
  Cuba("CU", "CU", "CUB", 192, "192", "Cuba", null, listOf("029", "UN"), listOf()),
  CapeVerde("CV", "CV", "CPV", 132, "132", "Cape Verde", "Cabo Verde", listOf("011", "UN"), listOf()),
  Curacao("CW", "CW", "CUW", 531, "531", "Curaçao", null, listOf("029"), listOf()),
  ChristmasIsland("CX", "CX", "CXR", 162, "162", "Christmas Island", null, listOf("053"), listOf()),
  Cyprus("CY", "CY", "CYP", 196, "196", "Cyprus", null, listOf("145", "EU", "EZ", "UN"), listOf()),
  Czechia("CZ", "CZ", "CZE", 203, "203", "Czechia", "Czech Republic", listOf("151", "EU", "UN"), listOf()),
  Germany("DE", "DE", "DEU", 276, "276", "Germany", null, listOf("155", "EU", "EZ", "UN"), listOf()),
  DiegoGarcia("DG", "DG", "DGA", null, null, "Diego Garcia", null, listOf("QO"), listOf()),
  Djibouti("DJ", "DJ", "DJI", 262, "262", "Djibouti", null, listOf("014", "UN"), listOf()),
  Denmark("DK", "DK", "DNK", 208, "208", "Denmark", null, listOf("154", "EU", "UN"), listOf()),
  Dominica("DM", "DM", "DMA", 212, "212", "Dominica", null, listOf("029", "UN"), listOf()),
  DominicanRepublic("DO", "DO", "DOM", 214, "214", "Dominican Republic", null, listOf("029", "UN"), listOf()),
  Algeria("DZ", "DZ", "DZA", 12, "012", "Algeria", null, listOf("015", "UN"), listOf()),
  CeutaAndMelilla("EA", "EA", null, null, null, "Ceuta & Melilla", null, listOf("015"), listOf()),
  Ecuador("EC", "EC", "ECU", 218, "218", "Ecuador", null, listOf("005", "UN"), listOf()),
  Estonia("EE", "EE", "EST", 233, "233", "Estonia", null, listOf("154", "EU", "EZ", "UN"), listOf()),
  Egypt("EG", "EG", "EGY", 818, "818", "Egypt", null, listOf("015", "UN"), listOf()),
  WesternSahara("EH", "EH", "ESH", 732, "732", "Western Sahara", null, listOf("015"), listOf()),
  Eritrea("ER", "ER", "ERI", 232, "232", "Eritrea", null, listOf("014", "UN"), listOf()),
  Spain("ES", "ES", "ESP", 724, "724", "Spain", null, listOf("039", "EU", "EZ", "UN"), listOf()),
  Ethiopia("ET", "ET", "ETH", 231, "231", "Ethiopia", null, listOf("014", "UN"), listOf()),
  EuropeanUnion("EU", "EU", "QUU", 967, "967", "European Union", null, listOf("001-status-grouping"), listOf("AT", "BE", "CY", "CZ", "DE", "DK", "EE", "ES", "FI", "FR", "GR", "HR", "HU", "IE", "IT", "LT", "LU", "LV", "MT", "NL", "PL", "PT", "SE", "SI", "SK", "BG", "RO")),
  Eurozone("EZ", "EZ", null, null, null, "Eurozone", null, listOf("001-status-grouping"), listOf("AT", "BE", "CY", "DE", "EE", "ES", "FI", "FR", "GR", "IE", "IT", "LT", "LU", "LV", "MT", "NL", "PT", "SI", "SK")),
  Finland("FI", "FI", "FIN", 246, "246", "Finland", null, listOf("154", "EU", "EZ", "UN"), listOf()),
  Fiji("FJ", "FJ", "FJI", 242, "242", "Fiji", null, listOf("054", "UN"), listOf()),
  FalklandIslands("FK", "FK", "FLK", 238, "238", "Falkland Islands", "Falkland Islands (Islas Malvinas)", listOf("005"), listOf()),
  Micronesia("FM", "FM", "FSM", 583, "583", "Micronesia", null, listOf("057", "UN"), listOf()),
  FaroeIslands("FO", "FO", "FRO", 234, "234", "Faroe Islands", null, listOf("154"), listOf()),
  France("FR", "FR", "FRA", 250, "250", "France", null, listOf("155", "EU", "EZ", "UN"), listOf()),
  Gabon("GA", "GA", "GAB", 266, "266", "Gabon", null, listOf("017", "UN"), listOf()),
  UnitedKingdom("GB", "GB", "GBR", 826, "826", "United Kingdom", "UK", listOf("154", "UN"), listOf()),
  Grenada("GD", "GD", "GRD", 308, "308", "Grenada", null, listOf("029", "UN"), listOf()),
  Georgia("GE", "GE", "GEO", 268, "268", "Georgia", null, listOf("145", "UN"), listOf()),
  FrenchGuiana("GF", "GF", "GUF", 254, "254", "French Guiana", null, listOf("005"), listOf()),
  Guernsey("GG", "GG", "GGY", 831, "831", "Guernsey", null, listOf("154"), listOf()),
  Ghana("GH", "GH", "GHA", 288, "288", "Ghana", null, listOf("011", "UN"), listOf()),
  Gibraltar("GI", "GI", "GIB", 292, "292", "Gibraltar", null, listOf("039"), listOf()),
  Greenland("GL", "GL", "GRL", 304, "304", "Greenland", null, listOf("021"), listOf()),
  Gambia("GM", "GM", "GMB", 270, "270", "Gambia", null, listOf("011", "UN"), listOf()),
  Guinea("GN", "GN", "GIN", 324, "324", "Guinea", null, listOf("011", "UN"), listOf()),
  Guadeloupe("GP", "GP", "GLP", 312, "312", "Guadeloupe", null, listOf("029"), listOf()),
  EquatorialGuinea("GQ", "GQ", "GNQ", 226, "226", "Equatorial Guinea", null, listOf("017", "UN"), listOf()),
  Greece("GR", "GR", "GRC", 300, "300", "Greece", null, listOf("039", "EU", "EZ", "UN"), listOf()),
  SouthGeorgiaAndSouthSandwichIslands("GS", "GS", "SGS", 239, "239", "South Georgia & South Sandwich Islands", null, listOf("005"), listOf()),
  Guatemala("GT", "GT", "GTM", 320, "320", "Guatemala", null, listOf("013", "UN"), listOf()),
  Guam("GU", "GU", "GUM", 316, "316", "Guam", null, listOf("057"), listOf()),
  GuineaBissau("GW", "GW", "GNB", 624, "624", "Guinea-Bissau", null, listOf("011", "UN"), listOf()),
  Guyana("GY", "GY", "GUY", 328, "328", "Guyana", null, listOf("005", "UN"), listOf()),
  HongKong("HK", "HK", "HKG", 344, "344", "Hong Kong SAR China", "Hong Kong", listOf("030"), listOf()),
  HeardAndMcDonaldIslands("HM", "HM", "HMD", 334, "334", "Heard & McDonald Islands", null, listOf("053"), listOf()),
  Honduras("HN", "HN", "HND", 340, "340", "Honduras", null, listOf("013", "UN"), listOf()),
  Croatia("HR", "HR", "HRV", 191, "191", "Croatia", null, listOf("039", "EU", "UN"), listOf()),
  Haiti("HT", "HT", "HTI", 332, "332", "Haiti", null, listOf("029", "UN"), listOf()),
  Hungary("HU", "HU", "HUN", 348, "348", "Hungary", null, listOf("151", "EU", "UN"), listOf()),
  CanaryIslands("IC", "IC", null, null, null, "Canary Islands", null, listOf("015"), listOf()),
  Indonesia("ID", "ID", "IDN", 360, "360", "Indonesia", null, listOf("035", "UN"), listOf()),
  Ireland("IE", "IE", "IRL", 372, "372", "Ireland", null, listOf("154", "EU", "EZ", "UN"), listOf()),
  Israel("IL", "IL", "ISR", 376, "376", "Israel", null, listOf("145", "UN"), listOf()),
  IsleOfMan("IM", "IM", "IMN", 833, "833", "Isle of Man", null, listOf("154"), listOf()),
  India("IN", "IN", "IND", 356, "356", "India", null, listOf("034", "UN"), listOf()),
  BritishIndianOceanTerritory("IO", "IO", "IOT", 86, "086", "British Indian Ocean Territory", null, listOf("014"), listOf()),
  Iraq("IQ", "IQ", "IRQ", 368, "368", "Iraq", null, listOf("145", "UN"), listOf()),
  Iran("IR", "IR", "IRN", 364, "364", "Iran", null, listOf("034", "UN"), listOf()),
  Iceland("IS", "IS", "ISL", 352, "352", "Iceland", null, listOf("154", "UN"), listOf()),
  Italy("IT", "IT", "ITA", 380, "380", "Italy", null, listOf("039", "EU", "EZ", "UN"), listOf()),
  Jersey("JE", "JE", "JEY", 832, "832", "Jersey", null, listOf("154"), listOf()),
  Jamaica("JM", "JM", "JAM", 388, "388", "Jamaica", null, listOf("029", "UN"), listOf()),
  Jordan("JO", "JO", "JOR", 400, "400", "Jordan", null, listOf("145", "UN"), listOf()),
  Japan("JP", "JP", "JPN", 392, "392", "Japan", null, listOf("030", "UN"), listOf()),
  Kenya("KE", "KE", "KEN", 404, "404", "Kenya", null, listOf("014", "UN"), listOf()),
  Kyrgyzstan("KG", "KG", "KGZ", 417, "417", "Kyrgyzstan", null, listOf("143", "UN"), listOf()),
  Cambodia("KH", "KH", "KHM", 116, "116", "Cambodia", null, listOf("035", "UN"), listOf()),
  Kiribati("KI", "KI", "KIR", 296, "296", "Kiribati", null, listOf("057", "UN"), listOf()),
  Comoros("KM", "KM", "COM", 174, "174", "Comoros", null, listOf("014", "UN"), listOf()),
  SaintKittsAndNevis("KN", "KN", "KNA", 659, "659", "St. Kitts & Nevis", null, listOf("029", "UN"), listOf()),
  NorthKorea("KP", "KP", "PRK", 408, "408", "North Korea", null, listOf("030", "UN"), listOf()),
  SouthKorea("KR", "KR", "KOR", 410, "410", "South Korea", null, listOf("030", "UN"), listOf()),
  Kuwait("KW", "KW", "KWT", 414, "414", "Kuwait", null, listOf("145", "UN"), listOf()),
  CaymanIslands("KY", "KY", "CYM", 136, "136", "Cayman Islands", null, listOf("029"), listOf()),
  Kazakhstan("KZ", "KZ", "KAZ", 398, "398", "Kazakhstan", null, listOf("143", "UN"), listOf()),
  Laos("LA", "LA", "LAO", 418, "418", "Laos", null, listOf("035", "UN"), listOf()),
  Lebanon("LB", "LB", "LBN", 422, "422", "Lebanon", null, listOf("145", "UN"), listOf()),
  SaintLucia("LC", "LC", "LCA", 662, "662", "St. Lucia", null, listOf("029", "UN"), listOf()),
  Liechtenstein("LI", "LI", "LIE", 438, "438", "Liechtenstein", null, listOf("155", "UN"), listOf()),
  SriLanka("LK", "LK", "LKA", 144, "144", "Sri Lanka", null, listOf("034", "UN"), listOf()),
  Liberia("LR", "LR", "LBR", 430, "430", "Liberia", null, listOf("011", "UN"), listOf()),
  Lesotho("LS", "LS", "LSO", 426, "426", "Lesotho", null, listOf("018", "UN"), listOf()),
  Lithuania("LT", "LT", "LTU", 440, "440", "Lithuania", null, listOf("154", "EU", "EZ", "UN"), listOf()),
  Luxembourg("LU", "LU", "LUX", 442, "442", "Luxembourg", null, listOf("155", "EU", "EZ", "UN"), listOf()),
  Latvia("LV", "LV", "LVA", 428, "428", "Latvia", null, listOf("154", "EU", "EZ", "UN"), listOf()),
  Libya("LY", "LY", "LBY", 434, "434", "Libya", null, listOf("015", "UN"), listOf()),
  Morocco("MA", "MA", "MAR", 504, "504", "Morocco", null, listOf("015", "UN"), listOf()),
  Monaco("MC", "MC", "MCO", 492, "492", "Monaco", null, listOf("155", "UN"), listOf()),
  Moldova("MD", "MD", "MDA", 498, "498", "Moldova", null, listOf("151", "UN"), listOf()),
  Montenegro("ME", "ME", "MNE", 499, "499", "Montenegro", null, listOf("039", "UN"), listOf()),
  SaintMartin("MF", "MF", "MAF", 663, "663", "St. Martin", null, listOf("029"), listOf()),
  Madagascar("MG", "MG", "MDG", 450, "450", "Madagascar", null, listOf("014", "UN"), listOf()),
  MarshallIslands("MH", "MH", "MHL", 584, "584", "Marshall Islands", null, listOf("057", "UN"), listOf()),
  NorthMacedonia("MK", "MK", "MKD", 807, "807", "North Macedonia", null, listOf("039", "UN"), listOf()),
  Mali("ML", "ML", "MLI", 466, "466", "Mali", null, listOf("011", "UN"), listOf()),
  MyanmarBurma("MM", "MM", "MMR", 104, "104", "Myanmar (Burma)", "Myanmar", listOf("035", "UN"), listOf()),
  Mongolia("MN", "MN", "MNG", 496, "496", "Mongolia", null, listOf("030", "UN"), listOf()),
  Macao("MO", "MO", "MAC", 446, "446", "Macao SAR China", "Macao", listOf("030"), listOf()),
  NorthernMarianaIslands("MP", "MP", "MNP", 580, "580", "Northern Mariana Islands", null, listOf("057"), listOf()),
  Martinique("MQ", "MQ", "MTQ", 474, "474", "Martinique", null, listOf("029"), listOf()),
  Mauritania("MR", "MR", "MRT", 478, "478", "Mauritania", null, listOf("011", "UN"), listOf()),
  Montserrat("MS", "MS", "MSR", 500, "500", "Montserrat", null, listOf("029"), listOf()),
  Malta("MT", "MT", "MLT", 470, "470", "Malta", null, listOf("039", "EU", "EZ", "UN"), listOf()),
  Mauritius("MU", "MU", "MUS", 480, "480", "Mauritius", null, listOf("014", "UN"), listOf()),
  Maldives("MV", "MV", "MDV", 462, "462", "Maldives", null, listOf("034", "UN"), listOf()),
  Malawi("MW", "MW", "MWI", 454, "454", "Malawi", null, listOf("014", "UN"), listOf()),
  Mexico("MX", "MX", "MEX", 484, "484", "Mexico", null, listOf("013", "UN"), listOf()),
  Malaysia("MY", "MY", "MYS", 458, "458", "Malaysia", null, listOf("035", "UN"), listOf()),
  Mozambique("MZ", "MZ", "MOZ", 508, "508", "Mozambique", null, listOf("014", "UN"), listOf()),
  Namibia("NA", "NA", "NAM", 516, "516", "Namibia", null, listOf("018", "UN"), listOf()),
  NewCaledonia("NC", "NC", "NCL", 540, "540", "New Caledonia", null, listOf("054"), listOf()),
  Niger("NE", "NE", "NER", 562, "562", "Niger", null, listOf("011", "UN"), listOf()),
  NorfolkIsland("NF", "NF", "NFK", 574, "574", "Norfolk Island", null, listOf("053"), listOf()),
  Nigeria("NG", "NG", "NGA", 566, "566", "Nigeria", null, listOf("011", "UN"), listOf()),
  Nicaragua("NI", "NI", "NIC", 558, "558", "Nicaragua", null, listOf("013", "UN"), listOf()),
  Netherlands("NL", "NL", "NLD", 528, "528", "Netherlands", null, listOf("155", "EU", "EZ", "UN"), listOf()),
  Norway("NO", "NO", "NOR", 578, "578", "Norway", null, listOf("154", "UN"), listOf()),
  Nepal("NP", "NP", "NPL", 524, "524", "Nepal", null, listOf("034", "UN"), listOf()),
  Nauru("NR", "NR", "NRU", 520, "520", "Nauru", null, listOf("057", "UN"), listOf()),
  Niue("NU", "NU", "NIU", 570, "570", "Niue", null, listOf("061"), listOf()),
  NewZealand("NZ", "NZ", "NZL", 554, "554", "New Zealand", "Aotearoa New Zealand", listOf("053", "UN"), listOf()),
  Oman("OM", "OM", "OMN", 512, "512", "Oman", null, listOf("145", "UN"), listOf()),
  Panama("PA", "PA", "PAN", 591, "591", "Panama", null, listOf("013", "UN"), listOf()),
  Peru("PE", "PE", "PER", 604, "604", "Peru", null, listOf("005", "UN"), listOf()),
  FrenchPolynesia("PF", "PF", "PYF", 258, "258", "French Polynesia", null, listOf("061"), listOf()),
  PapuaNewGuinea("PG", "PG", "PNG", 598, "598", "Papua New Guinea", null, listOf("054", "UN"), listOf()),
  Philippines("PH", "PH", "PHL", 608, "608", "Philippines", null, listOf("035", "UN"), listOf()),
  Pakistan("PK", "PK", "PAK", 586, "586", "Pakistan", null, listOf("034", "UN"), listOf()),
  Poland("PL", "PL", "POL", 616, "616", "Poland", null, listOf("151", "EU", "UN"), listOf()),
  SaintPierreAndMiquelon("PM", "PM", "SPM", 666, "666", "St. Pierre & Miquelon", null, listOf("021"), listOf()),
  PitcairnIslands("PN", "PN", "PCN", 612, "612", "Pitcairn Islands", null, listOf("061"), listOf()),
  PuertoRico("PR", "PR", "PRI", 630, "630", "Puerto Rico", null, listOf("029"), listOf()),
  Palestine("PS", "PS", "PSE", 275, "275", "Palestinian Territories", "Palestine", listOf("145"), listOf()),
  Portugal("PT", "PT", "PRT", 620, "620", "Portugal", null, listOf("039", "EU", "EZ", "UN"), listOf()),
  Palau("PW", "PW", "PLW", 585, "585", "Palau", null, listOf("057", "UN"), listOf()),
  Paraguay("PY", "PY", "PRY", 600, "600", "Paraguay", null, listOf("005", "UN"), listOf()),
  Qatar("QA", "QA", "QAT", 634, "634", "Qatar", null, listOf("145", "UN"), listOf()),
  OutlyingOceania("QO", "QO", "QOO", 961, "961", "Outlying Oceania", null, listOf("009"), listOf("AQ", "AC", "CP", "DG", "TA")),
  Reunion("RE", "RE", "REU", 638, "638", "Réunion", null, listOf("014"), listOf()),
  Romania("RO", "RO", "ROU", 642, "642", "Romania", null, listOf("151", "EU", "UN"), listOf()),
  Serbia("RS", "RS", "SRB", 688, "688", "Serbia", null, listOf("039", "UN"), listOf()),
  Russia("RU", "RU", "RUS", 643, "643", "Russia", null, listOf("151", "UN"), listOf()),
  Rwanda("RW", "RW", "RWA", 646, "646", "Rwanda", null, listOf("014", "UN"), listOf()),
  SaudiArabia("SA", "SA", "SAU", 682, "682", "Saudi Arabia", null, listOf("145", "UN"), listOf()),
  SolomonIslands("SB", "SB", "SLB", 90, "090", "Solomon Islands", null, listOf("054", "UN"), listOf()),
  Seychelles("SC", "SC", "SYC", 690, "690", "Seychelles", null, listOf("014", "UN"), listOf()),
  Sudan("SD", "SD", "SDN", 729, "729", "Sudan", null, listOf("015", "UN"), listOf()),
  Sweden("SE", "SE", "SWE", 752, "752", "Sweden", null, listOf("154", "EU", "UN"), listOf()),
  Singapore("SG", "SG", "SGP", 702, "702", "Singapore", null, listOf("035", "UN"), listOf()),
  SaintHelena("SH", "SH", "SHN", 654, "654", "St. Helena", null, listOf("011"), listOf()),
  Slovenia("SI", "SI", "SVN", 705, "705", "Slovenia", null, listOf("039", "EU", "EZ", "UN"), listOf()),
  SvalbardAndJanMayen("SJ", "SJ", "SJM", 744, "744", "Svalbard & Jan Mayen", null, listOf("154"), listOf()),
  Slovakia("SK", "SK", "SVK", 703, "703", "Slovakia", null, listOf("151", "EU", "EZ", "UN"), listOf()),
  SierraLeone("SL", "SL", "SLE", 694, "694", "Sierra Leone", null, listOf("011", "UN"), listOf()),
  SanMarino("SM", "SM", "SMR", 674, "674", "San Marino", null, listOf("039", "UN"), listOf()),
  Senegal("SN", "SN", "SEN", 686, "686", "Senegal", null, listOf("011", "UN"), listOf()),
  Somalia("SO", "SO", "SOM", 706, "706", "Somalia", null, listOf("014", "UN"), listOf()),
  Suriname("SR", "SR", "SUR", 740, "740", "Suriname", null, listOf("005", "UN"), listOf()),
  SouthSudan("SS", "SS", "SSD", 728, "728", "South Sudan", null, listOf("014", "UN"), listOf()),
  SaoTomeAndPrincipe("ST", "ST", "STP", 678, "678", "São Tomé & Príncipe", null, listOf("017", "UN"), listOf()),
  ElSalvador("SV", "SV", "SLV", 222, "222", "El Salvador", null, listOf("013", "UN"), listOf()),
  SintMaarten("SX", "SX", "SXM", 534, "534", "Sint Maarten", null, listOf("029"), listOf()),
  Syria("SY", "SY", "SYR", 760, "760", "Syria", null, listOf("145", "UN"), listOf()),
  Eswatini("SZ", "SZ", "SWZ", 748, "748", "Eswatini", "Swaziland", listOf("018", "UN"), listOf()),
  TristanDaCunha("TA", "TA", "TAA", 654, "654", "Tristan da Cunha", null, listOf("QO"), listOf()),
  TurksAndCaicosIslands("TC", "TC", "TCA", 796, "796", "Turks & Caicos Islands", null, listOf("029"), listOf()),
  Chad("TD", "TD", "TCD", 148, "148", "Chad", null, listOf("017", "UN"), listOf()),
  FrenchSouthernTerritories("TF", "TF", "ATF", 260, "260", "French Southern Territories", null, listOf("014"), listOf()),
  Togo("TG", "TG", "TGO", 768, "768", "Togo", null, listOf("011", "UN"), listOf()),
  Thailand("TH", "TH", "THA", 764, "764", "Thailand", null, listOf("035", "UN"), listOf()),
  Tajikistan("TJ", "TJ", "TJK", 762, "762", "Tajikistan", null, listOf("143", "UN"), listOf()),
  Tokelau("TK", "TK", "TKL", 772, "772", "Tokelau", null, listOf("061"), listOf()),
  TimorLeste("TL", "TL", "TLS", 626, "626", "Timor-Leste", "East Timor", listOf("035", "UN"), listOf()),
  Turkmenistan("TM", "TM", "TKM", 795, "795", "Turkmenistan", null, listOf("143", "UN"), listOf()),
  Tunisia("TN", "TN", "TUN", 788, "788", "Tunisia", null, listOf("015", "UN"), listOf()),
  Tonga("TO", "TO", "TON", 776, "776", "Tonga", null, listOf("061", "UN"), listOf()),
  Turkiye("TR", "TR", "TUR", 792, "792", "Türkiye", "Turkey", listOf("145", "UN"), listOf()),
  TrinidadAndTobago("TT", "TT", "TTO", 780, "780", "Trinidad & Tobago", null, listOf("029", "UN"), listOf()),
  Tuvalu("TV", "TV", "TUV", 798, "798", "Tuvalu", null, listOf("061", "UN"), listOf()),
  Taiwan("TW", "TW", "TWN", 158, "158", "Taiwan", null, listOf("030"), listOf()),
  Tanzania("TZ", "TZ", "TZA", 834, "834", "Tanzania", null, listOf("014", "UN"), listOf()),
  Ukraine("UA", "UA", "UKR", 804, "804", "Ukraine", null, listOf("151", "UN"), listOf()),
  Uganda("UG", "UG", "UGA", 800, "800", "Uganda", null, listOf("014", "UN"), listOf()),
  USOutlyingIslands("UM", "UM", "UMI", 581, "581", "U.S. Outlying Islands", null, listOf("057"), listOf()),
  UnitedNations("UN", "UN", null, null, null, "United Nations", "UN", listOf("001-status-grouping"), listOf("AD", "AE", "AF", "AG", "AL", "AM", "AO", "AR", "AT", "AU", "AZ", "BA", "BB", "BD", "BE", "BF", "BG", "BH", "BI", "BJ", "BN", "BO", "BR", "BS", "BT", "BW", "BY", "BZ", "CA", "CD", "CF", "CG", "CH", "CI", "CL", "CM", "CN", "CO", "CR", "CU", "CV", "CY", "CZ", "DE", "DJ", "DK", "DM", "DO", "DZ", "EC", "EE", "EG", "ER", "ES", "ET", "FI", "FJ", "FM", "FR", "GA", "GB", "GD", "GE", "GH", "GM", "GN", "GQ", "GR", "GT", "GW", "GY", "HN", "HR", "HT", "HU", "ID", "IE", "IL", "IN", "IQ", "IR", "IS", "IT", "JM", "JO", "JP", "KE", "KG", "KH", "KI", "KM", "KN", "KP", "KR", "KW", "KZ", "LA", "LB", "LC", "LI", "LK", "LR", "LS", "LT", "LU", "LV", "LY", "MA", "MC", "MD", "ME", "MG", "MH", "MK", "ML", "MM", "MN", "MR", "MT", "MU", "MV", "MX", "MW", "MY", "MZ", "NA", "NE", "NG", "NI", "NL", "NO", "NR", "NP", "NZ", "OM", "PA", "PE", "PG", "PH", "PK", "PL", "PT", "PW", "PY", "QA", "RO", "RS", "RU", "RW", "SA", "SB", "SC", "SD", "SE", "SG", "SI", "SK", "SL", "SM", "SN", "SO", "SR", "SS", "ST", "SV", "SY", "SZ", "TD", "TG", "TH", "TJ", "TL", "TM", "TN", "TO", "TR", "TT", "TV", "TZ", "UA", "UG", "US", "UY", "UZ", "VC", "VE", "VN", "VU", "WS", "YE", "ZA", "ZM", "ZW")),
  UnitedStates("US", "US", "USA", 840, "840", "United States", "US", listOf("021", "UN"), listOf()),
  Uruguay("UY", "UY", "URY", 858, "858", "Uruguay", null, listOf("005", "UN"), listOf()),
  Uzbekistan("UZ", "UZ", "UZB", 860, "860", "Uzbekistan", null, listOf("143", "UN"), listOf()),
  VaticanCity("VA", "VA", "VAT", 336, "336", "Vatican City", null, listOf("039"), listOf()),
  SaintVincentAndGrenadines("VC", "VC", "VCT", 670, "670", "St. Vincent & Grenadines", null, listOf("029", "UN"), listOf()),
  Venezuela("VE", "VE", "VEN", 862, "862", "Venezuela", null, listOf("005", "UN"), listOf()),
  BritishVirginIslands("VG", "VG", "VGB", 92, "092", "British Virgin Islands", null, listOf("029"), listOf()),
  USVirginIslands("VI", "VI", "VIR", 850, "850", "U.S. Virgin Islands", null, listOf("029"), listOf()),
  Vietnam("VN", "VN", "VNM", 704, "704", "Vietnam", null, listOf("035", "UN"), listOf()),
  Vanuatu("VU", "VU", "VUT", 548, "548", "Vanuatu", null, listOf("054", "UN"), listOf()),
  WallisAndFutuna("WF", "WF", "WLF", 876, "876", "Wallis & Futuna", null, listOf("061"), listOf()),
  Samoa("WS", "WS", "WSM", 882, "882", "Samoa", null, listOf("061", "UN"), listOf()),
  PseudoAccents("XA", "XA", "XAA", 973, "973", "Pseudo-Accents", null, listOf(), listOf()),
  PseudoBidi("XB", "XB", "XBB", 974, "974", "Pseudo-Bidi", null, listOf(), listOf()),
  Kosovo("XK", "XK", "XKK", 983, "983", "Kosovo", null, listOf("039"), listOf()),
  Yemen("YE", "YE", "YEM", 887, "887", "Yemen", null, listOf("145", "UN"), listOf()),
  Mayotte("YT", "YT", "MYT", 175, "175", "Mayotte", null, listOf("014"), listOf()),
  SouthAfrica("ZA", "ZA", "ZAF", 710, "710", "South Africa", null, listOf("018", "UN"), listOf()),
  Zambia("ZM", "ZM", "ZMB", 894, "894", "Zambia", null, listOf("014", "UN"), listOf()),
  Zimbabwe("ZW", "ZW", "ZWE", 716, "716", "Zimbabwe", null, listOf("014", "UN"), listOf()),
  UnknownRegion("ZZ", "ZZ", "ZZZ", 999, "999", "Unknown Region", null, listOf(), listOf()),
}
