package net.codinux.i18n

import kotlin.Int
import kotlin.String

/**
 * @param code Either alpha-2 tor alpha-3 ISO code or numeric UN M.49 code.
 * @param alpha2Code alpha-2 two-letter country codes are "most prominently for the Internet's country code top-level
 * domains (with a few exceptions). They are also used as country identifiers extending the postal
 * code when appropriate within the international postal system for paper mail."
 *
 * User-assigned codes are: AA, QM to QZ, XA to XZ, and ZZ.
 * These can be freely used and will never be part of the standard.
 * @param alpha3Code alpha-3 three-letter country codes "allow a better visual association between the codes and the
 * country names than the alpha-2 codes. They represent countries, dependent territories, and special
 * areas of geographical interest. (...) They are used most prominently in ISO/IEC 7501-1 for
 * machine-readable passports."
 *
 * User-assigned codes are: AAA to AAZ, QMA to QZZ, XAA to XZZ, and ZZA to ZZZ.
 * These can be freely used and will never be part of the standard.
 * @param numericCode ISO 3166-1 `numeric` codes are three-digit country codes that originate from
 * [UN M.49](https://en.wikipedia.org/wiki/UN_M.49) standard, with the advantage of script (writing system)
 * independence, and hence useful for people or systems using non-Latin scripts such as Arabic or Chinese.
 *
 * The UN M.49 contains also codes for geographical and political regions like a continent and therefore
 * allow for hierarchical mapping of regions.
 *
 * User-assigned codes range from 900 to 999. These are reserved for users to add custom geographical names 
 * in their applications and will never be used by the ISO 3166 standard.
 * @param numericCodeAsString The value of [numericCode] as String, padded with zero to three digits.
 * @param englishName English name of the country or region.
 * @param variantName Optional a variant of the English name of the country or region (if available).
 */
enum class Region(
  val code: String,
  val alpha2Code: String?,
  val alpha3Code: String?,
  val numericCode: Int?,
  val numericCodeAsString: String?,
  val englishName: String,
  val variantName: String?,
) {
  World("001", null, null, 1, "001", "world", null),
  Africa("002", null, null, 2, "002", "Africa", null),
  NorthAmerica("003", null, null, 3, "003", "North America", null),
  SouthAmerica("005", null, null, 5, "005", "South America", null),
  Oceania("009", null, null, 9, "009", "Oceania", null),
  WesternAfrica("011", null, null, 11, "011", "Western Africa", null),
  CentralAmerica("013", null, null, 13, "013", "Central America", null),
  EasternAfrica("014", null, null, 14, "014", "Eastern Africa", null),
  NorthernAfrica("015", null, null, 15, "015", "Northern Africa", null),
  MiddleAfrica("017", null, null, 17, "017", "Middle Africa", null),
  SouthernAfrica("018", null, null, 18, "018", "Southern Africa", null),
  Americas("019", null, null, 19, "019", "Americas", null),
  NorthernAmerica("021", null, null, 21, "021", "Northern America", null),
  Caribbean("029", null, null, 29, "029", "Caribbean", null),
  EasternAsia("030", null, null, 30, "030", "Eastern Asia", null),
  SouthernAsia("034", null, null, 34, "034", "Southern Asia", null),
  SoutheastAsia("035", null, null, 35, "035", "Southeast Asia", null),
  SouthernEurope("039", null, null, 39, "039", "Southern Europe", null),
  Australasia("053", null, null, 53, "053", "Australasia", null),
  Melanesia("054", null, null, 54, "054", "Melanesia", null),
  MicronesianRegion("057", null, null, 57, "057", "Micronesian Region", null),
  Polynesia("061", null, null, 61, "061", "Polynesia", null),
  Asia("142", null, null, 142, "142", "Asia", null),
  CentralAsia("143", null, null, 143, "143", "Central Asia", null),
  WesternAsia("145", null, null, 145, "145", "Western Asia", null),
  Europe("150", null, null, 150, "150", "Europe", null),
  EasternEurope("151", null, null, 151, "151", "Eastern Europe", null),
  NorthernEurope("154", null, null, 154, "154", "Northern Europe", null),
  WesternEurope("155", null, null, 155, "155", "Western Europe", null),
  SubSaharanAfrica("202", null, null, 202, "202", "Sub-Saharan Africa", null),
  LatinAmerica("419", null, null, 419, "419", "Latin America", null),
  AscensionIsland("AC", "AC", "ASC", 654, "654", "Ascension Island", null),
  Andorra("AD", "AD", "AND", 20, "020", "Andorra", null),
  UnitedArabEmirates("AE", "AE", "ARE", 784, "784", "United Arab Emirates", null),
  Afghanistan("AF", "AF", "AFG", 4, "004", "Afghanistan", null),
  AntiguaAndBarbuda("AG", "AG", "ATG", 28, "028", "Antigua & Barbuda", null),
  Anguilla("AI", "AI", "AIA", 660, "660", "Anguilla", null),
  Albania("AL", "AL", "ALB", 8, "008", "Albania", null),
  Armenia("AM", "AM", "ARM", 51, "051", "Armenia", null),
  Angola("AO", "AO", "AGO", 24, "024", "Angola", null),
  Antarctica("AQ", "AQ", "ATA", 10, "010", "Antarctica", null),
  Argentina("AR", "AR", "ARG", 32, "032", "Argentina", null),
  AmericanSamoa("AS", "AS", "ASM", 16, "016", "American Samoa", null),
  Austria("AT", "AT", "AUT", 40, "040", "Austria", null),
  Australia("AU", "AU", "AUS", 36, "036", "Australia", null),
  Aruba("AW", "AW", "ABW", 533, "533", "Aruba", null),
  AlandIslands("AX", "AX", "ALA", 248, "248", "Åland Islands", null),
  Azerbaijan("AZ", "AZ", "AZE", 31, "031", "Azerbaijan", null),
  BosniaAndHerzegovina("BA", "BA", "BIH", 70, "070", "Bosnia & Herzegovina", "Bosnia"),
  Barbados("BB", "BB", "BRB", 52, "052", "Barbados", null),
  Bangladesh("BD", "BD", "BGD", 50, "050", "Bangladesh", null),
  Belgium("BE", "BE", "BEL", 56, "056", "Belgium", null),
  BurkinaFaso("BF", "BF", "BFA", 854, "854", "Burkina Faso", null),
  Bulgaria("BG", "BG", "BGR", 100, "100", "Bulgaria", null),
  Bahrain("BH", "BH", "BHR", 48, "048", "Bahrain", null),
  Burundi("BI", "BI", "BDI", 108, "108", "Burundi", null),
  Benin("BJ", "BJ", "BEN", 204, "204", "Benin", null),
  SaintBarthelemy("BL", "BL", "BLM", 652, "652", "St. Barthélemy", null),
  Bermuda("BM", "BM", "BMU", 60, "060", "Bermuda", null),
  Brunei("BN", "BN", "BRN", 96, "096", "Brunei", null),
  Bolivia("BO", "BO", "BOL", 68, "068", "Bolivia", null),
  CaribbeanNetherlands("BQ", "BQ", "BES", 535, "535", "Caribbean Netherlands", null),
  Brazil("BR", "BR", "BRA", 76, "076", "Brazil", null),
  Bahamas("BS", "BS", "BHS", 44, "044", "Bahamas", null),
  Bhutan("BT", "BT", "BTN", 64, "064", "Bhutan", null),
  BouvetIsland("BV", "BV", "BVT", 74, "074", "Bouvet Island", null),
  Botswana("BW", "BW", "BWA", 72, "072", "Botswana", null),
  Belarus("BY", "BY", "BLR", 112, "112", "Belarus", null),
  Belize("BZ", "BZ", "BLZ", 84, "084", "Belize", null),
  Canada("CA", "CA", "CAN", 124, "124", "Canada", null),
  CocosKeelingIslands("CC", "CC", "CCK", 166, "166", "Cocos (Keeling) Islands", null),
  CongoDemocraticRepublic("CD", "CD", "COD", 180, "180", "Congo - Kinshasa", "Congo (DRC)"),
  CentralAfricanRepublic("CF", "CF", "CAF", 140, "140", "Central African Republic", null),
  Congo("CG", "CG", "COG", 178, "178", "Congo - Brazzaville", "Congo (Republic)"),
  Switzerland("CH", "CH", "CHE", 756, "756", "Switzerland", null),
  CoteDIvoire("CI", "CI", "CIV", 384, "384", "Côte d’Ivoire", "Ivory Coast"),
  CookIslands("CK", "CK", "COK", 184, "184", "Cook Islands", null),
  Chile("CL", "CL", "CHL", 152, "152", "Chile", null),
  Cameroon("CM", "CM", "CMR", 120, "120", "Cameroon", null),
  China("CN", "CN", "CHN", 156, "156", "China", null),
  Colombia("CO", "CO", "COL", 170, "170", "Colombia", null),
  ClippertonIsland("CP", "CP", "CPT", null, null, "Clipperton Island", null),
  Sark("CQ", "CQ", null, null, null, "Sark", null),
  CostaRica("CR", "CR", "CRI", 188, "188", "Costa Rica", null),
  Cuba("CU", "CU", "CUB", 192, "192", "Cuba", null),
  CapeVerde("CV", "CV", "CPV", 132, "132", "Cape Verde", "Cabo Verde"),
  Curacao("CW", "CW", "CUW", 531, "531", "Curaçao", null),
  ChristmasIsland("CX", "CX", "CXR", 162, "162", "Christmas Island", null),
  Cyprus("CY", "CY", "CYP", 196, "196", "Cyprus", null),
  Czechia("CZ", "CZ", "CZE", 203, "203", "Czechia", "Czech Republic"),
  Germany("DE", "DE", "DEU", 276, "276", "Germany", null),
  DiegoGarcia("DG", "DG", "DGA", null, null, "Diego Garcia", null),
  Djibouti("DJ", "DJ", "DJI", 262, "262", "Djibouti", null),
  Denmark("DK", "DK", "DNK", 208, "208", "Denmark", null),
  Dominica("DM", "DM", "DMA", 212, "212", "Dominica", null),
  DominicanRepublic("DO", "DO", "DOM", 214, "214", "Dominican Republic", null),
  Algeria("DZ", "DZ", "DZA", 12, "012", "Algeria", null),
  CeutaAndMelilla("EA", "EA", null, null, null, "Ceuta & Melilla", null),
  Ecuador("EC", "EC", "ECU", 218, "218", "Ecuador", null),
  Estonia("EE", "EE", "EST", 233, "233", "Estonia", null),
  Egypt("EG", "EG", "EGY", 818, "818", "Egypt", null),
  WesternSahara("EH", "EH", "ESH", 732, "732", "Western Sahara", null),
  Eritrea("ER", "ER", "ERI", 232, "232", "Eritrea", null),
  Spain("ES", "ES", "ESP", 724, "724", "Spain", null),
  Ethiopia("ET", "ET", "ETH", 231, "231", "Ethiopia", null),
  EuropeanUnion("EU", "EU", "QUU", 967, "967", "European Union", null),
  Eurozone("EZ", "EZ", null, null, null, "Eurozone", null),
  Finland("FI", "FI", "FIN", 246, "246", "Finland", null),
  Fiji("FJ", "FJ", "FJI", 242, "242", "Fiji", null),
  FalklandIslands("FK", "FK", "FLK", 238, "238", "Falkland Islands", "Falkland Islands (Islas Malvinas)"),
  Micronesia("FM", "FM", "FSM", 583, "583", "Micronesia", null),
  FaroeIslands("FO", "FO", "FRO", 234, "234", "Faroe Islands", null),
  France("FR", "FR", "FRA", 250, "250", "France", null),
  Gabon("GA", "GA", "GAB", 266, "266", "Gabon", null),
  UnitedKingdom("GB", "GB", "GBR", 826, "826", "United Kingdom", "UK"),
  Grenada("GD", "GD", "GRD", 308, "308", "Grenada", null),
  Georgia("GE", "GE", "GEO", 268, "268", "Georgia", null),
  FrenchGuiana("GF", "GF", "GUF", 254, "254", "French Guiana", null),
  Guernsey("GG", "GG", "GGY", 831, "831", "Guernsey", null),
  Ghana("GH", "GH", "GHA", 288, "288", "Ghana", null),
  Gibraltar("GI", "GI", "GIB", 292, "292", "Gibraltar", null),
  Greenland("GL", "GL", "GRL", 304, "304", "Greenland", null),
  Gambia("GM", "GM", "GMB", 270, "270", "Gambia", null),
  Guinea("GN", "GN", "GIN", 324, "324", "Guinea", null),
  Guadeloupe("GP", "GP", "GLP", 312, "312", "Guadeloupe", null),
  EquatorialGuinea("GQ", "GQ", "GNQ", 226, "226", "Equatorial Guinea", null),
  Greece("GR", "GR", "GRC", 300, "300", "Greece", null),
  SouthGeorgiaAndSouthSandwichIslands("GS", "GS", "SGS", 239, "239", "South Georgia & South Sandwich Islands", null),
  Guatemala("GT", "GT", "GTM", 320, "320", "Guatemala", null),
  Guam("GU", "GU", "GUM", 316, "316", "Guam", null),
  GuineaBissau("GW", "GW", "GNB", 624, "624", "Guinea-Bissau", null),
  Guyana("GY", "GY", "GUY", 328, "328", "Guyana", null),
  HongKong("HK", "HK", "HKG", 344, "344", "Hong Kong SAR China", "Hong Kong"),
  HeardAndMcDonaldIslands("HM", "HM", "HMD", 334, "334", "Heard & McDonald Islands", null),
  Honduras("HN", "HN", "HND", 340, "340", "Honduras", null),
  Croatia("HR", "HR", "HRV", 191, "191", "Croatia", null),
  Haiti("HT", "HT", "HTI", 332, "332", "Haiti", null),
  Hungary("HU", "HU", "HUN", 348, "348", "Hungary", null),
  CanaryIslands("IC", "IC", null, null, null, "Canary Islands", null),
  Indonesia("ID", "ID", "IDN", 360, "360", "Indonesia", null),
  Ireland("IE", "IE", "IRL", 372, "372", "Ireland", null),
  Israel("IL", "IL", "ISR", 376, "376", "Israel", null),
  IsleOfMan("IM", "IM", "IMN", 833, "833", "Isle of Man", null),
  India("IN", "IN", "IND", 356, "356", "India", null),
  BritishIndianOceanTerritory("IO", "IO", "IOT", 86, "086", "British Indian Ocean Territory", null),
  Iraq("IQ", "IQ", "IRQ", 368, "368", "Iraq", null),
  Iran("IR", "IR", "IRN", 364, "364", "Iran", null),
  Iceland("IS", "IS", "ISL", 352, "352", "Iceland", null),
  Italy("IT", "IT", "ITA", 380, "380", "Italy", null),
  Jersey("JE", "JE", "JEY", 832, "832", "Jersey", null),
  Jamaica("JM", "JM", "JAM", 388, "388", "Jamaica", null),
  Jordan("JO", "JO", "JOR", 400, "400", "Jordan", null),
  Japan("JP", "JP", "JPN", 392, "392", "Japan", null),
  Kenya("KE", "KE", "KEN", 404, "404", "Kenya", null),
  Kyrgyzstan("KG", "KG", "KGZ", 417, "417", "Kyrgyzstan", null),
  Cambodia("KH", "KH", "KHM", 116, "116", "Cambodia", null),
  Kiribati("KI", "KI", "KIR", 296, "296", "Kiribati", null),
  Comoros("KM", "KM", "COM", 174, "174", "Comoros", null),
  SaintKittsAndNevis("KN", "KN", "KNA", 659, "659", "St. Kitts & Nevis", null),
  NorthKorea("KP", "KP", "PRK", 408, "408", "North Korea", null),
  SouthKorea("KR", "KR", "KOR", 410, "410", "South Korea", null),
  Kuwait("KW", "KW", "KWT", 414, "414", "Kuwait", null),
  CaymanIslands("KY", "KY", "CYM", 136, "136", "Cayman Islands", null),
  Kazakhstan("KZ", "KZ", "KAZ", 398, "398", "Kazakhstan", null),
  Laos("LA", "LA", "LAO", 418, "418", "Laos", null),
  Lebanon("LB", "LB", "LBN", 422, "422", "Lebanon", null),
  SaintLucia("LC", "LC", "LCA", 662, "662", "St. Lucia", null),
  Liechtenstein("LI", "LI", "LIE", 438, "438", "Liechtenstein", null),
  SriLanka("LK", "LK", "LKA", 144, "144", "Sri Lanka", null),
  Liberia("LR", "LR", "LBR", 430, "430", "Liberia", null),
  Lesotho("LS", "LS", "LSO", 426, "426", "Lesotho", null),
  Lithuania("LT", "LT", "LTU", 440, "440", "Lithuania", null),
  Luxembourg("LU", "LU", "LUX", 442, "442", "Luxembourg", null),
  Latvia("LV", "LV", "LVA", 428, "428", "Latvia", null),
  Libya("LY", "LY", "LBY", 434, "434", "Libya", null),
  Morocco("MA", "MA", "MAR", 504, "504", "Morocco", null),
  Monaco("MC", "MC", "MCO", 492, "492", "Monaco", null),
  Moldova("MD", "MD", "MDA", 498, "498", "Moldova", null),
  Montenegro("ME", "ME", "MNE", 499, "499", "Montenegro", null),
  SaintMartin("MF", "MF", "MAF", 663, "663", "St. Martin", null),
  Madagascar("MG", "MG", "MDG", 450, "450", "Madagascar", null),
  MarshallIslands("MH", "MH", "MHL", 584, "584", "Marshall Islands", null),
  NorthMacedonia("MK", "MK", "MKD", 807, "807", "North Macedonia", null),
  Mali("ML", "ML", "MLI", 466, "466", "Mali", null),
  MyanmarBurma("MM", "MM", "MMR", 104, "104", "Myanmar (Burma)", "Myanmar"),
  Mongolia("MN", "MN", "MNG", 496, "496", "Mongolia", null),
  Macao("MO", "MO", "MAC", 446, "446", "Macao SAR China", "Macao"),
  NorthernMarianaIslands("MP", "MP", "MNP", 580, "580", "Northern Mariana Islands", null),
  Martinique("MQ", "MQ", "MTQ", 474, "474", "Martinique", null),
  Mauritania("MR", "MR", "MRT", 478, "478", "Mauritania", null),
  Montserrat("MS", "MS", "MSR", 500, "500", "Montserrat", null),
  Malta("MT", "MT", "MLT", 470, "470", "Malta", null),
  Mauritius("MU", "MU", "MUS", 480, "480", "Mauritius", null),
  Maldives("MV", "MV", "MDV", 462, "462", "Maldives", null),
  Malawi("MW", "MW", "MWI", 454, "454", "Malawi", null),
  Mexico("MX", "MX", "MEX", 484, "484", "Mexico", null),
  Malaysia("MY", "MY", "MYS", 458, "458", "Malaysia", null),
  Mozambique("MZ", "MZ", "MOZ", 508, "508", "Mozambique", null),
  Namibia("NA", "NA", "NAM", 516, "516", "Namibia", null),
  NewCaledonia("NC", "NC", "NCL", 540, "540", "New Caledonia", null),
  Niger("NE", "NE", "NER", 562, "562", "Niger", null),
  NorfolkIsland("NF", "NF", "NFK", 574, "574", "Norfolk Island", null),
  Nigeria("NG", "NG", "NGA", 566, "566", "Nigeria", null),
  Nicaragua("NI", "NI", "NIC", 558, "558", "Nicaragua", null),
  Netherlands("NL", "NL", "NLD", 528, "528", "Netherlands", null),
  Norway("NO", "NO", "NOR", 578, "578", "Norway", null),
  Nepal("NP", "NP", "NPL", 524, "524", "Nepal", null),
  Nauru("NR", "NR", "NRU", 520, "520", "Nauru", null),
  Niue("NU", "NU", "NIU", 570, "570", "Niue", null),
  NewZealand("NZ", "NZ", "NZL", 554, "554", "New Zealand", "Aotearoa New Zealand"),
  Oman("OM", "OM", "OMN", 512, "512", "Oman", null),
  Panama("PA", "PA", "PAN", 591, "591", "Panama", null),
  Peru("PE", "PE", "PER", 604, "604", "Peru", null),
  FrenchPolynesia("PF", "PF", "PYF", 258, "258", "French Polynesia", null),
  PapuaNewGuinea("PG", "PG", "PNG", 598, "598", "Papua New Guinea", null),
  Philippines("PH", "PH", "PHL", 608, "608", "Philippines", null),
  Pakistan("PK", "PK", "PAK", 586, "586", "Pakistan", null),
  Poland("PL", "PL", "POL", 616, "616", "Poland", null),
  SaintPierreAndMiquelon("PM", "PM", "SPM", 666, "666", "St. Pierre & Miquelon", null),
  PitcairnIslands("PN", "PN", "PCN", 612, "612", "Pitcairn Islands", null),
  PuertoRico("PR", "PR", "PRI", 630, "630", "Puerto Rico", null),
  Palestine("PS", "PS", "PSE", 275, "275", "Palestinian Territories", "Palestine"),
  Portugal("PT", "PT", "PRT", 620, "620", "Portugal", null),
  Palau("PW", "PW", "PLW", 585, "585", "Palau", null),
  Paraguay("PY", "PY", "PRY", 600, "600", "Paraguay", null),
  Qatar("QA", "QA", "QAT", 634, "634", "Qatar", null),
  OutlyingOceania("QO", "QO", "QOO", 961, "961", "Outlying Oceania", null),
  Reunion("RE", "RE", "REU", 638, "638", "Réunion", null),
  Romania("RO", "RO", "ROU", 642, "642", "Romania", null),
  Serbia("RS", "RS", "SRB", 688, "688", "Serbia", null),
  Russia("RU", "RU", "RUS", 643, "643", "Russia", null),
  Rwanda("RW", "RW", "RWA", 646, "646", "Rwanda", null),
  SaudiArabia("SA", "SA", "SAU", 682, "682", "Saudi Arabia", null),
  SolomonIslands("SB", "SB", "SLB", 90, "090", "Solomon Islands", null),
  Seychelles("SC", "SC", "SYC", 690, "690", "Seychelles", null),
  Sudan("SD", "SD", "SDN", 729, "729", "Sudan", null),
  Sweden("SE", "SE", "SWE", 752, "752", "Sweden", null),
  Singapore("SG", "SG", "SGP", 702, "702", "Singapore", null),
  SaintHelena("SH", "SH", "SHN", 654, "654", "St. Helena", null),
  Slovenia("SI", "SI", "SVN", 705, "705", "Slovenia", null),
  SvalbardAndJanMayen("SJ", "SJ", "SJM", 744, "744", "Svalbard & Jan Mayen", null),
  Slovakia("SK", "SK", "SVK", 703, "703", "Slovakia", null),
  SierraLeone("SL", "SL", "SLE", 694, "694", "Sierra Leone", null),
  SanMarino("SM", "SM", "SMR", 674, "674", "San Marino", null),
  Senegal("SN", "SN", "SEN", 686, "686", "Senegal", null),
  Somalia("SO", "SO", "SOM", 706, "706", "Somalia", null),
  Suriname("SR", "SR", "SUR", 740, "740", "Suriname", null),
  SouthSudan("SS", "SS", "SSD", 728, "728", "South Sudan", null),
  SaoTomeAndPrincipe("ST", "ST", "STP", 678, "678", "São Tomé & Príncipe", null),
  ElSalvador("SV", "SV", "SLV", 222, "222", "El Salvador", null),
  SintMaarten("SX", "SX", "SXM", 534, "534", "Sint Maarten", null),
  Syria("SY", "SY", "SYR", 760, "760", "Syria", null),
  Eswatini("SZ", "SZ", "SWZ", 748, "748", "Eswatini", "Swaziland"),
  TristanDaCunha("TA", "TA", "TAA", 654, "654", "Tristan da Cunha", null),
  TurksAndCaicosIslands("TC", "TC", "TCA", 796, "796", "Turks & Caicos Islands", null),
  Chad("TD", "TD", "TCD", 148, "148", "Chad", null),
  FrenchSouthernTerritories("TF", "TF", "ATF", 260, "260", "French Southern Territories", null),
  Togo("TG", "TG", "TGO", 768, "768", "Togo", null),
  Thailand("TH", "TH", "THA", 764, "764", "Thailand", null),
  Tajikistan("TJ", "TJ", "TJK", 762, "762", "Tajikistan", null),
  Tokelau("TK", "TK", "TKL", 772, "772", "Tokelau", null),
  TimorLeste("TL", "TL", "TLS", 626, "626", "Timor-Leste", "East Timor"),
  Turkmenistan("TM", "TM", "TKM", 795, "795", "Turkmenistan", null),
  Tunisia("TN", "TN", "TUN", 788, "788", "Tunisia", null),
  Tonga("TO", "TO", "TON", 776, "776", "Tonga", null),
  Turkiye("TR", "TR", "TUR", 792, "792", "Türkiye", "Turkey"),
  TrinidadAndTobago("TT", "TT", "TTO", 780, "780", "Trinidad & Tobago", null),
  Tuvalu("TV", "TV", "TUV", 798, "798", "Tuvalu", null),
  Taiwan("TW", "TW", "TWN", 158, "158", "Taiwan", null),
  Tanzania("TZ", "TZ", "TZA", 834, "834", "Tanzania", null),
  Ukraine("UA", "UA", "UKR", 804, "804", "Ukraine", null),
  Uganda("UG", "UG", "UGA", 800, "800", "Uganda", null),
  USOutlyingIslands("UM", "UM", "UMI", 581, "581", "U.S. Outlying Islands", null),
  UnitedNations("UN", "UN", null, null, null, "United Nations", "UN"),
  UnitedStates("US", "US", "USA", 840, "840", "United States", "US"),
  Uruguay("UY", "UY", "URY", 858, "858", "Uruguay", null),
  Uzbekistan("UZ", "UZ", "UZB", 860, "860", "Uzbekistan", null),
  VaticanCity("VA", "VA", "VAT", 336, "336", "Vatican City", null),
  SaintVincentAndGrenadines("VC", "VC", "VCT", 670, "670", "St. Vincent & Grenadines", null),
  Venezuela("VE", "VE", "VEN", 862, "862", "Venezuela", null),
  BritishVirginIslands("VG", "VG", "VGB", 92, "092", "British Virgin Islands", null),
  USVirginIslands("VI", "VI", "VIR", 850, "850", "U.S. Virgin Islands", null),
  Vietnam("VN", "VN", "VNM", 704, "704", "Vietnam", null),
  Vanuatu("VU", "VU", "VUT", 548, "548", "Vanuatu", null),
  WallisAndFutuna("WF", "WF", "WLF", 876, "876", "Wallis & Futuna", null),
  Samoa("WS", "WS", "WSM", 882, "882", "Samoa", null),
  PseudoAccents("XA", "XA", "XAA", 973, "973", "Pseudo-Accents", null),
  PseudoBidi("XB", "XB", "XBB", 974, "974", "Pseudo-Bidi", null),
  Kosovo("XK", "XK", "XKK", 983, "983", "Kosovo", null),
  Yemen("YE", "YE", "YEM", 887, "887", "Yemen", null),
  Mayotte("YT", "YT", "MYT", 175, "175", "Mayotte", null),
  SouthAfrica("ZA", "ZA", "ZAF", 710, "710", "South Africa", null),
  Zambia("ZM", "ZM", "ZMB", 894, "894", "Zambia", null),
  Zimbabwe("ZW", "ZW", "ZWE", 716, "716", "Zimbabwe", null),
  UnknownRegion("ZZ", "ZZ", "ZZZ", 999, "999", "Unknown Region", null),
}
