package net.coding.chenxiaobo.spring.data.jpa;

import net.coding.chenxiaobo.spring.data.jpa.extension.ConditionRestriction;
import net.coding.chenxiaobo.spring.data.jpa.extension.restriction.support.*;
import net.coding.chenxiaobo.spring.data.jpa.filter.PropertyFilter;
import net.coding.chenxiaobo.spring.data.jpa.filter.SpecificationSupport;
import org.springframework.data.jpa.domain.Specification;

import javax.persistence.criteria.Path;
import javax.persistence.criteria.Root;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 构造 {@link org.springframework.data.jpa.domain.Specification} 的工具类，通过该类，能够将
 * {@link net.coding.chenxiaobo.spring.data.jpa.filter.PropertyFilter} 构造成
 * {@link org.springframework.data.jpa.domain.Specification} 需要的数据
 *
 * @see Specifications#get(String, String)
 * @see Specifications#get(java.util.List)
 * @see Specifications#get(net.coding.chenxiaobo.spring.data.jpa.filter.PropertyFilter)
 *
 * @author maurice
 */
public class Specifications {

    // 所有条件约束集合
    private static Map<String, ConditionRestriction> builderMap = new HashMap<String, ConditionRestriction>();

    static {
        ConditionRestriction eqRestriction = new EqRestriction();
        ConditionRestriction neRestriction = new NeRestriction();
        ConditionRestriction geRestriction = new GeRestriction();
        ConditionRestriction gtRestriction = new GtRestriction();
        ConditionRestriction inRestriction = new InRestriction();
        ConditionRestriction lLikeRestriction = new LLikeRestriction();
        ConditionRestriction leRestriction = new LeRestriction();
        ConditionRestriction likeRestriction = new LikeRestriction();
        ConditionRestriction ltRestriction = new LtRestriction();
        ConditionRestriction notInRestriction = new NinRestriction();
        ConditionRestriction rLikeRestriction = new RLikeRestriction();

        builderMap.put(eqRestriction.getRestrictionName(), eqRestriction);
        builderMap.put(neRestriction.getRestrictionName(), neRestriction);
        builderMap.put(geRestriction.getRestrictionName(), geRestriction);
        builderMap.put(inRestriction.getRestrictionName(), inRestriction);
        builderMap.put(gtRestriction.getRestrictionName(), gtRestriction);
        builderMap.put(lLikeRestriction.getRestrictionName(), lLikeRestriction);
        builderMap.put(leRestriction.getRestrictionName(), leRestriction);
        builderMap.put(likeRestriction.getRestrictionName(), likeRestriction);
        builderMap.put(ltRestriction.getRestrictionName(), ltRestriction);
        builderMap.put(rLikeRestriction.getRestrictionName(), rLikeRestriction);
        builderMap.put(notInRestriction.getRestrictionName(), notInRestriction);
    }

    public static ConditionRestriction getRestriction(String name) {
        return builderMap.get(name);
    }

    /**
     * 获取属性名子路径
     *
     * @param property 属性名
     * @param root Query roots always reference entities
     *
     * @return {@link javax.persistence.criteria.Path}
     */
    public static Path<?> getPath(String property,Root<?> root) {

        Path<?> path = null;

        if (property.contains(".")) {
            String[] properties = property.split(".");
            path = root.get(properties[0]);
            for (int i = 1; i < properties.length; i++) {
                path = path.get(properties[i]);
            }
        } else {
            path = root.get(property);
        }

        return path;
    }

    /**
     * 获取 {@link org.springframework.data.jpa.domain.Specification}
     *
     * @param filters 属性过滤器集合
     * @param <T> 持久化类型
     *
     * @return {@link net.coding.chenxiaobo.spring.data.jpa.filter.SpecificationSupport}
     */
    public static <T> Specification<T> get(List<PropertyFilter> filters) {
        return new SpecificationSupport<T>(filters);
    }

    /**
     * 获取 {@link org.springframework.data.jpa.domain.Specification}
     *
     * @param filter 属性过滤器
     * @param <T> 持久化类型
     *
     * @return {@link net.coding.chenxiaobo.spring.data.jpa.filter.SpecificationSupport}
     */
    public static <T> Specification<T> get(PropertyFilter filter) {
        List<PropertyFilter> filters = new ArrayList<PropertyFilter>();
        filters.add(filter);
        return get(filters);
    }

    /**
     * 获取 {@link org.springframework.data.jpa.domain.Specification}
     *
     * @param filter 属性表达式
     * @param value 值表达式
     * @param <T> 持久化类型
     *
     * @return {@link net.coding.chenxiaobo.spring.data.jpa.filter.SpecificationSupport}
     */
    public static <T> Specification<T> get(String filter,String value) {
        return get(PropertyFilter.get(filter,value));
    }

}
