package net.coding.chenxiaobo.spring.data.jpa.filter;

import net.coding.chenxiaobo.spring.data.jpa.FieldType;
import org.apache.commons.lang3.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.io.Serializable;
import java.util.*;

/**
 *
 * 属性过滤器
 *
 * @author maurice
 */
public class PropertyFilter implements Serializable{

    /**
	 * 
	 */
	private static final long serialVersionUID = 4554768110883277471L;
	
	// 约束名称
    private String restrictionName;
    // 字段属性名称
    private String[] property;
    // 值表达式
    private ValueExpression valueExpression;

    /**
     * 属性过滤器
     *
     * @param restrictionName 约束名称
     * @param type 值类型
     * @param property 字段属性名称
     * @param valueExpression 值表达式
     *
     */
    public PropertyFilter(String restrictionName, Class<?> type, String[] property, String valueExpression) {
        this.restrictionName = restrictionName;
        this.property = property;
        this.valueExpression = new ValueExpression(valueExpression, type);
    }

    /**
     * 获取约束名称
     *
     * @return 约束名称
     */
    public String getRestrictionName() {
        return restrictionName;
    }

    /**
     * 获取值表达式
     *
     * @return 值表达式
     */
    public ValueExpression getValueExpression() {
        return valueExpression;
    }

    /**
     * 获取属性
     *
     * @return 属性数组
     */
    public String[] getProperties() {
        return property;
    }

    /**
     * 获取属性
     *
     * @return 单个属性，如果存在多个，则抛出异常
     */
    public String getProperty() {
        String[] result = getProperties();
        if(result.length > 1) {
            throw new RuntimeException("属性过滤器存在1个以上的过滤属性:" + result);
        }
        return result[0];
    }

    /**
     * 判断是否多属性
     *
     * @return true 表是多属性，否则 false
     */
    public boolean isMultipleProperties() {
        return property != null && property.length > 1;
    }

    /**
     * 通过属性表达式和值表达获取属性过滤器
     *
     * @param filter 属性表达式
     * @param value 值表达
     *
     * @return 属性过滤器
     */
    @SuppressWarnings("static-access")
	public static PropertyFilter get(String filter,String value) {

        String nameAndClassType = filter.substring(0,filter.indexOf("_"));

        String name = StringUtils.substring(nameAndClassType, 0, nameAndClassType.length() - 1);
        String classType = StringUtils.substring(nameAndClassType, nameAndClassType.length() - 1, nameAndClassType.length());

        FieldType FieldType = null;

        try {
            FieldType = FieldType.valueOf(classType);
        } catch (Exception e) {
            throw new IllegalAccessError("[" + filter + "]表达式找不到相应的属性类型,获取的值为:" + classType);
        }

        String[] property;

        if (filter.contains("_OR_")) {
            String temp = StringUtils.substringAfter(filter, nameAndClassType + "_");
            property = StringUtils.splitByWholeSeparator(temp, "_OR_");;
        } else {
            property = new String[]{ filter.substring(filter.indexOf("_") + 1, filter.length()) };
        }

        return new PropertyFilter(name, FieldType.getValue(), property, value);
    }

    /**
     * 通过 {@link javax.servlet.http.HttpServletRequest} 获取属性过滤器集合
     *
     * @param request http servlet request
     *
     * @return 属性过滤器集合
     */
    public static List<PropertyFilter> get(HttpServletRequest request) {
        return get(request,"filter_");
    }

    /**
     * 通过 {@link javax.servlet.http.HttpServletRequest} 获取属性过滤器集合
     *
     * @param request http servlet request
     * @param prefix 参数前缀
     *
     * @return 属性过滤器集合
     */
    public static List<PropertyFilter> get(HttpServletRequest request, String prefix) {
        Enumeration<?> paramNames = request.getParameterNames();
        Map<String, Object> params = new TreeMap<String, Object>();
        if (prefix == null) {
            prefix = "";
        }
        while (paramNames != null && paramNames.hasMoreElements()) {
            String paramName = (String) paramNames.nextElement();
            if ("".equals(prefix) || paramName.startsWith(prefix)) {
                String unprefixed = paramName.substring(prefix.length());
                String[] values = request.getParameterValues(paramName);
                if (values == null || values.length == 0) {

                } else if (values.length > 1) {
                    params.put(unprefixed, values);
                } else {
                    params.put(unprefixed, values[0]);
                }
            }
        }
        return get(params, true);
    }

    /**
     * 通过 map 获取属性过滤器集合
     *
     * @param filters 过滤器信息
     *
     * @return 属性过滤器集合
     */
    public static List<PropertyFilter> get(Map<String, Object> filters) {

        return get(filters,false);
    }

    /**
     * 通过 map 获取属性过滤器集合
     *
     * @param filters 过滤器信息
     * @param ignoreEmptyValue true表示当存在 null或者""值时忽略该PropertyFilter
     *
     * @return 属性过滤器集合
     */
    public static List<PropertyFilter> get(Map<String,Object> filters, boolean ignoreEmptyValue) {
        List<PropertyFilter> filterList = new ArrayList<PropertyFilter>();

        for (Map.Entry<String, Object> entry : filters.entrySet()) {
            String expression = entry.getKey();
            Object value = entry.getValue();

            if (ignoreEmptyValue && (value == null || value.toString().trim().equals(""))) {
                continue;
            }

            PropertyFilter filter = get(expression, value.toString());
            filterList.add(filter);

        }
        return filterList;
    }
}
