/*
 * Decompiled with CFR 0.152.
 */
package net.codecrete.qrbill.generator;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import net.codecrete.qrbill.generator.Address;
import net.codecrete.qrbill.generator.AlternativeScheme;
import net.codecrete.qrbill.generator.Bill;
import net.codecrete.qrbill.generator.BillFormat;
import net.codecrete.qrbill.generator.Payments;
import net.codecrete.qrbill.generator.Strings;
import net.codecrete.qrbill.generator.ValidationMessage;
import net.codecrete.qrbill.generator.ValidationResult;

class Validator {
    private final Bill billIn;
    private final Bill billOut;
    private final ValidationResult validationResult;
    private static final BigDecimal AMOUNT_MIN = BigDecimal.valueOf(1L, 2);
    private static final BigDecimal AMOUNT_MAX = BigDecimal.valueOf(99999999999L, 2);

    static ValidationResult validate(Bill bill) {
        Validator validator = new Validator(bill);
        return validator.validateBill();
    }

    private Validator(Bill bill) {
        this.billIn = bill;
        this.billOut = new Bill();
        this.validationResult = new ValidationResult();
    }

    private ValidationResult validateBill() {
        this.billOut.setFormat(this.billIn.getFormat() != null ? new BillFormat(this.billIn.getFormat()) : null);
        this.billOut.setVersion(this.billIn.getVersion());
        this.validateAccountNumber();
        this.validateCreditor();
        this.validateCurrency();
        this.validateAmount();
        this.validateDebtor();
        this.validateReference();
        this.validateAdditionalInformation();
        this.validateAlternativeSchemes();
        this.validationResult.setCleanedBill(this.billOut);
        return this.validationResult;
    }

    private void validateCurrency() {
        String currency = Strings.trimmed(this.billIn.getCurrency());
        if (this.validateMandatory(currency, "currency")) {
            if (!"CHF".equals(currency = currency.toUpperCase(Locale.US)) && !"EUR".equals(currency)) {
                this.validationResult.addMessage(ValidationMessage.Type.ERROR, "currency", "currency_is_chf_or_eur");
            } else {
                this.billOut.setCurrency(currency);
            }
        }
    }

    private void validateAmount() {
        BigDecimal amount = this.billIn.getAmount();
        if (amount == null) {
            this.billOut.setAmount(null);
        } else if (AMOUNT_MIN.compareTo(amount = amount.setScale(2, RoundingMode.HALF_UP)) > 0 || AMOUNT_MAX.compareTo(amount) < 0) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, "amount", "amount_in_valid_range");
        } else {
            this.billOut.setAmount(amount);
        }
    }

    private void validateAccountNumber() {
        String account = Strings.trimmed(this.billIn.getAccount());
        if (this.validateMandatory(account, "account") && this.validateIBAN(account = Strings.whiteSpaceRemoved(account).toUpperCase(Locale.US))) {
            if (!account.startsWith("CH") && !account.startsWith("LI")) {
                this.validationResult.addMessage(ValidationMessage.Type.ERROR, "account", "account_is_ch_li_iban");
            } else if (account.length() != 21) {
                this.validationResult.addMessage(ValidationMessage.Type.ERROR, "account", "account_is_valid_iban");
            } else {
                this.billOut.setAccount(account);
            }
        }
    }

    private void validateCreditor() {
        Address creditor = this.validateAddress(this.billIn.getCreditor(), "creditor", true);
        this.billOut.setCreditor(creditor);
    }

    private void validateReference() {
        String account = this.billOut.getAccount();
        boolean isValidAccount = account != null;
        boolean isQRBillIBAN = account != null && account.charAt(4) == '3' && (account.charAt(5) == '0' || account.charAt(5) == '1');
        String reference = Strings.trimmed(this.billIn.getReference());
        if (reference != null) {
            reference = Strings.whiteSpaceRemoved(reference);
        }
        if (isQRBillIBAN) {
            this.validateQRReference(reference);
        } else if (isValidAccount && reference != null) {
            this.validateISOReference(reference);
        } else {
            this.billOut.setReference(null);
        }
    }

    private void validateQRReference(String cleanedReference) {
        if (cleanedReference == null) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, "reference", "mandatory_for_qr_iban");
            return;
        }
        if (cleanedReference.length() < 27) {
            cleanedReference = "00000000000000000000000000".substring(0, 27 - cleanedReference.length()) + cleanedReference;
        }
        if (!Payments.isValidQRReference(cleanedReference)) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, "reference", "valid_qr_ref_no");
        } else {
            this.billOut.setReference(cleanedReference);
        }
    }

    private void validateISOReference(String cleanedReference) {
        if (!Payments.isValidISO11649Reference(cleanedReference)) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, "reference", "valid_iso11649_creditor_ref");
        } else {
            this.billOut.setReference(cleanedReference);
        }
    }

    private void validateAdditionalInformation() {
        String billInformation = Strings.trimmed(this.billIn.getBillInformation());
        String unstructuredMessage = Strings.trimmed(this.billIn.getUnstructuredMessage());
        if (!(billInformation == null || billInformation.startsWith("//") && billInformation.length() >= 4)) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, "billInformation", "bill_info_invalid");
            billInformation = null;
        }
        if (billInformation == null && unstructuredMessage == null) {
            return;
        }
        if (billInformation == null) {
            unstructuredMessage = this.cleanedValue(unstructuredMessage, "unstructuredMessage");
            unstructuredMessage = this.clippedValue(unstructuredMessage, 140, "unstructuredMessage");
            this.billOut.setUnstructuredMessage(unstructuredMessage);
        } else if (unstructuredMessage == null) {
            if (this.validateLength(billInformation = this.cleanedValue(billInformation, "billInformation"), 140, "billInformation")) {
                this.billOut.setBillInformation(billInformation);
            }
        } else {
            billInformation = this.cleanedValue(billInformation, "billInformation");
            unstructuredMessage = this.cleanedValue(unstructuredMessage, "unstructuredMessage");
            int combinedLength = billInformation.length() + unstructuredMessage.length();
            if (combinedLength > 140) {
                this.validationResult.addMessage(ValidationMessage.Type.ERROR, "unstructuredMessage", "additional_info_too_long");
                this.validationResult.addMessage(ValidationMessage.Type.ERROR, "billInformation", "additional_info_too_long");
            } else {
                this.billOut.setUnstructuredMessage(unstructuredMessage);
                this.billOut.setBillInformation(billInformation);
            }
        }
    }

    private void validateAlternativeSchemes() {
        List<AlternativeScheme> schemeList;
        AlternativeScheme[] schemesOut = null;
        if (this.billIn.getAlternativeSchemes() != null && !(schemeList = this.createCleanSchemeList()).isEmpty() && (schemesOut = schemeList.toArray(new AlternativeScheme[0])).length > 2) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, "altSchemes", "alt_scheme_max_exceed");
            schemesOut = Arrays.copyOfRange(schemesOut, 0, 2);
        }
        this.billOut.setAlternativeSchemes(schemesOut);
    }

    private List<AlternativeScheme> createCleanSchemeList() {
        int len = this.billIn.getAlternativeSchemes().length;
        ArrayList<AlternativeScheme> schemeList = new ArrayList<AlternativeScheme>(len);
        for (AlternativeScheme schemeIn : this.billIn.getAlternativeSchemes()) {
            AlternativeScheme schemeOut = new AlternativeScheme();
            schemeOut.setName(Strings.trimmed(schemeIn.getName()));
            schemeOut.setInstruction(Strings.trimmed(schemeIn.getInstruction()));
            if (schemeOut.getName() == null && schemeOut.getInstruction() == null || !this.validateLength(schemeOut.getInstruction(), 100, "altSchemes")) continue;
            schemeList.add(schemeOut);
        }
        return schemeList;
    }

    private void validateDebtor() {
        Address debtor = this.validateAddress(this.billIn.getDebtor(), "debtor", false);
        this.billOut.setDebtor(debtor);
    }

    private Address validateAddress(Address addressIn, String fieldRoot, boolean mandatory) {
        Address addressOut = this.cleanedPerson(addressIn, fieldRoot);
        if (addressOut == null) {
            this.validateEmptyAddress(fieldRoot, mandatory);
            return null;
        }
        if (addressOut.getType() == Address.Type.CONFLICTING) {
            this.emitErrorsForConflictingType(addressOut, fieldRoot);
        }
        this.checkMandatoryAddressFields(addressOut, fieldRoot);
        if (!(addressOut.getCountryCode() == null || addressOut.getCountryCode().length() == 2 && Payments.isAlphaNumeric(addressOut.getCountryCode()))) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, fieldRoot + ".countryCode", "valid_country_code");
        }
        this.cleanAddressFields(addressOut, fieldRoot);
        return addressOut;
    }

    private void validateEmptyAddress(String fieldRoot, boolean mandatory) {
        if (mandatory) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, fieldRoot + ".name", "field_is_mandatory");
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, fieldRoot + ".postalCode", "field_is_mandatory");
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, fieldRoot + ".addressLine2", "field_is_mandatory");
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, fieldRoot + ".town", "field_is_mandatory");
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, fieldRoot + ".countryCode", "field_is_mandatory");
        }
    }

    private void emitErrorsForConflictingType(Address addressOut, String fieldRoot) {
        if (addressOut.getAddressLine1() != null) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, fieldRoot + ".addressLine1", "address_type_conflict");
        }
        if (addressOut.getAddressLine2() != null) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, fieldRoot + ".addressLine2", "address_type_conflict");
        }
        if (addressOut.getStreet() != null) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, fieldRoot + ".street", "address_type_conflict");
        }
        if (addressOut.getHouseNo() != null) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, fieldRoot + ".houseNo", "address_type_conflict");
        }
        if (addressOut.getPostalCode() != null) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, fieldRoot + ".postalCode", "address_type_conflict");
        }
        if (addressOut.getTown() != null) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, fieldRoot + ".town", "address_type_conflict");
        }
    }

    private void checkMandatoryAddressFields(Address addressOut, String fieldRoot) {
        this.validateMandatory(addressOut.getName(), fieldRoot, ".name");
        if (addressOut.getType() == Address.Type.STRUCTURED || addressOut.getType() == Address.Type.UNDETERMINED) {
            this.validateMandatory(addressOut.getPostalCode(), fieldRoot, ".postalCode");
            this.validateMandatory(addressOut.getTown(), fieldRoot, ".town");
        }
        if (addressOut.getType() == Address.Type.COMBINED_ELEMENTS || addressOut.getType() == Address.Type.UNDETERMINED) {
            this.validateMandatory(addressOut.getAddressLine2(), fieldRoot, ".addressLine2");
        }
        this.validateMandatory(addressOut.getCountryCode(), fieldRoot, ".countryCode");
    }

    private void cleanAddressFields(Address addressOut, String fieldRoot) {
        addressOut.setName(this.clippedValue(addressOut.getName(), 70, fieldRoot, ".name"));
        if (addressOut.getType() == Address.Type.STRUCTURED) {
            addressOut.setStreet(this.clippedValue(addressOut.getStreet(), 70, fieldRoot, ".street"));
            addressOut.setHouseNo(this.clippedValue(addressOut.getHouseNo(), 16, fieldRoot, ".houseNo"));
            addressOut.setPostalCode(this.clippedValue(addressOut.getPostalCode(), 16, fieldRoot, ".postalCode"));
            addressOut.setTown(this.clippedValue(addressOut.getTown(), 35, fieldRoot, ".town"));
        }
        if (addressOut.getType() == Address.Type.COMBINED_ELEMENTS) {
            addressOut.setAddressLine1(this.clippedValue(addressOut.getAddressLine1(), 70, fieldRoot, ".addressLine1"));
            addressOut.setAddressLine2(this.clippedValue(addressOut.getAddressLine2(), 70, fieldRoot, ".addressLine2"));
        }
        if (addressOut.getCountryCode() != null) {
            addressOut.setCountryCode(addressOut.getCountryCode().toUpperCase(Locale.US));
        }
    }

    private boolean validateIBAN(String iban) {
        if (!Payments.isValidIBAN(iban)) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, "account", "account_is_valid_iban");
            return false;
        }
        return true;
    }

    private Address cleanedPerson(Address addressIn, String fieldRoot) {
        if (addressIn == null) {
            return null;
        }
        Address addressOut = new Address();
        addressOut.setName(this.cleanedValue(addressIn.getName(), fieldRoot, ".name"));
        String value = this.cleanedValue(addressIn.getAddressLine1(), fieldRoot, ".addressLine1");
        if (value != null) {
            addressOut.setAddressLine1(value);
        }
        if ((value = this.cleanedValue(addressIn.getAddressLine2(), fieldRoot, ".addressLine2")) != null) {
            addressOut.setAddressLine2(value);
        }
        if ((value = this.cleanedValue(addressIn.getStreet(), fieldRoot, ".street")) != null) {
            addressOut.setStreet(value);
        }
        if ((value = this.cleanedValue(addressIn.getHouseNo(), fieldRoot, ".houseNo")) != null) {
            addressOut.setHouseNo(value);
        }
        if ((value = this.cleanedValue(addressIn.getPostalCode(), fieldRoot, ".postalCode")) != null) {
            addressOut.setPostalCode(value);
        }
        if ((value = this.cleanedValue(addressIn.getTown(), fieldRoot, ".town")) != null) {
            addressOut.setTown(value);
        }
        addressOut.setCountryCode(Strings.trimmed(addressIn.getCountryCode()));
        if (addressOut.getName() == null && addressOut.getCountryCode() == null && addressOut.getType() == Address.Type.UNDETERMINED) {
            return null;
        }
        return addressOut;
    }

    private boolean validateMandatory(String value, String field) {
        if (Strings.isNullOrEmpty(value)) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, field, "field_is_mandatory");
            return false;
        }
        return true;
    }

    private void validateMandatory(String value, String fieldRoot, String subfield) {
        if (Strings.isNullOrEmpty(value)) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, fieldRoot + subfield, "field_is_mandatory");
        }
    }

    private boolean validateLength(String value, int maxLength, String field) {
        if (value != null && value.length() > maxLength) {
            this.validationResult.addMessage(ValidationMessage.Type.ERROR, field, "field_value_too_long", new String[]{Integer.toString(maxLength)});
            return false;
        }
        return true;
    }

    private String clippedValue(String value, int maxLength, String field) {
        if (value != null && value.length() > maxLength) {
            this.validationResult.addMessage(ValidationMessage.Type.WARNING, field, "field_clipped", new String[]{Integer.toString(maxLength)});
            return value.substring(0, maxLength);
        }
        return value;
    }

    private String clippedValue(String value, int maxLength, String fieldRoot, String subfield) {
        if (value != null && value.length() > maxLength) {
            this.validationResult.addMessage(ValidationMessage.Type.WARNING, fieldRoot + subfield, "field_clipped", new String[]{Integer.toString(maxLength)});
            return value.substring(0, maxLength);
        }
        return value;
    }

    private String cleanedValue(String value, String fieldRoot, String subfield) {
        Payments.CleaningResult result = new Payments.CleaningResult();
        Payments.cleanValue(value, result);
        if (result.replacedUnsupportedChars) {
            this.validationResult.addMessage(ValidationMessage.Type.WARNING, fieldRoot + subfield, "replaced_unsupported_characters");
        }
        return result.cleanedString;
    }

    private String cleanedValue(String value, String fieldName) {
        Payments.CleaningResult result = new Payments.CleaningResult();
        Payments.cleanValue(value, result);
        if (result.replacedUnsupportedChars) {
            this.validationResult.addMessage(ValidationMessage.Type.WARNING, fieldName, "replaced_unsupported_characters");
        }
        return result.cleanedString;
    }
}

