package com.cloudburo.evernote;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import com.evernote.auth.EvernoteAuth;
import com.evernote.auth.EvernoteService;
import com.evernote.clients.ClientFactory;
import com.evernote.clients.NoteStoreClient;
import com.evernote.clients.UserStoreClient;
import com.evernote.edam.error.EDAMNotFoundException;
import com.evernote.edam.error.EDAMSystemException;
import com.evernote.edam.error.EDAMUserException;
import com.evernote.edam.notestore.NoteFilter;
import com.evernote.edam.notestore.NoteList;
import com.evernote.edam.type.Note;
import com.evernote.edam.type.NoteAttributes;
import com.evernote.edam.type.NoteSortOrder;
import com.evernote.edam.type.Notebook;
import com.evernote.edam.type.Tag;
import com.evernote.thrift.TException;
import com.evernote.thrift.transport.TTransportException;
import com.github.slugify.Slugify;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.apache.commons.lang.StringEscapeUtils;

import com.cloudburo.grab.webcontent.Grabber;
import com.cloudburo.grab.webcontent.GrabberRecord;

/**
 * To execute the program
 * mvn exec:java -Dexec.args="cfg/config.properties <SANDBOX>" 
 *
 */
public class EvernoteBlogPostGenerator  {

  // Real applications authenticate with Evernote using OAuth, but for the
  // purpose of exploring the API, you can get a developer token that allows
  // you to access your own Evernote account. To get a developer token, visit
  // https://sandbox.evernote.com/api/DeveloperToken.action
  // Default value
  static final Logger logger = LoggerFactory.getLogger(EvernoteBlogPostGenerator.class);	
  
  private UserStoreClient userStore;
  private NoteStoreClient noteStore;
  private String blogName;
  private String targetDir;
  private Utility utility;
  private boolean serviceNews;
  private boolean serviceCuration;
  private static boolean moveNote = true;
  private boolean followLink;
  
  private static final String N_AUTOMATIC_GRAB = "-A-Fetch";
  private static final String N_PREPARE_BLOG = "-B-Prepare";
  private static final String N_NEWS_GEN = "-D-NewsGen";
  private static final String N_CURATION_GEN = "-E-CurateGen";
  private static final String N_ARCHIVE = "-Z-Archive";
  // FIXME A maximum of 20 notes
  private static final int MAX_SEARCH_RESULTS = 3;

  /**
   * Initialize UserStore and NoteStore clients. During this step, we
   * authenticate with the Evernote web service. 
 * @throws EDAMNotFoundException 
   */
  public EvernoteBlogPostGenerator(String token, Boolean productionEnv, String blogName, String targetDir,
		  Boolean moveN, boolean serviceCuration, boolean serviceNews, boolean followLink) throws 
  	EDAMSystemException,EDAMUserException,TTransportException,TException, EDAMNotFoundException, Exception  {
    // Set up the UserStore client and check that we can speak to the server
	this.blogName = blogName;
	this.targetDir = targetDir;
	this.serviceNews = serviceNews;
	this.serviceCuration = serviceCuration;
	this.followLink = followLink;
	moveNote = moveN;
	EvernoteAuth evernoteAuth;
	if (productionEnv)
		evernoteAuth = new EvernoteAuth(EvernoteService.PRODUCTION, token);
	else  
		evernoteAuth = new EvernoteAuth(EvernoteService.SANDBOX, token);
    ClientFactory factory = new ClientFactory(evernoteAuth);
    userStore = factory.createUserStoreClient();

    boolean versionOk = userStore.checkVersion("Evernote EDAMDemo (Java)",
        com.evernote.edam.userstore.Constants.EDAM_VERSION_MAJOR,
        com.evernote.edam.userstore.Constants.EDAM_VERSION_MINOR);
    if (!versionOk) {
      logger.error("Incompatible Evernote client protocol version");
      System.exit(1);
    }
    // Set up the NoteStore client
    noteStore = factory.createNoteStoreClient();
    utility = new Utility(noteStore);
    utility.createTags();
    if (serviceCuration)
    	checkAndFixNotebookStructure(true);
    if (serviceNews)
    	checkAndFixNotebookStructure(false);
    
  }
  
  /*
  public int generate(int maxArticles, boolean curation) throws Exception {
	 if (curation) {
		 logger.debug("Curation Blog Entry Generation");
		 return generateBlogEntry(curation,maxArticles);
	 }
	 else {
		 logger.debug("News Blog Entry Generation");
		 return generateBlogEntry(curation,maxArticles);		 
	 }
  }*/
 
  public void checkAndCreateTaggedNotes(String[] tags,String[] title) throws Exception {
	  // We expect one note with the tag 
	  List<Tag> tagsl = noteStore.listTags();
	  
      for (int i = 0; i<tags.length;i++) { 
    	  String taguid = null;
		  Iterator<Tag> it = tagsl.iterator();
		  while (it.hasNext()) {
			  Tag tag = it.next();
			  if (tag.getName().equals(tags[i])) {
				  taguid = tag.getGuid();
				  break;
			  }
		  }
		  boolean createNote = false;
		  List<String> tidList = new ArrayList<String>();
		  if (taguid == null) {
			  // Create a new tag it's missing so there is no note then
			  Tag newTag = new Tag();
			  newTag.setName(tags[i]);
			  taguid = noteStore.createTag(newTag).getGuid();
			  tidList.add(taguid);
			  createNote = true;
		  } else {
			  NoteFilter filter = new NoteFilter();
			  tidList.add(taguid);
			  filter.setTagGuids(tidList);
			  NoteList noteList = noteStore.findNotes(filter, 0, 1);
			  if (noteList.getNotes().size() == 0) createNote = true;
		  }
		  if (createNote) {
			  logger.info("Creating new note for tag '"+tags[i]+"' and title '"+title[i]+"'");
			  checkAndFixNotebookStructure(true);
			  Note note = new Note();
			  note.setTitle(title[i]);
			  note.setTagGuids(tidList);
			  note.setNotebookGuid(getNoteBook(N_PREPARE_BLOG).getGuid());
			  String content = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>"
				        + "<!DOCTYPE en-note SYSTEM \"http://xml.evernote.com/pub/enml2.dtd\">"
				        + "<en-note>"
				        + "<span style=\"color:green;\">Add your content</span><br/>"
				        + "<span style=\"color:red;\">Fügen sie ihren Text ein</span><br/>"
				        + "</en-note>";
			  note.setContent(content);
			  noteStore.createNote(note);
		  }
	  } 
  }
  
  private Notebook getNoteBook(String postFix) throws Exception {
	  String blogName = "blog-"+this.blogName+postFix;
	  logger.debug("Get Notebook "+blogName);
	  List<Notebook> notebooks = noteStore.listNotebooks();
	  for (Notebook notebook : notebooks) {
	      if (notebook.getName().equals(blogName)) return notebook;
	  }
	  throw new Exception("Couldn't find Notebook "+blogName);
  }
  
  protected Hashtable<String,String> checkAndFixNotebookStructure(boolean curationBlogType) throws Exception {
	  List<Notebook> notebooks = noteStore.listNotebooks();
	  String doneGID=null, inGID=null, prepGID=null, autoGID=null, queueGID = null;
	  Hashtable<String,String> table = new Hashtable<String,String>();
	  for (Notebook notebook : notebooks) {
		  // General ones
	      if (notebook.getName().equals("blog-"+blogName+N_ARCHIVE)) {
	    	  doneGID = notebook.getGuid();
	    	  table.put(N_ARCHIVE, doneGID);
	      } else if (notebook.getName().equals("blog-"+blogName+N_PREPARE_BLOG)) {
	    	  prepGID = notebook.getGuid();
	    	  table.put(N_PREPARE_BLOG, prepGID);
	      } // Curation Type
	      else if (notebook.getName().equals("blog-"+blogName+N_CURATION_GEN) && curationBlogType) {
	    	  inGID = notebook.getGuid();
	    	  table.put(N_CURATION_GEN,inGID);
	      }	// News Type
	      else if (notebook.getName().equals("blog-"+blogName+N_NEWS_GEN) && !curationBlogType) {
	    	  inGID = notebook.getGuid();
	    	  table.put(N_NEWS_GEN, inGID);
	      }
	      else if (notebook.getName().equals("blog-"+blogName+N_AUTOMATIC_GRAB) && !curationBlogType) {
	    	  autoGID = notebook.getGuid();
	    	  table.put(N_AUTOMATIC_GRAB,autoGID);
	      }
	  }
	  if (doneGID == null) {
    	  Notebook nb = new Notebook();
    	  nb.setName("blog-"+blogName+N_ARCHIVE);
    	  table.put(N_ARCHIVE,noteStore.createNotebook(nb).getGuid());
      }
	  if (prepGID == null) {
    	  Notebook nb = new Notebook();
    	  nb.setName("blog-"+blogName+N_PREPARE_BLOG);
    	  table.put(N_PREPARE_BLOG,noteStore.createNotebook(nb).getGuid());
      }
	  if (curationBlogType) {
		  if (inGID == null) {
	    	  Notebook nb = new Notebook();
	    	  nb.setName("blog-"+blogName+N_CURATION_GEN);
	    	  table.put(N_CURATION_GEN,noteStore.createNotebook(nb).getGuid());
	      }  
	  } else {
		  if (inGID == null) {
	    	  Notebook nb = new Notebook();
	    	  nb.setName("blog-"+blogName+N_NEWS_GEN);
	    	  table.put(N_NEWS_GEN, noteStore.createNotebook(nb).getGuid());
	      } 		  
		  if (autoGID == null) {
	    	  Notebook nb = new Notebook();
	    	  nb.setName("blog-"+blogName+N_AUTOMATIC_GRAB);
	    	  table.put(N_AUTOMATIC_GRAB, noteStore.createNotebook(nb).getGuid());
	      } 
	  }
	  return table;
  }
  
  public  void fetchHTMLContent() throws Exception {
	  // List the notes in the user's account 
	  // Every Notebook is prefixed with 'blog-'
	  String blogName = "blog-"+this.blogName;
	  logger.debug("Fetch News HTML pages for "+blogName);
	  Hashtable<String,String> table = checkAndFixNotebookStructure(false);
	   
	  Notebook notebook = noteStore.getNotebook(table.get(N_AUTOMATIC_GRAB));
	  NoteFilter filter = new NoteFilter();
	  filter.setNotebookGuid(notebook.getGuid());
	  filter.setOrder(NoteSortOrder.CREATED.getValue());
	  filter.setAscending(true);	  
	  NoteList noteList = noteStore.findNotes(filter, 0, MAX_SEARCH_RESULTS);
	  List<Note> notes = noteList.getNotes();
	  long startTime = System.currentTimeMillis();
	  Grabber grabber = new Grabber();
	  for (Note note : notes) {
		  String sourceURL = note.getAttributes().getSourceURL();
		  Note fullNote = noteStore.getNote(note.getGuid(), true, false,false,false);
		  try {
			  GrabberRecord rec = grabber.extractArticle(sourceURL,true);
			  rec.content = StringEscapeUtils.escapeHtml(rec.content);
			  String article = "<!DOCTYPE en-note SYSTEM \"http://xml.evernote.com/pub/enml2.dtd\"><en-note>"+rec.content+"</en-note>";
			  logger.debug("Fetched article: "+article);			  
			  fullNote.setContent(article);
			  NoteAttributes attr = fullNote.getAttributes();
			  logger.debug("url fetched: "+rec.url.toExternalForm());
			  attr.setSourceURL(rec.url.toExternalForm());
			  fullNote.setAttributes(attr);
			  fullNote.setNotebookGuid(table.get(N_PREPARE_BLOG));
			  noteStore.updateNote(fullNote);
			  logger.debug("Updated note; "+note.getTitle());
		  } catch (Exception ex1 ) {
			  logger.warn("Grabbing of htlm record failed, going to move it: "+ex1.getMessage(),ex1);
			  try {
			    note.setNotebookGuid(table.get(N_PREPARE_BLOG));
			    noteStore.updateNote(note);
			  } catch (Exception ex2) { logger.error("Move of record failed- go on: "+ex2.getMessage(),ex2);}
		  }
	  } // For Notes
	  long execTime = System.currentTimeMillis()-startTime;
	  logger.info("@"+blogName+" fetched Articles completed in '"+execTime+"' ms");
	  Thread.sleep(1000); 
  }
  
  public int generateBlogEntry(boolean curationBlogType, int maxNews, boolean cleanupStep) throws Exception {  
	  // Every Notebook is prefixed with 'blog-'
	  String blogName = "blog-"+this.blogName;
	  
	  BlogGenerator generator = new MiddleManBlogGenerator(this.blogName, noteStore, targetDir, moveNote, followLink);
	  
	  Hashtable<String,String> table = checkAndFixNotebookStructure(curationBlogType);
	  Notebook notebook = null;
	  if (curationBlogType)
	       notebook = noteStore.getNotebook(table.get(N_CURATION_GEN));
	  else 
		   notebook = noteStore.getNotebook(table.get(N_NEWS_GEN));

	  // Next, search for the first MAX_SEARCH_RESULTS notes in this notebook, ordering by creation date
	  NoteFilter filter = new NoteFilter();
	  filter.setNotebookGuid(notebook.getGuid());
	  filter.setOrder(NoteSortOrder.CREATED.getValue());
	  filter.setAscending(true);
	  
	  NoteList noteList = null;
	  
	  if (cleanupStep) {
	    List<String> tagsGuids = new ArrayList<String>();
	    tagsGuids.add(utility.getTagGuidForName("clb-processed"));
	    filter.setTagGuids(tagsGuids);
	    filter.setTagGuidsIsSet(true);
	    noteList = noteStore.findNotes(filter, 0, maxNews);
	    tagsGuids = new ArrayList<String>();
	    tagsGuids.add(utility.getTagGuidForName("clb-delete"));
	    filter.setTagGuids(tagsGuids);
	    NoteList noteList1 = noteStore.findNotes(filter, 0, maxNews);
	    Iterator<Note> it = noteList1.getNotesIterator();
	    while (it.hasNext()) {
	    	Note nt = it.next();
	    	if (!noteList.getNotes().contains(nt)) noteList.addToNotes(it.next());
	    }
	  } else {
	    noteList = noteStore.findNotes(filter, 0, maxNews);
	  }
	  List<Note> notes = noteList.getNotes();
	  long startTime = System.currentTimeMillis();
	  int skipped = 0, generate = 0, futurePost = 0,deletePost = 0;
	  StringBuffer logDescr = new StringBuffer();
	  // Here we iterate over the notes
	  for (Note note : notes) {
		  // Check if it i
		  logger.info("Processing html.slim for '" + note.getTitle()+"'");
		  String fileName = null;
		  String blogDateFull = null;
		  // The creation date of the blog entry will be used
		  Date bloggingDate = new Date();
		  boolean processFlag = utility.hasProcessedFlag(note);
		  // With the Update Date it's possible to schedule for the future

		  if (!processFlag && bloggingDate.before(new Date(note.getUpdated()))) {
			  futurePost++;
		  } else {
			  if (!processFlag) 
				  note.setCreated(bloggingDate.getTime());
			  else
				  bloggingDate = new Date(note.getCreated());
			  String title = (new Slugify()).slugify(note.getTitle());
			  if (title.length() > 80) title = title.substring(0, 80);
			  fileName = (new SimpleDateFormat("yyyy-MM-dd")).format(bloggingDate) + "-"+ title;
			  blogDateFull = (new SimpleDateFormat("yyyy-MM-dd HH:mm:ss")).format(bloggingDate);
			  if (utility.hasDeleteFlag(note)) {
				  if (deleteEntry(fileName,generator)) {
					 deletePost++; 
					 generate++;
					 utility.unsetDeleteTag(note);
				  }
				  if (moveNote) {
					  note.setNotebookGuid(table.get(N_ARCHIVE));
					  noteStore.updateNote(note);
					  logger.debug("Moved deleted processed Note");
				  }
			  } else {
				  generator.generateCurationFile(note, fileName, blogDateFull, table.get(N_ARCHIVE), logDescr, skipped, curationBlogType);
				  generate++;
				  utility.applyTagToNote(note,"clb-processed"); 
			  }
		  }	
	  } // For Notes
	  long execTime = System.currentTimeMillis()-startTime;
	  String msg = "";
	  if (curationBlogType) msg = "Curation";
	  else  msg = "News";
	  
	  int gen = generate-deletePost;
	  
	  logger.info("@"+blogName+ " - "+msg+" Entries Generation in '"+execTime+"' ms:'"+
		  " gen("+ gen +"), "+ "fut("+futurePost+"), del("+deletePost+ ") nok("+skipped+")");
	  Thread.sleep(1000); 
	  return generate;
  }
  
  private boolean deleteEntry(String fileName, BlogGenerator generator) {
	  String target = targetDir + "/" + fileName+generator.getPostFilePostfix();
	  File fi = new File(target);
	  if (fi.delete()) {
		  logger.info("Deleted File "+target);
		  return true;
		 
	  }
	  else
		  logger.warn("Deleting of File failed "+target);
	  return false;
	  
  }
  
}