package com.cloudburo.evernote;

import java.awt.image.BufferedImage;
import java.io.BufferedReader;
import java.io.Closeable;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.imageio.ImageIO;

//import org.apache.log4j.Logger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.imgscalr.Scalr;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Attribute;
import org.jsoup.nodes.Attributes;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Document.OutputSettings;
import org.jsoup.nodes.Element;
import org.jsoup.nodes.Node;
import org.jsoup.nodes.TextNode;
import org.jsoup.safety.Whitelist;
import org.jsoup.select.Elements;

import com.evernote.clients.NoteStoreClient;
import com.evernote.edam.error.EDAMNotFoundException;
import com.evernote.edam.error.EDAMSystemException;
import com.evernote.edam.error.EDAMUserException;
import com.evernote.edam.type.Note;
import com.evernote.edam.type.Resource;
import com.evernote.thrift.TException;


public abstract class BlogGenerator {
	
    static final Logger logger = LoggerFactory.getLogger(BlogGenerator.class);	
    
    private static final String T_MENU = "clb-menu";
    private static final String T_FOOTER = "clb-footer";    
    private static final String T_INDEX = "clb-index";
    private static final String T_MENU_PREFIX = "clb-";
     
    protected String imageDirectory;
	
	protected String blogName;
	protected String targetDir;
	protected NoteStoreClient noteStore;
	protected Utility utility;
	protected boolean moveNote;
	protected boolean followLink;
	
	public BlogGenerator(String blogName, NoteStoreClient store, String targetDir,  boolean moveNote, boolean followLink) {
			this.blogName = blogName;
			this.noteStore = store;
			this.targetDir = targetDir;
			this.moveNote = moveNote;
			this.followLink = followLink;
			utility = new Utility(store);
	}

	public abstract void generateCurationFile( Note note,String fileName, String blogDateFull, 
			String doneGID, StringBuffer logDescr, int skipped, boolean curationType) throws Exception;

	
	public abstract String getPostFilePostfix();
	
	public boolean isMenuPage(String tagName) {
		return tagName.startsWith(T_MENU) || tagName.startsWith(T_INDEX) || tagName.startsWith(T_FOOTER);
	}
	
	public String getMenuFileName(String tagName) {
		return tagName.substring(T_MENU_PREFIX.length());
	}
	
	
	
	@SuppressWarnings("resource")
	protected static void writeToFile(File file, Charset charset, String data) throws IOException {
		OutputStream out = new FileOutputStream(file);
		Closeable stream = out;
		try {
			Writer writer = new OutputStreamWriter(out, charset);
			stream = writer;
			writer.write(data);
		} finally {
			stream.close();
		}
	}

	protected int countWords(String input) {
		String[] words = input.split(" ");  // split input string on space
		return words.length;

	}

	protected String cleanupStyleAttribute(Element doc){
		Iterator<Element> it = doc.select("[style]").iterator();
		while (it.hasNext()) {
			Element e  = it.next();
			Attributes  styleAtt= e.attributes();
			Attribute a = styleAtt.asList().get(0);           
			if(a.getKey().equals("style")){
				String[] items = a.getValue().trim().split(";");
				String newValue = "";
				for(String item: items){
					if(item.contains("font-size:")|| item.contains("text-decoration:") ||
							item.contains("color:")||item.contains("font-weight:")||item.contains("font-style:")
							|| item.contains("text-align")|| item.contains("border")|| item.contains("padding")
							|| item.contains("margin")|| item.contains("table-layout") || item.contains("font-family")
							|| item.contains("border-collapse")|| item.contains("min-width")){
						logger.debug("Style Item takeover: "+item);
						newValue = newValue.concat(item).concat(";");
					} 
				}
				a.setValue(newValue);                 
			}

		}
		return doc.html();
	}

	protected String cleanupHTML(String elem) {
		Whitelist list = Whitelist.simpleText();
		list.addTags(
                "a", "b", "blockquote", "br", "cite", "code", "dd", "dl", "dt", "em",
                "i", "li", "ol", "p", "pre", "q", "small", "span", "strike", "strong", "sub",
                "sup", "u", "ul","table","td","tr","iframe","font");
        list.addAttributes("a", "href");
        list.addAttributes("blockquote", "cite");
        list.addAttributes("q", "cite");
        list.addAttributes("div","class");
        list.addAttributes("iframe","class","frameborder","allowfullscreen","src");
        list.addAttributes("font","face");
        

        list.addProtocols("a", "href", "ftp", "http", "https", "mailto");
        list.addProtocols("blockquote", "cite", "http", "https");
        list.addProtocols("cite", "cite", "http", "https");
		list.addTags("div");
        list.addTags("img");
        list.addAttributes("img", "align", "alt", "height", "src", "title", "width","text-align");
        list.addProtocols("img", "src", "http", "https");
		
		if (!followLink)
			list.addEnforcedAttribute("a", "rel", "nofollow");
		list.addAttributes(":all","style");
		//elem = elem.replaceAll("&nbsp;"," ");
		elem = VideoHelper.processVideoTags(elem);
		logger.debug("Jsoup.before: "+elem);
		String doc = Jsoup.clean(elem, list);
		//doc = doc.replaceAll("\\n", "");
		//doc = doc.replaceAll("\\r", "");
		logger.debug("Jsoup.clean 2: "+doc);
		Document xmlDoc = Jsoup.parse(doc);
		doc = cleanupStyleAttribute(xmlDoc);
		// Nasty fix
		doc =  doc.replaceAll(" \n", "\n");
		logger.debug("Jsoup.clean 1: "+doc);
		return doc;
	}
	
	protected String processNote(org.jsoup.nodes.Document doc, String prependSpaces, 
	  String noteGuid, Note note) throws Exception {
		  StringBuffer fullNote = new StringBuffer("");
		  logger.debug("Process and extract en-media elements");
		  processExtractMedia(doc,prependSpaces,"jpeg",noteGuid,note);
		  processExtractMedia(doc,prependSpaces,"jpg",noteGuid,note);
		  processExtractMedia(doc,prependSpaces,"png",noteGuid,note);
		  processExtractMedia(doc,prependSpaces,"gif",noteGuid,note);
		  // processExtractImgTag(doc,prependSpaces,noteGuid,note);
		  Iterator<Element> nodesIt = doc.select("en-note").iterator(); 
		  if (nodesIt.hasNext()) {
			String html = cleanupHTML(nodesIt.next().html());
			doc = Jsoup.parse(html);
			OutputSettings settings = new OutputSettings();
		    settings.prettyPrint(false);
		    doc.outputSettings(settings);
			Elements body = doc.select("body");
			String fullDoc = body.toString();
			fullDoc = fullDoc.replaceAll("<body>", " ");
			fullDoc = fullDoc.replaceAll("</body>", " ");			
			
			BufferedReader bufReader = new BufferedReader(new StringReader(fullDoc));
			String line=null;
			try {
				while( (line=bufReader.readLine()) != null )
				{
					fullNote.append(prependSpaces);
					fullNote.append(line);
					fullNote.append("\n");
				}
			} catch (IOException ex) {
				logger.error("Reading en-note resulted in an IO Exception, skip...");
				fullNote = new StringBuffer("");
			}
			String output = youTubeFetcher(codeFetcher(fullNote));	
			output = htmlEmbedFetcher(output,prependSpaces);
			return output.replaceAll("&nbsp;", " ");
		  } else {
			  logger.error("Document: "+doc.toString()+" has no 'en-note' element");
		  }
		  return fullNote.toString();
	}
	
	// Code Fragments Processor
	// =============================
	// <div>
    // &lt;code java&gt; 
    // </div>
	// http://www.regexplanet.com/advanced/java/index.html
	private String codeFetcher(StringBuffer fullNote) {
		Pattern codeStartPattern = Pattern.compile("(?s)<div>(.*?)&lt;code&gt;(.*?)</div>(.*?)<div>(.*?)&lt;/code&gt;");
		Matcher matcher;
		matcher = codeStartPattern.matcher(fullNote);				
		// TODO: This is SLIM Dependent refactor it
		String prepend = "      ";
		StringBuffer output = new StringBuffer("");
		int lastEndPos = 0;
		while (matcher.find()) {
			lastEndPos = matcher.end();
			String lang = matcher.group(2).trim();
			String codePart = matcher.group(4);
			logger.debug("Code Language '"+lang+"'");
			logger.debug("Code Part Uncleaned: "+codePart);		
			codePart = codePart.replaceAll("</div>","</div>LINESEP");
			Document codeDoc = Jsoup.parse(codePart);
			Whitelist wl = new Whitelist();
			wl.addTags("br","p");
			codePart = codeDoc.html();
			String cleanDoc = Jsoup.clean(codePart, wl);
			logger.debug(" Jsoup Clean; \n"+cleanDoc);
			codeDoc = Jsoup.parse(cleanDoc);
			codeDoc.outputSettings().prettyPrint(false);
			logger.debug("CodeDoc Clean Parsed: \n"+codeDoc.html());
			//String cleanedCode = "";
			// Get every div element
			String codeFull = analyzeElement(codeDoc,prepend);
			codeFull = codeFull.replaceAll("LINESEP","\n");
			String[] lineSplit = codeFull.split("\n");
			codeFull = "";
			for (int i= 0; i<lineSplit.length; i++) {
				codeFull+=prepend+lineSplit[i]+"\n";
			}
			codeFull = codeFull.replace("\u00a0"," ");
			codeFull = codeFull.replaceAll("&nbsp;", " ");
			logger.debug("Code Part Ready: \n"+codeFull);		
			output.append(prepend+"<div>"+matcher.group(1)+"\n"+prepend+"``` "+matcher.group(2)+"\n  "+codeFull+"\n"+prepend+"```\n"); 
		}
		output.append("\n"+prepend+fullNote.substring(lastEndPos));
		return output.toString();
	}
	
	
	private String youTubeFetcher(String fullNote) {	
		String regPattern = "(?s)(.*?)&lt;video-youtube&gt;(.*?)&lt;/video-youtube&gt;(.*?)";
		String replacementStr = "$1<div class=\"responsive-video\"><iframe src=\"https://www.youtube.com/embed/$2?rel=0\" frameborder=\"0\" allowfullscreen></iframe></div>$3";
		
		//Pattern codeStartPattern = Pattern.compile(regPattern);
		//Matcher matcher;
		//matcher = codeStartPattern.matcher(fullNote);		
		// <iframe width="560" height="315" src="https://www.youtube.com/embed/kQ-BnKcLKpM?rel=0" frameborder="0" allowfullscreen></iframe>
		//String replacementStr = "$1<iframe  src=\"https://www.youtube.com/embed/\"$2?rel=0 frameborder=\"0\" allowfullscreen></iframe>$3";
		//matcher.replaceAll(replacementStr);
		return fullNote.replaceAll(regPattern, replacementStr);
	}
	
	private String htmlEmbedFetcher(String fullNote,String prependSpaces) {
		// String regPattern = "(?s)(.*?)&lt;htmlembed&gt;(.*?)&lt;/htmlembed&gt;(.*?)";
		String startTag = "&lt;htmlembed&gt;";
		String stopTag = "&lt;/htmlembed&gt;";
		String patterTxt = "(?s)(.*?)"+startTag+"(.*?)"+stopTag+"(.*?)";
		if (fullNote.contains("&lt;htmlembed&gt;")) {
			Pattern embedPattern = Pattern.compile(patterTxt);
			Matcher matcher;
			matcher = embedPattern.matcher(fullNote);
	        while(matcher.find()) {
	            String txt2 = matcher.group(2);
	            txt2 = txt2.replace("<br>", "");
	            txt2 = txt2.replace("&lt;", "<");
		        txt2 = txt2.replace("&gt;", ">");
	            fullNote = fullNote.substring(0,matcher.start(2)-startTag.length())+txt2+fullNote.substring(matcher.end(2)+stopTag.length());
	            matcher = embedPattern.matcher(fullNote);
	        }
		}
		return fullNote;
	}
	
	private String analyzeElement(Element cell, String prepend) {
		String text = "";
	    List<Node> childNodes = cell.childNodes();
	    for (int i=0; i< childNodes.size();i++) {
	    	Node childNode = childNodes.get(i);
	        if (childNode instanceof TextNode) {
	        	//String foundTxt = ((TextNode)childNode).getWholeText();
	            text += ((TextNode)childNode).getWholeText();
	        } else if (childNode instanceof Element) {
	          text += analyzeElement((Element)childNode, prepend);  
	        }
	    }
	    return text;
	}
	
	protected void processExtractMedia(org.jsoup.nodes.Document doc, String prependSpaces, String mime,  
			String noteGuid, Note note) throws Exception {
	  Elements elems = doc.select("en-media[type$="+mime+"]");
	  Iterator<Element> it = elems.iterator();
	  while (it.hasNext()) {
		  Element elem = it.next();
		  String hash = elem.attr("hash");
		  String img = " img1_"+hash+"."+mime+" ";
		  logger.debug("Extend en-note with image tag "+img);
		  elem.prepend(img);
		  Resource res  = noteStore.getResourceByHash(noteGuid, hexStringToByteArray(hash), true, false , false);
		  extractMediaResource(res, imageDirectory,  hash, note); 
	  }
	}


	// This one added but not productive, seems an Evernot Bug
	protected void processExtractImgTag(org.jsoup.nodes.Document doc, String prependSpaces,
									   String noteGuid, Note note) throws Exception {
		Elements elems = doc.select("img");
		Iterator<Element> it = elems.iterator();
		while (it.hasNext()) {
			Element elem = it.next();
			String link = elem.attr("src");
			String fileName = link.substring(link.lastIndexOf('/')+1, link.length());
			String hash = fileName.substring(0, fileName.lastIndexOf('.'));
			String img = " img1_"+fileName+" ";
			logger.debug("Extend en-note with image1 tag "+img);
			elem.prepend(img);
			Resource res  = noteStore.getResourceByHash(noteGuid, hexStringToByteArray(hash), true, false , false);
			extractMediaResource(res, imageDirectory,  hash, note);
		}

	}
	
	protected void extractMediaResource(Resource res,String imageDir, String hashCode, Note note) throws Exception {
		  int imgSize = 400;
		  int imgSizeFull = 878;
		  int tbimgSize = 200;
		  int tb1imgSize = 80;
		  if (hasLightBox(note)) {
			  imgSizeFull = 1520;
		  }
		  String mime = res.getMime().split("/")[1];
		  Resource resource = noteStore.getResource(res.getGuid(), true, false, true, false);
		  
		  String fullFiName = imageDir+"/"+hashCode +"."+mime;
		  String tbFiName = imageDir+"/"+"tb-"+hashCode+"."+mime;
		  String tb1FiName = imageDir+"/"+"tb1-"+hashCode+"."+mime;
		  DataOutputStream os = new DataOutputStream(new FileOutputStream(fullFiName));
		  os.write(resource.getData().getBody());
		  os.close();
		  BufferedImage img = ImageIO.read(new File(fullFiName));
		  BufferedImage renderedImg;
		  logger.debug("Got image with width "+img.getWidth()+"/ height "+img.getHeight());
		  if (!hasNoPullImage(note)) {
			  // Don't blow up pictures
			  if (img.getWidth() > imgSize) {
				  logger.debug("Resize the image");
				  renderedImg = Scalr.resize(img, imgSize);
			  } else {
				  renderedImg = img;
			  }
		  } else {
			  if (img.getWidth() > imgSize) {
				  logger.debug("Resize the image - no pull");
				  renderedImg = Scalr.resize(img, imgSizeFull);
			  } else {
				  renderedImg = img;
			  }
		  }
		  ImageIO.write(renderedImg, mime, new File(fullFiName));
		  renderedImg = Scalr.resize(img, tbimgSize);
		  ImageIO.write(renderedImg, mime, new File(tbFiName));
		  renderedImg = Scalr.resize(img, 
				  Scalr.Method.SPEED,
				  Scalr.Mode.FIT_EXACT,
				  tb1imgSize,
				  tb1imgSize,
				  Scalr.OP_ANTIALIAS,Scalr.OP_BRIGHTER
				  );
		  ImageIO.write(renderedImg, mime, new File(tb1FiName));
	}
	
	protected boolean hasBlogImages(Note note) throws TException, EDAMUserException, EDAMSystemException, EDAMNotFoundException {
		return utility.hasBlogImages(note);
	}
	
	protected boolean hasFixedImagePosition(Note note) throws TException, EDAMUserException, EDAMSystemException, EDAMNotFoundException {
		return utility.hasFixedImagePosition(note);
	}
	
	protected boolean hasNoPullImage(Note note) throws TException, EDAMUserException, EDAMSystemException, EDAMNotFoundException {
		return utility.hasNoPullImage(note);
	}
	
	protected boolean hasPullRightImage(Note note)throws TException, EDAMUserException, EDAMSystemException, EDAMNotFoundException {
		return utility.hasPullRight(note);
	}
	
	protected boolean hasLightBox(Note note)throws TException, EDAMUserException, EDAMSystemException, EDAMNotFoundException {
		return utility.hasLightbox(note);
	}

	protected boolean hasNoBorderImage(Note note)throws TException, EDAMUserException, EDAMSystemException, EDAMNotFoundException {
		return utility.hasNoBorder(note);
	}
	
	protected boolean hasLightBoxTitle(Note note)throws TException, EDAMUserException, EDAMSystemException, EDAMNotFoundException {
		return utility.hasLightboxTitle(note);
	}	

	private static byte[] hexStringToByteArray(String s) {
	    int len = s.length();
	    byte[] data = new byte[len / 2];
	    for (int i = 0; i < len; i += 2) {
	        data[i / 2] = (byte) ((Character.digit(s.charAt(i), 16) << 4)
	                             + Character.digit(s.charAt(i+1), 16));
	    }
	    return data;
	}

}
