package com.cloudburo.groovy.blogen

import org.joda.time.Period

import java.nio.charset.Charset

import groovy.json.JsonOutput

import groovy.json.JsonSlurper
import groovy.time.TimeDuration
import groovy.time.TimeCategory
import groovy.util.logging.Slf4j

import org.joda.time.DateTime
import org.joda.time.DateTimeZone
import org.joda.time.format.DateTimeFormatter
import org.joda.time.format.ISODateTimeFormat

import org.slf4j.LoggerFactory

import ch.qos.logback.classic.Logger
import ch.qos.logback.classic.LoggerContext
import ch.qos.logback.classic.Level

import com.evernote.edam.error.EDAMErrorCode
import com.evernote.edam.error.EDAMSystemException
import com.evernote.edam.error.EDAMUserException
import com.evernote.thrift.transport.TTransportException


import fi.iki.elonen.NanoHTTPD
import fi.iki.elonen.NanoHTTPD.Response
import fi.iki.elonen.NanoHTTPD.IHTTPSession

import com.cloudburo.evernote.EvernoteBlogPostGenerator
import com.cloudburo.evernote.EvernoteInfo
import com.cloudburo.groovy.github.BitBucketHttpHandler
import com.cloudburo.groovy.github.HttpHandler
import com.cloudburo.utility.Utilities
import com.cloudburo.utility.Cryptor
import com.cloudburo.utility.Cloudflare
import com.cloudburo.utility.AwsS3
import com.cloudburo.utility.Mailgun
import com.cloudburo.utility.Slack
import com.cloudburo.utility.AwsSQS

@Slf4j
class BlogGenerator implements Runnable {
	// Enum Definitions
	private Object appObj
	public static enum ThreadType {
		CONFIG_LISTENER,
		EVERNOTE_LISTENER,
		RECON_LISTENER,
		PING_LISTENER
	}
	public static enum EndUserEmailType {
		NEW_BLOG,
		ARTICLE_PUBLISHED,
		EVERNOTE_TOKEN_EXPIRED
	}

	private class PingListener extends NanoHTTPD {

		public PingListener() throws IOException {
			super(8888);
			start(NanoHTTPD.SOCKET_READ_TIMEOUT, false);
			log.info("INF.Env.PingListener started on port 8888")
		}

		@Override
		public Response serve(IHTTPSession session) {
			DateTime nowTime = DateTime.now(DateTimeZone.UTC)
			DateTimeFormatter isoFormat = ISODateTimeFormat.dateTime()
			String timeStamp = nowTime.toString(isoFormat)
			String startTimeStamp = startTime.toString(isoFormat)
			Period period = new Period(startTime,nowTime)
			long deltaDays = Math.abs(period.getDays())
			long deltaHours = Math.abs(period.getHours())

			String msg = "<!DOCTYPE html><html>" +
					"<head>\n"+
					"  <style>\n"+
					"    table {"+
					"	   border-collapse: collapse;"+
					"    }"+
			        "    td, th {"+
				    "      border: 1px solid #999;"+
					"      padding: 0.5rem;"+
				    "      text-align: left;"+
			        "   }"+
					"  </style>\n" +
					"</head>\n" +
					"<body><h2>Cloudburo PublishingBot Status for Shard Instance: ${shard}</h2>\n" +
					"<p><table>" +
					"   <tr><td>Execution Time</td><td>${timeStamp}</td></tr>\n" +
					"   <tr><td>Bot Startup Time</td><td>${startTimeStamp}</td></tr>\n" +
					"   <tr><td>Bot Up Time</td><td>${deltaDays} days / ${deltaHours} hours</td></tr>\n" +
					"   <tr><td>Initial Configuration</td><td> ${configInitOK}</td></tr>\n"+
					"</table></p>" +
					"<h3>Sites Configurations</h3>" +
					"<p><table >\n" +
  					"  <tr><th>SiteName</th><th>Active</th><th>Online</th><th>Site</th><th>StopDate</th></tr>\n"
			sitesConfigObjByBlogName.keySet().each { blogName1 ->
				ConfigObject blogObj = sitesConfigObjByBlogName.get(blogName1)
				String siteURL= "<a href=\"https://${blogName1}.curation.space\">${blogName1}.curation.space</a>"
				if (blogObj.cname !="") {
					siteURL= "<a href=\"https://${blogObj.cname}\">${blogObj.cname}</a>"
				}
				msg += "<tr><td>${blogName1}</td><td>${blogObj.active}</td><td>${blogObj.online}</td><td>${siteURL}</td><td>${blogObj.stopDate}</td></tr>\n"
			}
			msg += "</table></p></body></html>"
			return newFixedLengthResponse(msg);
		}
	}


	public String slackURL
	public String slackProblemChannel
	volatile public ThreadType threadType
	
	static boolean configInitOK = false
	static DateTime startTime  = DateTime.now(DateTimeZone.UTC)
	volatile static Decryptor cryp
	volatile static Decryptor crypCommon
	static String localConfigDir
	static String shard
	static String nginxInstanceFile = "_nginxInstance"

	volatile static int blogCount
	volatile static ConfigObject sitesConfigObjByBlogName
	volatile static ConfigObject sitesConfigObjByUserId
	volatile static Hashtable<String,Integer> inactivation = new Hashtable<String,Integer>()


	String S3ConfigBucket
	String S3CommonConfigBucket
	String SQSQueue
	String gitBotToken
	String cfUser
	String cfKey
	String mailgunKey
	String masterUser
	String masterSecret
	String deployDir
	String env
	String localBootstrapDir
	String localTemplateDir

	int slp
	int reconSlp

	Hashtable<String,Boolean> processState = new Hashtable<String,Boolean>()

	public BlogGenerator(String env, ThreadType type) {
		if (env == 'dry') return
		
		this.threadType = type
		this.env = env
		this.shard = Utilities.getEnvVar('CLB_SHARD')
		if (!this.shard) throw new Exception("No Shard Instance variable found stop")
		log.info("INF.Env.ShardInstance: Bootstrapping ${shard}")
		this.localBootstrapDir =  System.getProperty("user.dir")+"/build/"
		this.localConfigDir  = localBootstrapDir+"config"
		this.localTemplateDir = "${localConfigDir}/templates/"
		
		// Initialize static stuff
		if (cryp == null) {
			cryp = DecryptorFactory.createDectyptor(DecryptorFactory.factoryType.AWSKMS,"")
			crypCommon = DecryptorFactory.createDectyptor(DecryptorFactory.factoryType.AWSKMS,"")
			log.info("INF.Env.CryptorFactor initalized")
		}
		S3ConfigBucket = "clb.ms.curationplatform."+Cryptor.getAWSRegion()
		S3CommonConfigBucket = "clb.ms.common."+Cryptor.getAWSRegion()
		SQSQueue = "clb_ms_publishingbot_"+Cryptor.getAWSRegion()
		if (env != 'prod') S3ConfigBucket = "tst.ms.curationplatform."+Cryptor.getAWSRegion()
		if (env != 'prod') S3CommonConfigBucket = "tst.ms.common."+Cryptor.getAWSRegion()
		if (env != 'prod') SQSQueue = "tst_ms_publishingbot_"+Cryptor.getAWSRegion()
		log.info("INF.Env.S3Configbucket: ${S3ConfigBucket}")
		log.info("INF.Env.S3CommonConfigbucket: ${S3CommonConfigBucket}")
		log.info("INF.Env.SQSQueue: ${SQSQueue}")
		cryp.setLocalDirOrBucket(S3ConfigBucket)
		crypCommon.setLocalDirOrBucket(S3CommonConfigBucket)
		getThirdPartyLoginInformation()
	}
	
	public static void executeBlogGenerator(String env) {
		if (  env != "dry") {
			// THREAD 1: Ping Listener
			BlogGenerator gen0 = new BlogGenerator(env, ThreadType.PING_LISTENER)
			(new Thread(gen0)).start()
			// THREAD 2:  Configuration Listener
			BlogGenerator gen = new BlogGenerator(env, ThreadType.CONFIG_LISTENER)
			gen.syncCurationPlatformConfig()
			Object appObj = gen.loadApplicationObject()
			gen.syncSiteTemplateDirectories(appObj)
			gen.nginxInstanceCheck(appObj)
			Integer numThreads =   appObj.evernoteListenerThreads.toInteger()
			(new Thread(gen)).start()
			while (!BlogGenerator.configInitOK) {
				log.info("INF.Env.InitialConfig: ongoing, waiting ...")
				sleep(10000)
			}
			// THREAD 3: One Reconcile
			BlogGenerator gen1 = new BlogGenerator(env, ThreadType.RECON_LISTENER)
			(new Thread(gen1)).start()
			// The workers
			Thread[] threads = new Thread[numThreads]
			log.info("INF.Env.EvernoteListenerThreads: "+numThreads)
			for (int i=0; i< numThreads; i++) {
				// THREAD: one to many Evernote Listener
				threads[i] = new Thread(new BlogGenerator(env, ThreadType.EVERNOTE_LISTENER))
			}
			for (int i=0; i< numThreads; i++) {
				if (threads[i].getState() == Thread.State.NEW) {
					threads[i].start()
				} else if (threads[i].getState() == Thread.State.TERMINATED) {
					log.info("INF.EverThread.Terminated: A thread exited going to restarted")
					threads[i] = new BlogGenerator(env, ThreadType.EVERNOTE_LISTENER)
					threads[i].start()
				}
			}
		}
	}
		
	static void main(def args) {
		assert args[0]
		def env = args[0]  // dev, test, prod or dry		
		executeBlogGenerator(env)
	}
		
	private void getThirdPartyLoginInformation() {
		// Prepare third party login information
		String clfconfig = cryp.decrypt("/keys",'cfkey.txt')
		StringTokenizer tok = new StringTokenizer(clfconfig, ";")
		if (tok.countTokens() ==2) {
			this.cfUser = tok.nextToken();
			this.cfKey = tok.nextToken();
			
		} else {
		  log.error("Expected two tokens in cfkey.txt but got "+tok.countTokens()+" stop")
		  throw new Exception("Expected two tokens in cfkey.txt but got "+tok.countTokens()+" stop")
		}
		String bbconfig = cryp.decrypt("/keys",'bbkey.txt')
		tok = new StringTokenizer(bbconfig, ";")
		if (tok.countTokens() ==2) {
			this.masterUser = tok.nextToken();
			this.masterSecret = tok.nextToken();
			
		} else {
		  log.error("Expected two tokens in bbkey.txt but got "+tok.countTokens()+" stop")
		  throw new Exception("Expected two tokens in bbkey.txt but got "+tok.countTokens()+" stop")
		}
		this.gitBotToken = cryp.decrypt("/keys",'gbot.txt')
		this.mailgunKey = cryp.decrypt("/keys","mg.txt")
		log.info("INF.Env.LoginInformation refreshed")
	}
	
	void setProcessState(blogName,boolean stat) {
		if (stat) { 
			log.debug("setProcessState to true for ${blogName}")
			processState.put(blogName, new Boolean(true))
		}
	}

	void unsetProcessState(blogName) {
		processState.put(blogName, new Boolean(false))
	}
	
	boolean getProcessState(blogName) {
		boolean state =  processState.get(blogName).asBoolean()
		log.debug("getProcess of ${state} for ${blogName}")
		return state
	}
	
	private void checkForTemplateChanges(blogName,blogRootDir,blogObj) {
		//
		// 
		log.debug("checkForTemplateChanges ${blogName}")
		def excludeFile = "exclude-list.txt"
		if ((new File("${localBootstrapDir}${blogName}/source/blog.html.slim")).exists()) {
			excludeFile = "exclude-list1.txt"
		}
		def template = "mm-blog-tmpl-basic"
		if (blogObj.get("premiumtheme") != null && blogObj.get("premiumtheme") != "undefined" ) {
			template = "mm-blog-tmpl-"+blogObj.get("premiumtheme")
		}
		
		def cmd = "rsync -ai --progress --exclude-from '"+excludeFile+"' . ${localBootstrapDir}${blogName}"
		def ret = Utilities.executeOnShell(cmd,new File("${localTemplateDir}${template}"))
		if (checkIfLocalChanges(blogName, blogRootDir, "ConfigThread.Template"))
			checkinBlogChanges(blogRootDir, blogObj,"ConfigThread.Template","master")
	}
	
	private boolean checkIfLocalChanges(blogName, blogRootDir, tagName) {
		// refer http://stackoverflow.com/questions/5139290/how-to-check-if-theres-nothing-to-be-committed-in-the-current-branch
		
		def ret = Utilities.executeOnShell("git diff --exit-code",  new File(blogRootDir))
		if (ret == 1) {
			setProcessState(blogName,true)
			log.info("INF.localChanges.${tagName}: Unstaged changes regeneration required")
			return true
		}
		else {
			log.debug("INF.localChanges.${tagName}: no changes")
			return false
		}
	}
	
	private void checkinBlogChanges(blogRootDir, blogObj,msgTag,branch) {
		log.info("INF.${msgTag}: Checkin Blog Changes ${blogObj}")
		def ret = Utilities.executeOnShell("git add --all .",new File(blogRootDir))
		assert ret.value == 0, log.error("ERR.${msgTag}: Adding entries of ${blogRootDir} to Git failed with error code "+ret)
		ret = Utilities.executeOnShell("git commit -m 'PublishingBot Changes: "+msgTag+"'",new File(blogRootDir))
		assert ret.value == 0 || ret.value == 1, log.error("ERR.${msgTag}: Commit ${blogRootDir} to Git failed with error code "+ret)
		if (blogObj.get("remoteGIT")) {
			remotePush(blogRootDir, branch, msgTag)
		}
	}
	
	private void remotePush(rootDir, branch,msgTag) {
		log.info("INF.${msgTag}: GIT Remote Push")
		def ret = Utilities.executeOnShell("git push -u origin ${branch}",new File(rootDir))
		while (ret.value != 0) {
			log.error("Err.${msgTag} Remote Push ${rootDir} for ${branch} to Git failed with error code "+ret+" - go on trying ...")
			sleep(60000)
			ret = Utilities.executeOnShell("git push -u origin ${branch}",new File(rootDir))
		}
	}

	private boolean syncToS3(String fromFile) {

		int ret = AwsS3.s3copy( "${localConfigDir}${fromFile}", "s3://${S3ConfigBucket}${fromFile}")
		assert ret == 0, log.error("Syncing back from ${S3ConfigBucket}/siteconfigs failed with error code "+ret)
	}
	
	private boolean syncFromS3() {
		int ret
		boolean reparseIt = false;
		if (!new File(localConfigDir).exists()) {
			log.info "INF.App.EnvBootstrap: '${env}' bootstrapping from '${S3ConfigBucket}'"
			reparseIt = true
			ret = AwsS3.s3sync("s3://${S3ConfigBucket}/siteconfigs", "${localConfigDir}/siteconfigs")
			assert ret == 0, log.error("Bootstrapping from ${S3ConfigBucket}/siteconfigs failed with error code "+ret)
			ret = AwsS3.s3sync("s3://${S3ConfigBucket}/configs", "${localConfigDir}/configs")
			assert ret == 0, log.error("Bootstrapping from ${S3ConfigBucket}/configs failed with error code "+ret)
			
			if (env != "dev" ) {
				ret = Utilities.executeOnShell("git config --global user.name 'CurationDockerEngine'")
				assert ret.value == 0, log.error("Git config user.name failed with error code "+ret)
				assert ret == 0, log.error("Git config user.name failed with error code "+ret)
				ret = Utilities.executeOnShell("git config --global user.email 'felix@cloudburo.net'")
				assert ret.value == 0, log.error("Git config user.email  failed with error code "+ret)
				assert ret == 0, log.error("Git config user.email  failed with error code "+ret)
			}	
		} else {
			if (AwsS3.syncNecessary("s3://${S3ConfigBucket}/siteconfigs", "${localConfigDir}/siteconfigs")) {
				log.info "INF.App.EnvReload: '${env}' reload from '${S3ConfigBucket}/siteconfigs' due to updates"
				ret = AwsS3.s3sync("s3://${S3ConfigBucket}/siteconfigs", "${localConfigDir}/siteconfigs")
				if (ret != 0) {
					log.warn("Warm CurationBot: S3 sync failed from ${S3ConfigBucket}/siteconfigs retry it in the next round")
				} else {
				  reparseIt = true
				}
			}
			if (AwsS3.syncNecessary("s3://${S3ConfigBucket}/configs", "${localConfigDir}/configs")) {
				log.info "INF.App.EnvReload: '${env}' reload from '${S3ConfigBucket}/configs' due to updates"
				ret = AwsS3.s3sync("s3://${S3ConfigBucket}/configs", "${localConfigDir}/configs")
				if (ret != 0) {
					log.warn("Warm CurationBot: S3 sync failed from ${S3ConfigBucket}/configs retry it in the next round")
				} else {
				  reparseIt = true
				}
			} 
			if (AwsS3.syncNecessary("s3://${S3ConfigBucket}/sites", "${localConfigDir}/sites")) {
				log.info "INF.App.EnvReload: '${env}' reload from '${S3ConfigBucket}/sites' due to updates"
				ret = AwsS3.s3sync("s3://${S3ConfigBucket}/sites", "${localConfigDir}/sites")
				if (ret != 0) {
					log.warn("Warm CurationBot: S3 sync failed from ${S3ConfigBucket}/sites retry it in the next round")
				} else {
				  reparseIt = true
				}
			}
		}
		return reparseIt
	}
	
	private boolean syncCurationPlatformConfig() {
		boolean reparseIt = syncFromS3()
		if (reparseIt || sitesConfigObjByBlogName == null) {
			reloadSiteConfigs()
		}
		return reparseIt
	}

	private int getShardId() {
		if (this.shard.startsWith("_shard")) {
			String nr = this.shard.substring(6)
			return nr.toInteger()
		} else {
			log.error("ERR.BlogGenerator.getShardId shard name of instance has wrong format: ${this.shard}")
			return -1
		}
	}

	private String nginxHostName () {
		File fi =  new File("${localConfigDir}/siteconfigs/${shard}/${nginxInstanceFile}")
		return fi.text
	}

	private void nginxInstanceCheck (appObj) {
		int shardId = getShardId()
		if (shardId == -1) {
			return
		}
		if (appObj.shardNginx.size < shardId) {
			log.error("ERR.BlogGenerator.NginxInstanceCheck app.json has no nginx instance defined for shard with if '${this.shard}', stop the check")
			return
		}
		String fileName = "${localConfigDir}/siteconfigs/${shard}/${nginxInstanceFile}"
		File fi = new File(fileName)
		boolean doChange = true
		String toBeConfiguredNginxInstance = appObj.shardNginx.get(shardId - 1)
		if (fi.exists()) {
			String configuredNginxInstance = fi.text
			if (!configuredNginxInstance.equals(toBeConfiguredNginxInstance)) {
				log.error("ERR.BlogGenerator.NginxInstanceCheck Shard instance ${shardId} changes Nginx instance from ${configuredNginxInstance} to ${toBeConfiguredNginxInstance}, has to reconfigure DNS")
			} else {
				doChange = false;
			}
		}
		if (doChange) {
			Writer outFi = new FileWriter(fileName)
			log.info("INF.App.NginxInstanceCheck - Write out file ${fileName} with entry '${toBeConfiguredNginxInstance}'")
			outFi.write(toBeConfiguredNginxInstance)
			outFi.close()
			if (!syncToS3("/siteconfigs/${shard}/${nginxInstanceFile}")) {
				log.error("writeSigCongfig S3 sync failed for /siteconfigs/${shard}/${nginxInstanceFile}")
			}
		}
	}

	private generateNginxServerBlockFile(String siteName, ConfigObject blogObj) {
		boolean active = blogObj.get('active')
		boolean online = blogObj.get('online')
		String domainName = getDomainName(blogObj)
		if (blogObj.get("cname") !="") {
			domainName = blogObj.get("cname")
		}
		String fileName = "${localConfigDir}/siteconfigs/${siteName}.nginx"
		File targetFile = new File(fileName)
		File templateFile = new File("${localConfigDir}/configs/serverblock.nginx")
		String serverblock = templateFile.text
		serverblock = serverblock.replace("%DOMAINNAME%",domainName)
		serverblock = serverblock.replace("%SITENAME%", siteName)

		Writer outFi = new FileWriter(fileName)
		log.info("INF.App.NginxServerBlock - Write out file ${fileName}")
		outFi.write(serverblock)
		outFi.close()
		syncToS3("/siteconfigs/${siteName}.nginx")
		// Configure Nginx
		def ln1 = "/etc/nginx/sites-available/${siteName}.nginx"
		def ln2 =  "/etc/nginx/sites-enabled/${siteName}.nginx"
		if (!(new File(ln1)).exists()) {
			def ret = Utilities.executeOnShell("ln -s  ${fileName} ${ln1}")
			if (ret!= 0) {
				log.error("ERR.generateNginxServerBlockFileFailed to do symbolic link from ${fileName} to ${ln1}")
			}
		}
		if ( online && !(new File(ln2)).exists()) {
			log.info("INF.generateNginxServerBlockFile Add symbolic link ${ln2}")
			def ret = Utilities.executeOnShell("ln -s ${fileName} ${ln2} ")
			if (ret!= 0) {
				log.error("ERR.generateNginxServerBlockFile Failed to do symbolic link from ${fileName} to ${ln2}")
			}
			ret = Utilities.executeOnShell("sudo /usr/sbin/service nginx restart")
			if (ret!= 0) {
				log.error("ERR.generateNginxServerBlockFile Failed to restart nginx")
			}

		}

		if ( !online && (new File(ln2)).exists()) {
			def ret = Utilities.executeOnShell("rm ${ln2}")
			log.info("INF.generateNginxServerBlockFile Remove symbolic link ${ln2}")
			if (ret!= 0) {
				log.error("ERR.generateNginxServerBlockFile Failed to do remove symbolic link to ${ln2}")
			}
			ret = Utilities.executeOnShell("sudo /usr/sbin/service nginx start")
			if (ret!= 0) {
				log.error("ERR.generateNginxServerBlockFile Failed to restart nginx")
			}
		}


	}
	
	private boolean syncSiteTemplateDirectories(Object appObj) {
		// Iterate through the templates and clone, refresh them
		def update = false
		if (!new File(localTemplateDir).exists()) new File(localTemplateDir).mkdir()
		appObj.blogTemplates.each  { template ->
			def ret
			Utilities.SENSIBLEOUTPUT  = true
			def bbURL = "https://${masterUser}:${masterSecret}@${appObj.remoteMasterRepos}${template}"
			if (!new File("${localTemplateDir}/${template}").exists()) {
			  update = true
			  ret = Utilities.executeOnShell("git clone ${bbURL} ${template}", new File("${localTemplateDir}"))
			} else {			
			    // http://stackoverflow.com/questions/3258243/check-if-pull-needed-in-git 
				def cmd1 = '[ $(git rev-parse HEAD) = $(git ls-remote $(git rev-parse --abbrev-ref @{u} | '
				cmd1 +=	 " sed 's/\\// /g') | cut -f1) ] && echo no || echo yes"
				StringBuffer buf = new StringBuffer()
				ret = Utilities.executeOnShell(cmd1, new File("${localTemplateDir}/${template}"),buf)
				if (buf.contains("yes")) { 
					log.info("INF.syncSiteTemplateDirectory: ${template} changed going to pull")
					update = true
					ret = Utilities.executeOnShell("git pull", new File("${localTemplateDir}/${template}"))
				}
			}
			if (ret.value != 0) {
				log.warn("Bitbucket.TemplateSync ${template} failed with error code ${ret}, go on")
			}
			Utilities.SENSIBLEOUTPUT  = false
		}
		return update;
	}

	private synchronized void writeSiteConfig(String blogName, ConfigObject blogObj) {
		if (blogName != '') {
			String fileName = "${localConfigDir}/siteconfigs/${blogName}.json"
			log.debug("Going to write blog configuration file: ${fileName}")
			String blogJSON = JsonOutput.toJson(blogObj)
			Writer outFi = new FileWriter(fileName)
			log.debug("Write out json file: ${blogJSON}")
			outFi.write(blogJSON)
			outFi.close()
			if (!syncToS3("/siteconfigs/${blogName}.json")) {
				log.error("writeSigCongfig S3 sync failed for siteconfigs/${blogName}.json")
			}

		} else {
			log.error("ERR.Evernote.writeSiteConfig: Tried to write siteConfig whithout name")
		}
	}
	
	private static synchronized void reloadSiteConfigs() {
		String activeBlogs = ""
		blogCount = 0
		File dir = new File("${localConfigDir}/siteconfigs")
		sitesConfigObjByBlogName = new ConfigObject()
		sitesConfigObjByUserId = new ConfigObject()
		(new File("${localConfigDir}/siteconfigs/${shard}")).listFiles().each { shardFile ->
			def siteName = shardFile.getName()
			if (!siteName.equals(nginxInstanceFile)) {
				def jsFi = new File("${localConfigDir}/siteconfigs/${siteName}.json")
				if (jsFi.exists()) {
					def siteObj = new JsonSlurper().parseText(jsFi.getText("UTF-8"))
					if (siteObj.get("active")) {
						blogCount++
						activeBlogs += ",${siteName}"
					}
					sitesConfigObjByBlogName.put(siteName, siteObj)
					sitesConfigObjByUserId.put(siteObj.userid, siteObj)
				} else {
					log.error("ERR.App.Config: ${siteName} found in {shard} but no json config file exists")
				}
			}
		}
		log.info("INF.App.ActiveSites: ${blogCount} active sites config reloaded - ${activeBlogs}")
	}
	
	private void setupLogger(Object appObj) {
		// Setup Logger
		LoggerContext loggerContext = (LoggerContext)LoggerFactory.getILoggerFactory();
		Logger rootLogger = loggerContext.getLogger(Logger.ROOT_LOGGER_NAME)
		log.info("INF.App.Config: LogLevel : ${appObj.logLevel}")
		if (appObj.logLevel == "trace")
			rootLogger.setLevel(Level.TRACE)
		else if (appObj.logLevel == "debug")
			rootLogger.setLevel(Level.DEBUG)
		else if (appObj.logLevel == "info")
			  rootLogger.setLevel(Level.INFO)
		else if (appObj.logLevel == "warn")
			  rootLogger.setLevel(Level.DEBUG)
		else if (appObj.logLevel == "error")
			  rootLogger.setLevel(Level.ERROR)
		else if (appObj.logLevel == "all")
			rootLogger.setLevel(Level.ALL)
		else if (appObj.logLevel == "off")
			rootLogger.setLevel(Level.OFF)
	}
	
	private Object loadApplicationObject() {
		def appObj = new JsonSlurper().parseText(new File("${localConfigDir}/configs/app.json").getText("UTF-8"))
		this.deployDir = appObj.deployDir
		this.slp = appObj.sleepTime
		this.reconSlp = appObj.reconSleepTime
		this.slackURL = appObj.slackURL
		this.slackProblemChannel = appObj.slackProblemChannel
		setupLogger(appObj)
		return appObj
	}
	
	private void setBlogRun(String blogName) {
		while (AwsS3.getObjectWithExistenceCheck(S3ConfigBucket, "run/${blogName}")  != null) {
			log.info("INF.setBlogRun ${blogName}, sleep for 60 secs")
			sleep(60000)
		}
		def id = Thread.currentThread().getId()
		def fi = File.createTempFile("temp",".tmp").with {
			write "${shard}"
			println absolutePath
			AwsS3.putObject(S3ConfigBucket, "run/${blogName}", it)
			log.info("INF.setBlogRun ${blogName} block set")
			it.delete()
		}
	}
	
	private void unsetBlogRun(String blogName) {
	  AwsS3.deleteObject(S3ConfigBucket, "run/${blogName}")
	  log.info("INF.setBlogRun ${blogName} unblock set")
	}
	
	private void cleanBlogRun(String blogName) {
		String out = AwsS3.getObjectWithExistenceCheck(S3ConfigBucket, "run/${blogName}")
		if (out != null && out.equals(shard)) {
		    log.info("INF.cleanBlogRun: cleaning run lock for ${blogName}")
			AwsS3.deleteObject(S3ConfigBucket, "run/${blogName}")
		}
	}

	// In case the Token is not valid anymore we will decativate the account and inform the owner.
	private void evernoteTokenInvalid(Object appObj,String blogName, ConfigObject blogObj) {
		cryp.invalidateEntry( "${blogObj.userid}.evernote")
		Integer count = inactivation.get("${blogObj.userid}")
		if (count == null) count = 0
		if (count > 10) {
			log.info("INF.evernoteTokenInvalid: ${blogObj.userid} is invalide for ${count} attempts, going to inactivate for ${blogName}")
			blogObj.setProperty("active", false)
			writeSiteConfig(blogName, blogObj)
			sendEmail(appObj, blogObj, EndUserEmailType.EVERNOTE_TOKEN_EXPIRED)
			inactivation.put("${blogObj.userid}",0)
		} else {
			count = count+1
			log.info("INF.evernoteTokenInvalid: ${blogObj.userid} is invalid for ${count} attempts, still ongoing for ${blogName}")
			inactivation.put("${blogObj.userid}",count)
		}
	}
		
	public void run () {
		boolean templateChanged = false
		boolean configChanged = syncCurationPlatformConfig()
		Object appObj = loadApplicationObject()
		syncSiteTemplateDirectories(appObj)
		nginxInstanceCheck(appObj)
		boolean first = true
		ConfigObject blogObj
		Hashtable<String, String> notebooksinScopeGuids = new Hashtable()
		Set<String> notebooksOutOfScopeGuids = new HashSet<String>()
		if (this.threadType == ThreadType.PING_LISTENER) {
			new PingListener();
		}
		while (true) {
			String blogName = ""
			String errMsg
			String fromEmail = appObj.get("fromEmail")
			boolean blogRun = false
			try {
				if (this.threadType == ThreadType.EVERNOTE_LISTENER) {
					// ================================ //
					// THREAD: Evernote Change Listener //
					// ================================ //
					log.info("INF.Task.EvernoteChangeListener Iteration started")
					String msg = AwsSQS.receiveMessage(SQSQueue, Cryptor.getAWSAccountId())
					log.info("INF.EverThread.SQSMsg : " + msg)

					if (msg != null) {
						def sqsMsg = new JsonSlurper().parseText(msg)
						if (sqsMsg.get("clbuserId") != null) {
							// Attention this gives back only the first blogObj
							// We need that for the tokens
							blogObj = sitesConfigObjByUserId.get(sqsMsg.clbuserId)
							if (blogObj != null) {
								def everNoteToken = cryp.decrypt("/sitekeys", "${blogObj.userid}.evernote")
								if (sqsMsg.notebookGuid != null) {
									if (!notebooksOutOfScopeGuids.contains(sqsMsg.notebookGuid)) {
										String cachedBlogName = notebooksinScopeGuids.get(sqsMsg.notebookGuid)
										if (cachedBlogName != null) {
											blogName = cachedBlogName
											setBlogRun(blogName)
											blogRun = true
											processBlog(blogName, appObj, everNoteToken)
										} else {
											EvernoteInfo evi = new EvernoteInfo(everNoteToken, true)
											String notebookName = evi.getNotebookName(sqsMsg.notebookGuid)
											if (notebookName.endsWith("-D-NewsGen") || notebookName.endsWith("-E-CurateGen") || notebookName.endsWith("-A-Fetch")) {
												notebookName = notebookName.substring(5, notebookName.length());
												if (notebookName.endsWith("-D-NewsGen")) blogName = notebookName.substring(0, notebookName.length() - 10);
												else if (notebookName.endsWith("-A-Fetch")) blogName = notebookName.substring(0, notebookName.length() - 8);
												else if (notebookName.endsWith("-E-CurateGen")) blogName = notebookName.substring(0, notebookName.length() - 12);
												notebooksinScopeGuids.put(sqsMsg.notebookGuid, blogName)
												setBlogRun(blogName)
												blogRun = true
												processBlog(blogName, appObj, everNoteToken)
											} else {
												notebooksOutOfScopeGuids.add(sqsMsg.notebookGuid)
												log.debug("INF.EverThread.Notbook: Update took place not on our notebooks: " + notebookName + " for " + sqsMsg.clbuserId + " (add it to out of scope cache)")
											}
										}
									}
								} else {
									log.error("ERR.EverThread.SQSGet: Message doesn't contain notebookGuid: " + msg)
								}
							} else {
								log.error("ERR.EverThread.SQSGet: Provided clbuserId doesn't match any site - " + sqsMsg.clbuserId + ": " + msg)
							}
						} else {
							log.error("ERR.EverThread.SQSGet: Message has no 'clbuserId' key: " + msg)
						}
					}

				}
				else if (this.threadType == ThreadType.RECON_LISTENER) {
					// ================================================
					// THREAD:
					// We will go through all notesbooks and check for changes
					// ======================================================
					log.info("INF.Task.ReconListener Iteration started")
					def execTimeStart1 = new Date()
					reloadSiteConfigs()
					sitesConfigObjByBlogName.keySet().each { blogName1 ->
						blogName = blogName1
						blogObj = sitesConfigObjByBlogName.get(blogName)
						DateTime stopDateTime = ISODateTimeFormat.dateTimeParser().parseDateTime(blogObj.get("stopDate"));
						def everNoteToken = cryp.decrypt("/sitekeys", "${blogObj.userid}.evernote")
						if (blogObj.get("active") && !checkDelayNecessary(deployDir, blogName) && stopDateTime.isAfterNow()) {
							unsetProcessState(blogName)
							setBlogRun(blogName)
							blogRun = true
							processBlog(blogName, appObj, everNoteToken)
							unsetBlogRun(blogName)
							blogRun = false
						}
					}
					def execTimeStop = new Date()
					TimeDuration execDuration = TimeCategory.minus(execTimeStop, execTimeStart1)
					def msg = "STAT.ReconThread.ThreadRoundtripTime: ${execDuration}"
					log.info(msg)
					new Slack().sendSlackMessage(appObj.slackURL, "${msg}", appObj.slackStatChannel, "", "", "")
					log.info("INF.ReconThread.Sleep: ${reconSlp} ms")
					sleep(reconSlp)
				}
				else if (this.threadType == ThreadType.CONFIG_LISTENER) {
					// =======================================================
					// THREAD
					// Iterate through the blogs for any configuration changes
					// This is represented through S3 Filesystem Changes or Template Changes
					// ======================================================
					log.info("INF.Task.ConfigChangeListener Iteration started")
					int ret
					def execTimeStart = new Date()
					if (configChanged || templateChanged || first) {
						if (first) {
							sitesConfigObjByBlogName.keySet().each { blogName1 ->
								cleanBlogRun(blogName1)
							}
						}
						first = false
						int processedSites = 0
						log.info("INF.ConfigThread.S3: Filesystem changed")
						appObj = loadApplicationObject()
						getThirdPartyLoginInformation()
						sitesConfigObjByBlogName.keySet().each { blogName2 ->
							setBlogRun(blogName2)
							blogRun = true
							boolean newBlog = false
							blogObj = sitesConfigObjByBlogName.get(blogName2)
							String blogRootDir = "${deployDir}${blogName2}/";
							String timeZone = blogObj.get("timeZone")
							boolean followLink = blogObj.get("followLink")
							if (!blogObj.getAt("cname").endsWith("curation.space")) followLink = true
							log.debug("Config: ${blogName2}: ${blogObj.get("active")} ")
							DateTime stopDateTime = ISODateTimeFormat.dateTimeParser().parseDateTime(blogObj.get("stopDate"));
							if (stopDateTime.isBeforeNow()) log.info("INF.ConfigThread.Expired: ${blogName2} has stop date in the past, will skip ${blogObj.get('stopDate')}")
							boolean setupProblem = new File("${deployDir}/_setup_${blogName2}").exists()
							if (setupProblem) log.error("Setup problem exists for ${blogName2}, will skip")
							// Check if blog is eligible for the rune
							if ( !setupProblem && !checkDelayNecessary(deployDir, blogName2) ) {
								def iniTimeStart = new Date()
								unsetProcessState(blogName2)
								def changed = false
								def timeStart = new Date()
								processedSites++
								log.info("INF.ConfigThread.Generation: ${processedSites}/${blogCount}: ${blogRootDir}")
								log.debug("BlogObject: ${blogObj}")
								try {
									// =======================
									// Blog Setup Check
									// =======================
									if (!(new File(blogRootDir + ".git")).exists()) {
										// Either the curationBot isn't warm or a complete new blog was added
										newBlog = coldBlogSetup(appObj, blogName2, blogRootDir, blogObj)
									} else {
										warmBlogSetup(blogRootDir,appObj)
									}
									// =========================
									// Do Nginx configuration
									// =========================
									generateNginxServerBlockFile(blogName2, blogObj)
									// =========================
									// Template Change Check
									// =========================
									checkForTemplateChanges(blogName2, blogRootDir, blogObj)
									// ===================================
									// Generate for a MiddleMan based Blog
									// ===================================
									if (blogObj.get("active") && stopDateTime.isAfterNow()) {
										String everNoteToken = cryp.decrypt("/sitekeys", "${blogObj.get('userid')}.evernote")
										EvernoteBlogPostGenerator blogGen = new EvernoteBlogPostGenerator(
												everNoteToken, true, blogName2, blogRootDir + "source/posts/"
												, appObj.get("blogMove"), blogObj.get("serviceCuration"),
												blogObj.get("serviceNews"), followLink)

										(new File(blogRootDir + "source/posts/")).mkdir()
										log.info("INF.ConfigThread.MiddleMan: Generate Data Files and CName")
										generateSocialMediaDataFile(blogName2, blogObj, blogRootDir + "/data")
										generateMenuDataFile(blogName2, blogObj, blogRootDir + "/data")
										generateOtherMenuDataFile(blogName2, blogObj, blogRootDir, blogGen)
										generateFooterMenuDataFile(blogName2, blogObj, blogRootDir + "/data", blogGen)
										generateDisqusDataFile(blogName2, blogObj, blogRootDir + "/data")
										generateMetaDataFile(blogName2, blogObj, blogRootDir + "/data")
										String pubHostName
										String dnsName = appObj.get("dnsName")
										if (appObj.useGitPages) {
											pubHostName = appObj.get("gitPubTarget")
											if (!blogObj.get("online")) {
												dnsName = "curation.space"
											}
										} else {
											pubHostName = nginxHostName()
										}
										generateCnameFile(blogName2, dnsName, pubHostName, blogRootDir + "/source", blogObj)

										replaceTemplateText(blogObj, blogName2, deployDir)
										processBlogEntryToDelete(blogName2, blogRootDir)
										def iniTimeStop = new Date()
										TimeDuration duration = TimeCategory.minus(iniTimeStop, iniTimeStart)
										log.info "STAT.Config.SetupRoundtripTime.${blogName2}:  ${duration}"
										checkIfLocalChanges(blogName2, blogRootDir, "INF.ConfigThread.MiddleMan")
									}
									// ==================
									// Do Blog Generation
									// ==================
									doMiddleMenGeneration(blogName2, blogRootDir)
									// ===========================
									// Checking to Version Control
									// ===========================
									if (newBlog) {
										doCheckin(blogRootDir, blogObj, blogName2, appObj, EndUserEmailType.NEW_BLOG)
									}
									else {
										doCheckin(blogRootDir, blogObj, blogName2, appObj, EndUserEmailType.ARTICLE_PUBLISHED)
									}

								} catch (EDAMUserException e) {
									if (e.getErrorCode() == EDAMErrorCode.AUTH_EXPIRED) {
										evernoteTokenInvalid(appObj,blogName2, blogObj)
										errMsg = "ERR.Evernote.ConfigThread.AuthTokExp.${blogName2}: Authentication token is expired!"
									} else if (e.getErrorCode() == EDAMErrorCode.INVALID_AUTH) {
										evernoteTokenInvalid(appObj,blogName2, blogObj)
										errMsg = "ERR.Evernote.ConfigThread.AuthTokInv.${blogName2}: Authentication token is invalid!"
									} else if (e.getErrorCode() == EDAMErrorCode.QUOTA_REACHED) {
										errMsg = "ERR.Evernote.ConfigThread.Quota.${blogName2}: Quota is reached!"
									} else {
										errMsg = "ERR.Evernote.ConfigThread.${blogName2}: " + e.getErrorCode().toString() + " parameter: " + e.getParameter()
									}
									log.error(errMsg);
									new Slack().sendSlackMessage(appObj.slackURL, "${errMsg} (${fromEmail})", appObj.slackProblemChannel, "", "", "")
									sleep(60000)
								} catch (EDAMSystemException e) {
									boolean err= true
									if (e.getErrorCode() == EDAMErrorCode.RATE_LIMIT_REACHED) {
										err = false
										errMsg = "ERR.Evernote.ConfigThread.RateLimit.${blogName2}: Rate Limit reached delay  for " + e.rateLimitDuration
										delayProcess(e.rateLimitDuration, deployDir, blogName2)
									} else {
										errMsg = "ERR.Evernote.ConfigThread.${blogName2}: System error: " + e.getErrorCode().toString()
									}
									if (err) {
										log.error(errMsg)
										new Slack().sendSlackMessage(appObj.slackURL, "${errMsg} (${fromEmail})", appObj.slackProblemChannel, "", "", "")
									} else {
										log.warn(errMsg)
									}
									sleep(60000)
								} catch (TTransportException t) {
									errMsg = "ERR.Evernote.ConfigThread.${blogName2}: Networking error: " + t.getMessage().replace('"', '')
									log.error(errMsg);
									new Slack().sendSlackMessage(appObj.slackURL, "${errMsg} (${fromEmail})", appObj.slackProblemChannel, "", "", "")
									sleep(60000)
								} catch (Exception e1) {
									errMsg = "ERR.ConfigThread.General Exception: " + e1.getMessage().replace('"', '')
									log.error(errMsg, e1)
									new Slack().sendSlackMessage(appObj.slackURL, "${errMsg} (${fromEmail})", appObj.slackProblemChannel, "", "", "")
									sleep(60000)
								} catch (AssertionError e1) {
									errMsg = "ERR.ConfigThread.Assertion (sleep for 5 min): " + e1.getMessage().replace('"', '')
									log.error(errMsg, e1)
									new Slack().sendSlackMessage(appObj.slackURL, "${errMsg} (${fromEmail})", appObj.slackProblemChannel, "", "", "")
									sleep(60000)
								}
								def timeStop = new Date()
								TimeDuration duration = TimeCategory.minus(timeStop, timeStart)
								log.info "STAT.ConfigThread.FullRoundtripTime.${blogName2}: ${duration}"
							} // End First if
							if (blogRun) {
								unsetBlogRun(blogName2)
							}
						}  // End Site Config Loop
					}
					def execTimeStop = new Date()
					TimeDuration execDuration = TimeCategory.minus(execTimeStop, execTimeStart)
					if (execDuration.seconds > 0) {
						def msg = "STAT.ConfigThread.ThreadRoundtripTime: ${execDuration}"
						log.info(msg)
						new Slack().sendSlackMessage(appObj.slackURL, "${msg}", appObj.slackStatChannel, "", "", "")
					}
					configInitOK = true
					log.info("INF.ConfigThread.Sleep: ${slp} ms")
					sleep(slp)
					configChanged = syncCurationPlatformConfig()
					if (configChanged) {
						nginxInstanceCheck(appObj)
					}
					templateChanged = syncSiteTemplateDirectories(appObj)
				} // if
			} catch (EDAMUserException e) {
				boolean err = true;
				if (e.getErrorCode() == EDAMErrorCode.AUTH_EXPIRED) {
					errMsg = "ERR.Evernote.EverThread.AuthTokExp.${this.threadType}.${blogName}: Authentication token is expired, adjust configuration!"
					evernoteTokenInvalid(appObj,blogName, blogObj)
				} else if (e.getErrorCode() == EDAMErrorCode.INVALID_AUTH) {
					errMsg = "ERR.Evernote.EverThread.AuthTokInv.${this.threadType}.${blogName}: Authentication token is invalid, adjust configuration!"
					evernoteTokenInvalid(appObj,blogName, blogObj)
				} else if (e.getErrorCode() == EDAMErrorCode.QUOTA_REACHED) {
					errMsg = "ERR.Evernote.EverThread.Quota.${this.threadType}.${blogName}: Quota is reached, going to sleep !"
					sleep(300000)
				} else {
					errMsg = "ERR.Evernote.EverThread.${this.threadType}.${blogName}: " + e.getErrorCode().toString() + " parameter: " + e.getParameter()
					if (e.getErrorCode().toString().equals("PERMISSION_DENIED")) err = false
				}
				if (err) {
					log.error(errMsg)
					new Slack().sendSlackMessage(appObj.slackURL, "${errMsg} (${fromEmail})", appObj.slackProblemChannel, "", "", "")
				} else {
					log.warn(errMsg)
				}
			} catch (EDAMSystemException e) {
				boolean err = true
				if (e.getErrorCode() == EDAMErrorCode.RATE_LIMIT_REACHED) {
					err = false
					errMsg = "ERR.Evernote.RateLimit.EverThread.${this.threadType}.${blogName}: Rate Limit reached delay  for " + e.rateLimitDuration
					delayProcess(e.rateLimitDuration, deployDir, blogName)
				} else {
					errMsg = "ERR.Evernote.EverThread.${this.threadType}.${blogName}: System error: " + e.getErrorCode().toString()
				}
				if (err) {
					log.error(errMsg)
					new Slack().sendSlackMessage(appObj.slackURL, "${errMsg} (${fromEmail})", appObj.slackProblemChannel, "", "", "")
				} else {
					log.warn(errMsg)
				}
			} catch (TTransportException t) {
				errMsg = "ERR.Evernote.EverThread.${this.threadType}.${blogName}: Networking error (sleep for 5 min): " + ("" + t.getMessage()).replace('"', '')
				log.error(errMsg);
				new Slack().sendSlackMessage(appObj.slackURL, "${errMsg} (${fromEmail})", appObj.slackProblemChannel, "", "", "")
				sleep(300000)
			} catch (com.evernote.edam.error.EDAMNotFoundException e1) {
				errMsg = "Exception: Asks to perform an operation on an object that does not exist, going on...";
				log.error(errMsg)
				new Slack().sendSlackMessage(appObj.slackURL, "${errMsg} (${fromEmail})", appObj.slackProblemChannel, "", "", "")
			} catch (Exception e1) {
				errMsg = "ERR.EverThread.${this.threadType}.General Exception (sleep for 5 min): " + ("" + e1.getMessage()).replace('"', '')
				log.error(errMsg, e1)
				new Slack().sendSlackMessage(appObj.slackURL, "${errMsg} (${fromEmail}) sleep for 5 min", appObj.slackProblemChannel, "", "", "")
				sleep(300000)
			} catch (AssertionError e1) {
				errMsg = "ERR.EverThread.${this.threadType}.Assertion Failed (sleep for 5 min): " +  ("" + e1.getMessage()).replace('"', '')
				log.error(errMsg, e1)
				new Slack().sendSlackMessage(appObj.slackURL, "${errMsg} (${fromEmail}) sleep for 5 min", appObj.slackProblemChannel, "", "", "")
				sleep(300000)
			}
			if (blogRun && (this.threadType == ThreadType.RECON_LISTENER || this.threadType == ThreadType.EVERNOTE_LISTENER)) {
				unsetBlogRun(blogName)
			}
		} // endless while
	}
	
	private void propgatePostListToS3(String blogName, String blogrootDir) {
		File fi = new File("${blogrootDir}source/posts")
		String json = '{ "files" : [\n'
		fi.listFiles().reverseEach { 
			if (!it.name.startsWith(".")) {
			  StringTokenizer tok = new StringTokenizer(it.name,".")
			  json += '    { '+ '"filename" : ' + '"'+it.name +'", "name" : "'+tok.nextToken()+ '"},\n'
			}
		}
		json = json.substring(0,json.length()-2)+"\n"
		json += "]}"
		log.debug("Writing json file"+json)
		File fi1 = new File("${localBootstrapDir}/${blogName}.json")
		fi1.write(json)
		AwsS3.putObject(S3ConfigBucket, "sites/posts/${blogName}.json", fi1)	
	}
	
	private void processBlog(String blogName, applObj, everNoteToken) {
		ConfigObject blogObj = sitesConfigObjByBlogName.get(blogName)
		if (blogObj == null) {
			log.info("INF.EverThread - force a reloadSiteConfigs")
			reloadSiteConfigs()
			blogObj = sitesConfigObjByBlogName.get(blogName)
		}
		if (blogObj != null) {
			log.info("INF.EverThread.Hook: Event gotten for blog "+blogName)
			String blogRootDir = "${deployDir}${blogName}/"
			String timeZone = blogObj.timeZone
			boolean followLink = blogObj.followLink
			if (!blogObj.cname.endsWith("curation.space")) followLink = true
			EvernoteBlogPostGenerator blogGen = new EvernoteBlogPostGenerator(
				everNoteToken,true,blogName,blogRootDir + "source/posts/"
				,applObj.blogMove,blogObj.serviceCuration,blogObj.serviceNews,followLink)
			DateTime stopDateTime = ISODateTimeFormat.dateTimeParser().parseDateTime(blogObj.get("stopDate"));
			if (stopDateTime.isBeforeNow()) log.info("INF.EverThread: ${blogName} has stop date in the past, will skip ${blogObj.get('stopDate')}")
			boolean setupProblem = new File("${deployDir}/_setup_${blogName}").exists()
			if (setupProblem) log.error("ERR.EverThread.Setup: Problem exists for ${blogName}, will skip")
			if (blogObj.get("active") && !setupProblem && !checkDelayNecessary(deployDir,blogName) && stopDateTime.isAfterNow()) {
				def evTimeStart = new Date()
				if (blogObj.get("serviceCuration")) {
					 log.info("INF.EverThread.ServiceCuration")
					 setProcessState(blogName,processEvernote(blogName,blogObj,timeZone,deployDir, blogGen,true))
				}
				if (blogObj.get("serviceNews")) {
					log.info("INF.EverThread.NewsCuration")
					blogGen.fetchHTMLContent()
					setProcessState(blogName,processEvernote(blogName,blogObj,timeZone,deployDir, blogGen,false))
				}
				def evTimeStop = new Date()
				def duration = TimeCategory.minus(evTimeStop, evTimeStart)
				log.info "STAT.EverThread.RoundtripTime.${blogName}: ${duration}"
				// ===========================
				// Checking to Version Control
				// ===========================
				doMiddleMenFileCheck(blogRootDir)
				doMiddleMenGeneration (blogName,blogRootDir)
				// ===========================
				// Checking to Version Control
				// ===========================
				doCheckin (blogRootDir,blogObj,blogName,applObj,EndUserEmailType.ARTICLE_PUBLISHED)
			}
		} else {
			log.error("ERR.EverThread.BlogName: Blog Name doesn't exist: "+blogName)
		}
	}
	
	private boolean processBlogEntryToDelete(String blogName, String blogrootDir) {
		List files = AwsS3.getFileNames(S3ConfigBucket, "sites/deletes")
		files.each {
			StringTokenizer tok = new StringTokenizer(it,"@")
			if (tok.countTokens() == 2 ) {
				String sitePath = tok.nextToken()
				String fileName = tok.nextToken()
				if (sitePath.endsWith(blogName)) {
					boolean ok = new File("${blogrootDir}source/posts/${fileName}").delete()
					log.info("INF.Blog.Delete: ${sitePath} - ${fileName} - ${ok}")
					AwsS3.deleteObject(S3ConfigBucket,"${sitePath}@${fileName}")
				} else
					return false;
			} 
		}
		return true
	}
	
	private boolean processEvernote(String blogName, def blogObj, String timeZone, String deployDir, EvernoteBlogPostGenerator generator,curationType) {
	   def val = blogObj.get("newsq")
	   String[] days = blogObj.get("newsq_worksOn")
	   Integer cd = Utilities.getCurrentDay()
	   String currentDay = cd.toString()
	   // Any reprocessing or deletion
	   int gen = generator.generateBlogEntry(curationType,3,true)
	   log.debug("processEvernote: Processed ${gen} entries (reprocess+delete)")
	   if (currentDay == days.find { it == currentDay }) {
		   String newsc = blogObj.get("newsq_newsPerHour")
		   if (newsc.equals(""))
		     newsc = "10000"
		   log.debug("processEvernote: Max entries to process: ${newsc}")
		   int max = Integer.parseInt(newsc).value
		   String[] hours = blogObj.get("newsq_worksHour")
		   int currentHour = Utilities.getCurrentHour(timeZone)
		   boolean changeHappen = false
		   if (gen>0) changeHappen = true
		   hours.each {
			   int hour = Integer.parseInt(it)
			   if (currentHour == hour) {
				   int entries = Utilities.getProcessEntryInHour(deployDir, "newsq", blogName)
				   int process = max - entries
				   if  (process > 0) {
					   log.debug("processEvernote: Going to process max ${process} entries")
					   int gen1 = generator.generateBlogEntry(curationType, process,false)
					   Utilities.updateProcessEntryInHour("build", "newsq", blogName, gen1)
					   log.debug("processEvernote: Processed ${gen1} entries (create)")
					   if (gen>0 || gen1>0)
					     changeHappen = true
				   }
			   }
		   }
		   return changeHappen
	   }
	   return (gen>0);
	}
	
	private void delayProcess(int duration, String deployDir,String blogName) {
		File fi = new File("${deployDir}/_delay_${blogName}")
		long time = System.currentTimeMillis()+ duration * 1000; 
		fi.text =  String.valueOf(time);
	}
	
	private boolean checkDelayNecessary(String deployDir, String blogName) {
		if (new File("${deployDir}_delay_${blogName}").exists()) {
			long current = System.currentTimeMillis()
			String ts = new File("${deployDir}_delay_${blogName}").text.trim()
			long target = Long.parseLong(ts)
			log.debug("checkDelay: ${deployDir}_delay_${blogName} - ${current} <-> ${target}")
			if (current < target) return true
			log.info("INF.Evernote.${blogName}: Rate Limit completed")
			new File("${deployDir}_delay_${blogName}").delete()
		}
		return false;
	}
	
	private  void replaceTemplateText(ConfigObject config, String blogId, String rootDir) {
		replaceConfigTemplate(config,blogId,rootDir)
		replaceSyntaxhighlightingTemplate(config,blogId,rootDir)
	}

	private String escape(String txt) {
		//def str =  txt.replaceAll("'", "\\\\\\'")
		def str = txt.replace("'", "\\\\'");
		//log.debug("Escaped txt2: "+str)
		return str
	}
	
	private  void replaceConfigTemplate(ConfigObject config, String blogId, String rootDir) {
		log.debug "Replace Config Template for ${blogId}"
		def blogPath = rootDir + blogId + "/"
		def title =  escape(config.get("title"))
		def subtitle =  escape(config.get("subtitle"))
		if (subtitle == null) subtitle = ""
		def author = escape(config.get("author"))
		if (author == null) author = "Author not set"
		def theme =  config.get("theme")
		def domainName = getDomainName(config)
		def siteurl = "https://"+ domainName
		def googleanal = escape(config.get("googleanalytics"))
		if (googleanal == "") 
			googleanal = "false"
		else
			googleanal = '"'+googleanal+'"'
		def disqussn = escape(config.get("disqus_shortname"))
		if (disqussn == "") disqussn = "undefined"
				
		new File("${blogPath}config.rb").delete()
		new File("${blogPath}config.rb").text = new File("${blogPath}config.template").text 
		
		def fidir = new File(blogPath)
		def ind = 0
		def exts = [".rb"]
		def replaceBy = [title,
						 subtitle,
						 author,
						 theme,
						 siteurl,
						 googleanal,
						 disqussn
						  ]
		["_TITLE_","_SUBTITLE_","_AUTHOR_","_THEME_","_SITEURL_","_GOOGLEANALYTICS_","_DISQUSSHORTNAME_"].each {
			Utilities.replaceTextInFiles(fidir,exts,it,replaceBy[ind])
			ind++
		}
		generateNginxServerBlockFile(blogId, config)
	}
	
	private void replaceSyntaxhighlightingTemplate(ConfigObject config, String blogId, String rootDir) {
		log.debug "Replace Syntax Highlighting Theme for ${blogId}"
		def blogPath = rootDir + blogId + "/"
		def cssFile = blogPath+"source/css/_syntaxhighlighting.css.erb"
		def theme =  config.get("syntaxtheme")
		if (theme == null) theme = "Github"
		
		new File(cssFile).delete()
		new File(cssFile).text = new File("${blogPath}syntaxhighlighting.css.template").text
		
		def fidir = new File("${blogPath}source/css")
		def ind = 0
		def exts = [".erb"]
		def replaceBy = [theme]
		["_THEME_"].each {
			Utilities.replaceTextInFiles(fidir,exts,it,replaceBy[ind])
			ind++
		}
	}
	
	private String getDomainName(config) {
		if (config.cname.equals("")) return "${config.siteName}.curation.space"
		return "${config.cname}"
		
	}

	private boolean generateCnameFile(String blogId, String domainName, String pubHostName, String dataDir, ConfigObject blogObj) {
		String cname = blogObj.get('cname')
		String fileContents = new File("${dataDir}/CNAME").text
		if (cname == "" ||cname.endsWith("${domainName}") ) 
		  cname = "${blogId}.${domainName}"
		if (!cname.equals(fileContents)) {	
			log.info("INF.App.DNS: CNAME changed from '${fileContents}' to '${cname}' going to update cloudflare and nginx")
			generateNginxServerBlockFile(blogId,blogObj)
			def Cloudflare flare = new Cloudflare()
			// Is the DNS Name managed by us ?
			if (cname.endsWith(domainName)) {
				if (flare.configureCloudFlareDomainName(cfKey,cfUser,domainName,blogId, pubHostName )) {
				  new File("${dataDir}/CNAME").text = blogId+".curation.space";
				  setProcessState(blogId,true)
				  return true
				}
				else
				  log.error("generateCnameFile: CNAME change on Cloudflare failed, try in next cycle")
			} else {
			    // User uses its own domain, trying to remove our configuration
				new File("${dataDir}/CNAME").text = cname
				setProcessState(blogId,true)
			}
		}
		return false
	}
	
	private String generateMetaDataFile(String blogId, ConfigObject blogObj, String dataDir) {
		String dataFile = "${dataDir}/meta.yml"   
		StringBuffer dataContent = new StringBuffer("---\n") 
		File dataFi = new File(dataFile)
		["contactPage","blogLanguage","address_line1","address_line2","address_line3","address_telephone","address_email","address_map","copyright"].each { key ->
			def url = blogObj.get(key)
			if (!(url == null) && (!url.equals(""))) {
				log.info("INF.Blog.DataFile: Add ${key} with ${url} to meta.yml")
				dataContent.append("${key}: ${url}\n")
			}
		}
		setProcessState(blogId, Utilities.fileChanged(dataFi,dataContent.toString()))
		return dataContent.toString();	
	}
	
	private String generateDisqusDataFile(String blogId, ConfigObject blogObj, String dataDir) {
		String dataFile = "${dataDir}/disqus.yml"
		StringBuffer dataContent = new StringBuffer("---\n")
		File dataFi = new File(dataFile)
		def val1 = blogObj.get("disqus_shortname")
		if (val1 == "")
		  dataContent.append("enabled: false\n")
		else
		  dataContent.append("enabled: true\n")
		dataContent.append("shortname: ${val1}\n");
		setProcessState(blogId, Utilities.fileChanged(dataFi,dataContent.toString()))
		return dataContent.toString();
	}
	
	private String generateSocialMediaDataFile(String blogId, ConfigObject blogObj, String dataDir) {
		String dataFile = "${dataDir}/socialLinks.yml"
		StringBuffer dataContent = new StringBuffer("---\n") 
		File dataFi = new File(dataFile)
		["twitter","facebook","googleplus","linkedin","github"].each { key ->
			def url = blogObj.get("socialURL_"+key)
			if (!url.equals("")) {
				log.info("INF.Blog.DataFile: Add ${key} with ${url} to sociaLinks.yml")
				dataContent.append("${key}: ${url}\n")
			}
		}
		setProcessState(blogId, Utilities.fileChanged(dataFi,dataContent.toString()))
		return dataContent.toString();	
	}
	
	private String generateMenuDataFile(String blogId, ConfigObject blogObj,  String dataDir) {
		String dataFile = "${dataDir}/blogMenu.yml"
		StringBuffer dataContent = new StringBuffer("---\n")
		File dataFi = new File(dataFile)
		def flag = blogObj.get("blogMenu_categories")
		dataContent.append("categories: ${flag}\n")
		flag = blogObj.get("blogMenu_byYear")
		dataContent.append("byYear: ${flag}\n")
		setProcessState(blogId, Utilities.fileChanged(dataFi,dataContent.toString()))
		return dataContent.toString();
	}
	
	private String generateOtherMenuDataFile(String blogId, ConfigObject blogObj,  String rootDir, EvernoteBlogPostGenerator gen) {
		String dataDir = "${rootDir}/data"
		String dataFile = "${dataDir}/otherMenu.yml"
		String sourceDir = "${rootDir}/source"
		StringBuffer dataContent = new StringBuffer("---\n")
		File dataFi = new File(dataFile)
		String[] menu = new String[1]
		String[] name = new String[1]
		def indexPageChange = false		
		def menuPageChange = false
		// Handling of a dedicated index page
	    if (blogObj.get("indexPage")) {
			String blogMenuName = blogObj.get("blogMenuEntry")
			dataContent.append("-\n");
			menu[0] = "clb-index"
			name[0] = "Index Page";
			// The menu will point to the blog page
			log.debug("Adding blog menu name ${blogMenuName}")
			dataContent.append("    name: ${blogMenuName}\n")
			dataContent.append("    url: blog.html\n")
		    if ( !new File("${sourceDir}/blog.html.slim").exists()) {
				new File("${sourceDir}/index.html.slim").bytes = new File("${rootDir}/index.template").bytes
				new File("${sourceDir}/blog.html.slim").bytes = new File("${rootDir}/blog.template").bytes
				indexPageChange = true
				setProcessState(blogId,indexPageChange)
			}				
		} else {
		    if ( new File("${sourceDir}/blog.html.slim").exists()) {
				new File("${sourceDir}/index.html.slim").bytes = new File("${rootDir}/blog.template").bytes
				new File("${sourceDir}/blog.html.slim").delete()
				indexPageChange = true
				setProcessState(blogId,indexPageChange)
			}
		}
		// Handling of menu pages		
		["menu1","menu2","menu3","menu4"].each { key ->
			def val = blogObj.get("otherMenu_"+key);
			if (val != "") {
				menu[0] = "clb-"+key
				name[0] = blogObj.get("otherMenu_"+key);
				dataContent.append("-\n");
				log.debug("Adding other menu name ${name[0]}")
				dataContent.append("    name: ${name[0]}\n")
				dataContent.append("    url: ${key}.html\n")
			}
		}
		menuPageChange= Utilities.fileChanged(dataFi,dataContent.toString())
		if (indexPageChange || menuPageChange ) {
			log.debug("IndexPage or Other Menus change - Evernote Notes Creation: "+indexPageChange+"-"+menuPageChange)
			checkAndGenerateOtherMenuTaggedNotes(blogObj,rootDir,gen)
			setProcessState(blogId, true)
		}
		return dataContent.toString();
	}
	
	// Reconstruct in case of changes
	private String checkAndGenerateOtherMenuTaggedNotes(ConfigObject blogObj,  String rootDir, EvernoteBlogPostGenerator gen) {
		String dataDir = "${rootDir}/data"
		String dataFile = "${dataDir}/otherMenu.yml"
		String sourceDir = "${rootDir}/source"
		String[] menu = new String[1]
		String[] name = new String[1]
		// Handling of a dedicated index page
		if (blogObj.get("indexPage")) {
			menu[0] = "clb-index"
			name[0] = "Index Page";
			gen.checkAndCreateTaggedNotes(menu, name)
		} 
		// Handling of menu pages
		["menu1","menu2","menu3","menu4"].each { key ->
			def val = blogObj.get("otherMenu_"+key);
			if (val != "") {
				menu[0] = "clb-"+key
				name[0] = blogObj.get("otherMenu_"+key);	
				gen.checkAndCreateTaggedNotes(menu, name)
			}
		}
	}
	
	private void checkAndGenerateFooterMenuTaggeNotes(ConfigObject blogObj, EvernoteBlogPostGenerator gen) {
		String[] menu = new String[1]
		String[] name = new String[1]
		int index=1
		["menu1","menu2","menu3","menu4"].each { key ->
			def val = blogObj.get("footerMenu_"+key);
			if (val != "") {
				String menuN = "footer${index}"
				menu[0] = "clb-${menuN}"
				name[0] = val
				gen.checkAndCreateTaggedNotes(menu, name)
				index++
			}
		}
	}
	
	private String generateFooterMenuDataFile(String blogId, ConfigObject blogObj, String dataDir,EvernoteBlogPostGenerator gen) {
		String dataFile = "${dataDir}/footerMenu.yml"
		StringBuffer dataContent = new StringBuffer("---\n")
		File dataFi = new File(dataFile)
		String[] menu = new String[1]
		String[] name = new String[1]
		int index=1
		["menu1","menu2","menu3","menu4"].each { key ->
			def val = blogObj.get("footerMenu_"+key);
			if (val != "") {
				String menuN = "footer${index}"
				menu[0] = "clb-${menuN}"
				name[0] = val
				dataContent.append("-\n")
				log.debug("Adding footer menu name ${name[0]}")
				dataContent.append("    name: ${name[0]}\n")
				dataContent.append("    url: ${menuN}.html\n")
				index++
			}
		}
		if (Utilities.fileChanged(dataFi,dataContent.toString())) {
			setProcessState(blogId, true )
			checkAndGenerateFooterMenuTaggeNotes( blogObj,  gen)
		}
		return dataContent.toString();
	}
	
	private forceRemove(String dir,String blogName) {
		log.debug("Force Removal of directory ${dir}")
		if (new File(dir).exists()) {
			def ret = Utilities.executeOnShell("rm -rf ${dir}")
			if (ret != 0) {
				log.error("Force Removal of ${dir} for ${blogName} failed possibly inconsistent file system")
				throw new Exception("Force Removal of ${dir} for ${blogName} failed possibly inconsistent file system")
			}
		} 
	}
	
	private boolean coldBlogSetup (def appObj, def blogName, def blogRootDir, def blogObj) throws Exception {
		def ret
		def newBlog = false
		// Either the curationBot isn't warm or a complete new blog is added
		// Any failure in this setup step must result in a stop and retry
		log.debug("INF.Blog.Bootstrap: Remote transfer data to ${blogRootDir}")
		// Check if the curationBot has already a blog
		HttpHandler gitHandler = new HttpHandler(appObj.gitCuratorBot,gitBotToken)
		BitBucketHttpHandler bitbucketHandler = new BitBucketHttpHandler(masterUser,masterSecret,true)
		if (bitbucketHandler.checkRepositoryExistence(blogName)) {
			// ==========================================================
			// The repository already exists so clone it to the directory
			// We are in starting up a new microservice container
			// ==========================================================
			log.info("INF.Blog.ColdStart: ${blogName} exists in remote repository, clone it")
			def gitCloneURL = "https://${gitBotToken}@${appObj.gitProviderURL}${appObj.gitCuratorBot}/${blogName}.git"
			def bitbucketCloneURL = "https://${masterUser}:${masterSecret}@${appObj.blogMasterRepos}${blogName}.git"
			log.debug("Cold CurationBot ${blogName} : Clone master to ${deployDir}/${blogName}")
			Utilities.SENSIBLEOUTPUT = true
			ret = Utilities.executeOnShell("git clone ${bitbucketCloneURL} ${blogName}", new File(deployDir))
			assert ret.value == 0, log.error("Git clone failed with error code ${ret}")
			// =====================
			// gh-pages branch clone
			// =====================
			if (appObj.useGitPages) {
				log.debug("Cold CurationBot ${blogName}: Clone the gh-pages branch")
				ret = Utilities.executeOnShell("git clone -b gh-pages ${gitCloneURL} build", new File(blogRootDir))
				assert ret.value == 0, log.error("Cold CurationBot: Git clone failed for gh-pages with error code ${ret}")

			}
			Utilities.SENSIBLEOUTPUT = false
			// ================================
			// Update the bundle in the container
			// http://words.steveklabnik.com/how-to-not-rely-on-rubygemsorg-for-deployment
			// ================================
			if (env != "dev") {
			  log.debug("Cold CurationBot ${blogName}: Do the bundle install")
			  ret = Utilities.executeOnShell("bundle install --deployment --path ../vendor/bundle", new File(blogRootDir))
			  assert ret.value == 0, log.error("Cold CurationBot: bundle install  failed with error code "+ret)
			}
		} else {
			try {
				newBlog = true
				// ==============================================================================
				// Do a clone from the Cloudburo Template Repository fechting the template of the user
				// ==============================================================================
				log.info("INF.Blog.TemplateStart:  ${blogName} create from template")
				setProcessState(blogName,true)
				Utilities.SENSIBLEOUTPUT  = true
				def bbURL = "https://${masterUser}:${masterSecret}@${appObj.remoteMasterRepos}${blogObj.template}"
				log.debug("Initial clone  Template ${bbURL} to ${deployDir}/${blogName}")
				ret = Utilities.executeOnShell("git clone ${bbURL} ${blogName}", new File(deployDir))
				Utilities.SENSIBLEOUTPUT  = false
				assert ret.value == 0, log.error("Git clone of template failed with error code ${ret}")
				// Detach and recreate git context
				ret = Utilities.executeOnShell("rm -rf .git", new File(blogRootDir))
				assert ret.value == 0, log.error("Removing ${blogRootDir}/.git directory failed: ${ret}")
				ret = Utilities.executeOnShell("git init", new File(blogRootDir))
				assert ret.value == 0, log.error("git init in ${blogRootDir} failed: ${ret}")
				replaceTemplateText(blogObj,blogName,deployDir)
				ret = Utilities.executeOnShell("git add -A", new File(blogRootDir))
				assert ret.value == 0, log.error("git add -A in ${blogRootDir} failed: ${ret}")
				ret = Utilities.executeOnShell("git commit -m 'Initial commit by curation engine'", new File(blogRootDir))
				assert ret.value == 0, log.error("git commit failed: ${ret}")
				log.info("INF.Blog.TemplateStart: Step 1 - Local git from Template: ${blogName} ready")
				// ==========================================================
				// Attach remote to GitHub and push the initial repository
				// ==========================================================
				gitHandler.createRepository(blogName,"", "Cloudburo Curation Platform: Blog for ${blogObj.title}")
				bitbucketHandler.createRepository(blogName,"", "Cloudburo Curation Platform: Blog for ${blogObj.title}")
				Utilities.SENSIBLEOUTPUT  = true
				ret = Utilities.executeOnShell("git remote add origin https://${masterUser}:${masterSecret}@${appObj.blogMasterRepos}${blogName}.git", new File(blogRootDir))
				assert ret.value == 0, log.error("git remote add origin https://${masterUser}:${masterSecret}@${appObj.blogMasterRepos}${blogName}.git failed: ${ret}")
				Utilities.SENSIBLEOUTPUT  = false
				
				// If these steps are failing we potentially have a inconsistent stage on github
				assert new File("${deployDir}_setup_${blogName}").createNewFile(), log.error("Cannot create setup marker file")
				remotePush(blogRootDir, "master", "initialOriginPush")
				log.info("INF.Blog.TemplateStart:  Step 2 - Remote git Master: ${blogName} ready")
				if (appObj.useGitPages) {
					// ================================================
					// Let's create a build directory for the gh-pages
					// ================================================
					ret = Utilities.executeOnShell("mkdir build", new File(blogRootDir))
					assert ret.value == 0, log.error("creation of build directory failed: ${ret}")
					def buildDir = blogRootDir + "build"
					ret = Utilities.executeOnShell("git init", new File(buildDir))
					assert ret.value == 0, log.error("git init: ${ret}")
					Utilities.SENSIBLEOUTPUT = true
					ret = Utilities.executeOnShell("git remote add origin https://${gitBotToken}@${appObj.gitProviderURL}${appObj.gitCuratorBot}/${blogName}.git", new File(buildDir))
					assert ret.value == 0, log.error("git add origin  build directory  https://${gitBotToken}@${appObj.gitProviderURL}${appObj.gitCuratorBot}/${blogName}.git failed: ${ret}")
					Utilities.SENSIBLEOUTPUT = false
					ret = Utilities.executeOnShell("git checkout -b gh-pages", new File(buildDir))
					assert ret.value == 0, log.error("Checkout of branch gh-pages failed: ${ret}")
					ret = Utilities.executeOnShell("touch dummy.txt", new File(buildDir))
					assert ret.value == 0, log.error("touch dummy.txt build directory failed: ${ret}")
					ret = Utilities.executeOnShell("git add -A", new File(buildDir))
					assert ret.value == 0, log.error("git add -A in ${buildDir} failed: ${ret}")
					ret = Utilities.executeOnShell("git commit -m 'Delete All'", new File(buildDir))
					assert ret.value == 0, log.error("Commit to Git failed with error code " + ret)
					remotePush(buildDir, "gh-pages", "initialGhpagesPush")
					assert ret.value == 0, log.error("Push to Git failed with error code " + ret)
					assert new File("${deployDir}_setup_${blogName}").delete(), log.error("Couldn't remove ${deployDir}/_setup_${blogName}, investigate: ${ret}")
					log.info("INF.Blog.TemplateStart: Step 3 - Remote git branch gh-pages: ${blogName} ready")
				}
			} catch (all) {
				String errMsg = "Initial setup failed: ${all}, force remove of ${blogName}"
				forceRemove("${deployDir}/${blogName}", blogName)
				log.error(errMsg, all)
				new Slack().sendSlackMessage(gen.slackURL,"${errMsg}",gen.slackProblemChannel, "", "", "")
				throw new Exception("Initial setup failed, interrupt: ${all}")
			}
			if (env != "dev") {
				ret = Utilities.executeOnShell("bundle install --deployment --path ../vendor/bundle", new File(blogRootDir))
				assert ret.value == 0, log.error("bundle install  failed with error code "+ret)
				log.info("INF.Blog.TemplateStart: Step 4 - Bundle install: ${blogName} ready, setup complete")
			}
		}
		return newBlog
	}
	
	private def warmBlogSetup(blogRootDir,appObj) {
		def ret
		// ======================================
		// We are on warm curation bot, do a pull
		// A failure with a dependent service will be ignored and logged
		// ======================================
		log.info("INF.Blog.Warm: git pull master ${blogRootDir} master")
		ret = Utilities.executeOnShell("git pull origin master",  new File(blogRootDir))
		if (ret.value != 0) {
			log.warn("Warm CurationBot: git pull origin master ${blogRootDir} failed, going to try a force")
			ret = Utilities.executeOnShell("git fetch --all",  new File(blogRootDir))
			if (ret.value != 0) log.error("Warm Curationbot: Git fetch -all failed with error code "+ret)
			ret = Utilities.executeOnShell("git reset --hard origin/master",  new File(blogRootDir))
			if (ret.value != 0) log.error("Warm Curationbot: Git reset --hard origin/master failed with error code "+ret)
		}
		// =====================
		// Now pull the gh-pages
		// =====================
		if (appObj.useGitPages) {
			log.info("INF.Blog.Warm: git pull origin gh-pages ${blogRootDir}build")
			new File(blogRootDir + "build").mkdir()
			ret = Utilities.executeOnShell("git pull origin gh-pages", new File(blogRootDir + "build"))
			if (ret.value != 0) {
				log.warn("Warm CurationBot: git pull origin gh-pages ${blogRootDir} failed, going to do a force")
				ret = Utilities.executeOnShell("git fetch --all", new File(blogRootDir))
				if (ret.value != 0) log.error("Warm Curationbot: Git fetch -all failed with error code " + ret)
				ret = Utilities.executeOnShell("git reset --hard origin/master", new File(blogRootDir))
				if (ret.value != 0) log.error("Warm Curationbot: Git reset --hard origin/master failed with error code " + ret)
			}
		}
	}
	
	private void doCheckin (String blogRootDir,ConfigObject blogObj,String blogName,Object appObj,  emailType) {
		def gitTimeStart = new Date()
		def ret = Utilities.executeOnShell("git pull origin master",  new File(blogRootDir))
			if (ret.value != 0) {
			log.warn("Do Checkin: git pull origin master ${blogRootDir} failed, going to try a force")
			ret = Utilities.executeOnShell("git fetch --all",  new File(blogRootDir))
			if (ret.value != 0) log.error("Do Checkin: Git fetch -all failed with error code "+ret)
			ret = Utilities.executeOnShell("git reset --hard origin/master",  new File(blogRootDir))
			if (ret.value != 0) log.error("Do Checkin: Git reset --hard origin/master failed with error code "+ret)
		}
		// ======================================
		// Git add the master, run it in any case
		// ====================================== 
		checkinBlogChanges(blogRootDir, blogObj,"INF.Blog.Bitbucket","master")
		
		if (getProcessState(blogName)) {
			if (appObj.useGitPages) {
				checkinBlogChanges(blogRootDir + "build", blogObj, "INF.Blog.Github", "gh-pages")
			}
			sendEmail(appObj, blogObj, emailType)
		}
		def gitTimeStop = new Date()
		def duration = TimeCategory.minus(gitTimeStop, gitTimeStart)
		log.info "STAT.GITRoundtripTime.${blogName}: ${duration}"
	}

	private void sendEmail(Object appObj, ConfigObject blogObj, EndUserEmailType emailType) {
		// TODO: Take in consideration user language blogObj.get('language') when sending out Emails to the Enduser
		String language = "en"



		String mailgunDomain = appObj.get("mailgunDomain")
		String toEmail = blogObj.get("email")
		String fromEmail = appObj.get("fromEmail")
		String mailSubject
		String mailTxt
		String slackAdminTxt
		String slackUserTxt
		String userSlackLink = null
		try {
			def slackCredentials = crypCommon.decrypt("/credentials", "${blogObj.get('userid')}.slackApp")
			def content = new JsonSlurper().parseText(slackCredentials)
			userSlackLink = content.serviceData.tokens.incoming_webhook.url
			log.debug("Got a Slack URL: ")
		} catch (Exception e) {
			log.debug("No Slack Credentials: ${e}")
		}
		if (emailType == EndUserEmailType.NEW_BLOG) {
			mailSubject = blogObj.get("siteName") + appObj.get("newBlogLiveSubject_${language}")
			mailTxt = appObj.get("newBlogLiveContent_${language}")
			String url = "https://" + blogObj.get("siteName") + "." + appObj.get("dnsName")
			mailTxt += url
			mailTxt += appObj.get("newBlogLiveContent1_${language}")
			slackUserTxt = mailTxt
			slackAdminTxt = "New Blog (${fromEmail}): <${url}>"

		} else if (emailType == EndUserEmailType.ARTICLE_PUBLISHED) {
			mailSubject = blogObj.get("siteName") + appObj.get("modBlogLiveSubject_${language}")
			def siteUrl
			if (blogObj.get("cname").equals("")) {
				siteUrl = "https://" + blogObj.get("siteName") + "." + appObj.get("dnsName")
			} else {
				siteUrl = "https://" + blogObj.get("cname")
			}
			mailTxt = appObj.get("modBlogLiveContent_${language}")
			slackUserTxt = appObj.get("modBlogLiveContent1_${language}")

			mailTxt += siteUrl
			slackUserTxt += siteUrl
			slackAdminTxt = "Update Blog (${fromEmail}): <${siteUrl}>"

		} else if (emailType == EndUserEmailType.EVERNOTE_TOKEN_EXPIRED) {
			mailSubject = blogObj.get("siteName") + appObj.get("evernoteTokenProblemSubject_${language}")
			mailTxt = appObj.get("evernoteTokenProblemContent_${language}")
			slackUserTxt = appObj.get("evernoteTokenProblemContent1_${language}")
			slackAdminTxt = "Evernote Token expired for ${fromEmail} using blog: "+blogObj.get("siteName")
		} else  {
			log.error("sendMail called with unknown type")
			return
		}
		// Send to user either via Email or Slack
		if (userSlackLink == null || userSlackLink.equals("")) {
			Mailgun mail = new Mailgun()
			mail.sendEmail(this.mailgunKey, mailgunDomain, fromEmail, toEmail, mailSubject, mailTxt)
		} else {
			new Slack().sendSlackMessage(userSlackLink, slackUserTxt, "", "", "", "")
		}
		new Slack().sendSlackMessage(appObj.slackURL, slackAdminTxt, appObj.slackStatusChannel, "", "", "")
	}

	private def doMiddleMenFileCheck(blogRootDir) {
		File fi = new File(blogRootDir+"/data/blogImages.yml");
		if (!fi.exists()) writeToFile(fi,Charset.forName("UTF-8"),"backgroundImage_1:")
	}
	
	private static void writeToFile(File file, Charset charset, String data) throws IOException {
		OutputStream out = new FileOutputStream(file);
		Closeable stream = out;
		try {
			Writer writer = new OutputStreamWriter(out, charset);
			stream = writer;
			writer.write(data);
		} finally {
			stream.close();
		}
	}
	
	private def doMiddleMenGeneration (blogName,blogRootDir) {
		if (getProcessState(blogName)) {
			def mmTimeStart = new Date()
			log.info("INF.Blog.MiddleMan: Generation ${blogRootDir}")
			def ret = Utilities.executeOnShell("bundle exec middleman build --verbose --no-clean",new File(blogRootDir))
			if (ret.value !=0 && ret.value != 1) {
				log.error("MiddleMan Generation ${blogRootDir} failed with error code, try to checkin the generated stuff "+ret)
			}
			if (ret.value == 1) {
				log.warn("MiddleMan Generation ${blogRootDir} returned error code 1, potential problem")
			}
			def mmTimeStop = new Date()
			propgatePostListToS3(blogName,blogRootDir)
			def duration = TimeCategory.minus(mmTimeStop, mmTimeStart)
			log.info "STAT.MMRoundtripTime.${blogName}: ${duration}"
		}
	}
}
