/*
 * Copyright (c) 2017-present, Takayuki Maruyama
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package net.bis5.mattermost.model;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import java.io.IOException;
import java.util.Arrays;
import lombok.Getter;
import net.bis5.mattermost.model.AuthService.AuthServiceDeserializer;
import net.bis5.mattermost.model.serialize.HasCodeSerializer;

/**
 * The type of auth service.
 * 
 * @author Takayuki Maruyama
 */
@JsonSerialize(using = HasCodeSerializer.class)
@JsonDeserialize(using = AuthServiceDeserializer.class)
@Getter
public enum AuthService implements HasCode<AuthService> {

  Email("email"), Saml("saml"), GitLab("gitlab"), Google("google"), Office365("office365");
  private final String code;

  AuthService(String code) {
    this.code = code;
  }

  public static AuthService of(String code) {
    return Arrays.stream(values()).filter(e -> e.getCode().equals(code)).findFirst().orElse(Email);
  }

  static class AuthServiceDeserializer extends JsonDeserializer<AuthService> {

    @Override
    public AuthService deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
      String jsonValue = p.getText();
      return of(jsonValue);
    }
  }
}
