@file:Suppress("MemberVisibilityCanBePrivate")

package net.axay.memoire

import kotlinx.datetime.Instant
import okio.Path

class DiskCache<K, V, P>(
    val config: DiskCacheConfig<K, V, P>,
    validationConfig: CacheValidationConfig,
) : SyncCache<K, Path, V, P>(validationConfig) {

    private val fileSystem = config.fileSystem
    init {
        fileSystem.createDirectories(config.directory)
    }

    override fun internalGet(key: Path): V? {
        return try {
            fileSystem.read(key, config.deserializer)
        } catch (exc: okio.IOException) {
            null
        }
    }

    override fun internalContains(key: Path): Boolean {
        return fileSystem.exists(key)
    }

    override fun internalPut(key: Path, value: P) {
        fileSystem.write(key) {
            config.serializer(this, value)
        }
    }

    override fun internalPutAndGet(key: Path, value: P): V? {
        internalPut(key, value)
        return internalGet(key)
    }

    override fun internalRemove(key: Path): Boolean {
        return try {
            fileSystem.delete(key)
            true
        } catch (notFound: okio.FileNotFoundException) {
            false
        }
    }

    override fun internalGetAndRemove(key: Path): V? {
        return internalGet(key).also { internalRemove(key) }
    }

    override fun internalIsInvalid(key: Path): Boolean? {
        if (!fileSystem.exists(key)) return null

        val metadata = fileSystem.metadata(key)

        return !validationConfig.isValid(
            metadata.lastModifiedAtMillis?.let(Instant::fromEpochMilliseconds) ?: return true,
            metadata.lastAccessedAtMillis?.let(Instant::fromEpochMilliseconds) ?: return true
        )
    }


    override fun internalEvict() {
        fileSystem
            .list(config.directory)
            .forEach(::internalRemoveIfInvalid)
    }

    override val K.internal: Path
        get() = config.pathFromKey(this)
}
