package net.axay.memoire

import net.axay.memoire.internal.ReadWriteMutex

abstract class SyncCache<KeyType, InternalKeyType, ValueType, PutType>(
    validationConfig: CacheValidationConfig
) : Cache<KeyType, ValueType, PutType>(validationConfig) {

    val rwMutex = ReadWriteMutex()

    override suspend fun get(key: KeyType): ValueType? {
        val iKey = key.internal
        val shouldRemove = when (rwMutex.read { internalIsInvalid(iKey) }) {
            false -> return internalGet(iKey)
            true -> true
            null -> false
        }
        if (shouldRemove) {
            rwMutex.write {
                internalRemoveIfInvalid(iKey)
            }
        }
        return null
    }

    override suspend fun contains(key: KeyType): Boolean {
        val iKey = key.internal
        return when (rwMutex.read { internalIsInvalid(iKey) }) {
            null -> false
            true -> {
                rwMutex.write { internalRemoveIfInvalid(iKey) }
                false
            }
            false -> true
        }
    }

    override suspend fun put(key: KeyType, value: PutType) {
        val iKey = key.internal
        return rwMutex.write { internalPut(iKey, value) }
    }

    override suspend fun putAndGet(key: KeyType, value: PutType): ValueType? {
        val iKey = key.internal
        return rwMutex.write { internalPutAndGet(iKey, value) }
    }

    override suspend fun remove(key: KeyType): Boolean {
        val iKey = key.internal
        return rwMutex.write { internalRemove(iKey) }
    }

    override suspend fun getAndRemove(key: KeyType): ValueType? {
        val iKey = key.internal
        return rwMutex.write { internalGetAndRemove(iKey) }
    }

    override suspend fun removeIfInvalid(key: KeyType): Boolean? {
        val iKey = key.internal
        return rwMutex.write { internalRemoveIfInvalid(iKey) }
    }

    override suspend fun evict() {
        rwMutex.write { internalEvict() }
    }

    abstract val KeyType.internal: InternalKeyType

    abstract fun internalGet(key: InternalKeyType): ValueType?

    abstract fun internalContains(key: InternalKeyType): Boolean

    abstract fun internalPut(key: InternalKeyType, value: PutType)

    abstract fun internalPutAndGet(key: InternalKeyType, value: PutType): ValueType?

    abstract fun internalRemove(key: InternalKeyType): Boolean

    abstract fun internalGetAndRemove(key: InternalKeyType): ValueType?

    abstract fun internalIsInvalid(key: InternalKeyType): Boolean?

    fun internalRemoveIfInvalid(key: InternalKeyType): Boolean? {
        val invalid = internalIsInvalid(key) ?: return null
        if (invalid) internalRemove(key)
        return invalid
    }

    abstract fun internalEvict()
}
